-- ======================================================
-- ADMIN DASHBOARD DATABASE SETUP
-- Tables for User Management and Activity Logging
-- ======================================================

-- Table 1: Users Management
-- Stores all system users with their roles and creation metadata
CREATE TABLE IF NOT EXISTS users (
    id INT AUTO_INCREMENT PRIMARY KEY COMMENT 'Unique User ID',
    username VARCHAR(100) UNIQUE NOT NULL COMMENT 'Unique login name',
    password VARCHAR(255) NOT NULL COMMENT 'Hashed password (store as plain for now)',
    role ENUM('Administrator', 'Controller') NOT NULL DEFAULT 'Controller' COMMENT 'User role/permission level',
    created_by INT COMMENT 'ID of the Admin who created this user (Self-Referencing FK)',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP COMMENT 'Date and time of account creation',
    FOREIGN KEY (created_by) REFERENCES users(id) ON DELETE SET NULL,
    INDEX idx_username (username),
    INDEX idx_role (role),
    INDEX idx_created_by (created_by)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci COMMENT='System users table for authentication and role management';

-- Table 2: Activity Log
-- Tracks all user activities and data entries for audit purposes
CREATE TABLE IF NOT EXISTS activity_log (
    entry_id INT AUTO_INCREMENT PRIMARY KEY COMMENT 'Unique entry ID',
    user_id INT NOT NULL COMMENT 'The ID of the user who made the entry',
    username VARCHAR(100) NOT NULL COMMENT 'Username for reference',
    employee_name VARCHAR(255) COMMENT 'Name of the employee data being stored',
    record_count INT DEFAULT 1 COMMENT 'Number of records added in this session',
    entry_time TIMESTAMP DEFAULT CURRENT_TIMESTAMP COMMENT 'Time of the entry',
    action_type VARCHAR(100) DEFAULT 'DATA_ENTRY' COMMENT 'Type of action (DATA_ENTRY, EDIT, DELETE, etc.)',
    details TEXT COMMENT 'Additional details about the activity',
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_user_id (user_id),
    INDEX idx_entry_time (entry_time),
    INDEX idx_username (username)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci COMMENT='Activity audit log - Users cannot delete records';

-- ======================================================
-- INITIALIZE DEFAULT ADMIN USER (Optional - Uncomment if needed)
-- ======================================================
-- This creates an initial Administrator account
-- Password should be hashed in production
-- INSERT INTO users (username, password, role, created_by) 
-- VALUES ('ADMIN', 'admin123', 'Administrator', NULL);

-- ======================================================
-- SECURITY NOTES:
-- ======================================================
-- 1. Self-Referencing Foreign Key: 'created_by' tracks which admin created each user
-- 2. Activity Log has NO DELETE route accessible to non-admins
-- 3. Role-based deletion: Only users with role 'Controller' or 'User' can be deleted
-- 4. Timestamp tracking: Automatic timestamps for audit trail
-- ======================================================
