-- ============================================
-- MACHINE MAINTENANCE MANAGEMENT TABLES
-- ============================================

-- Create machines table
CREATE TABLE IF NOT EXISTS machines (
    id INT AUTO_INCREMENT PRIMARY KEY,
    machine_name VARCHAR(255) NOT NULL UNIQUE,
    machine_code VARCHAR(100),
    purchase_date DATE,
    maintenance_interval_days INT DEFAULT 90,
    last_maintenance_date DATE,
    next_due_date DATE,
    status ENUM('Active', 'Inactive', 'Under Maintenance') DEFAULT 'Active',
    notes TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_next_due (next_due_date),
    INDEX idx_status (status)
);

-- Create maintenance history log table
CREATE TABLE IF NOT EXISTS maintenance_history (
    id INT AUTO_INCREMENT PRIMARY KEY,
    machine_id INT NOT NULL,
    maintenance_date DATE NOT NULL,
    maintenance_type ENUM('Preventive', 'Corrective', 'Inspection') DEFAULT 'Preventive',
    description TEXT,
    technician_name VARCHAR(255),
    duration_hours DECIMAL(10, 2),
    cost DECIMAL(12, 2),
    next_due_date DATE,
    created_by VARCHAR(255),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (machine_id) REFERENCES machines(id) ON DELETE CASCADE,
    INDEX idx_machine_id (machine_id),
    INDEX idx_maintenance_date (maintenance_date)
);

-- ============================================
-- TOOL STROKE LIFE MANAGEMENT TABLES
-- ============================================

-- Extend existing tools table or create new one
CREATE TABLE IF NOT EXISTS tools_extended (
    id INT AUTO_INCREMENT PRIMARY KEY,
    tool_id INT,
    tool_name VARCHAR(255) NOT NULL,
    tool_code VARCHAR(100),
    max_life_strokes INT NOT NULL DEFAULT 100000,
    current_strokes_used INT DEFAULT 0,
    warning_threshold INT DEFAULT 1000,
    status ENUM('Active', 'Inactive', 'Needs Replacement') DEFAULT 'Active',
    last_refurbished_date DATE,
    refurbish_count INT DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_status (status),
    INDEX idx_strokes_remaining (current_strokes_used)
);

-- Create tool stroke history log
CREATE TABLE IF NOT EXISTS tool_stroke_history (
    id INT AUTO_INCREMENT PRIMARY KEY,
    tool_id INT NOT NULL,
    production_entry_id INT,
    strokes_added INT NOT NULL,
    remaining_strokes INT,
    log_date DATE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (tool_id) REFERENCES tools_extended(id) ON DELETE CASCADE,
    INDEX idx_tool_id (tool_id),
    INDEX idx_log_date (log_date)
);

-- Create tool refurbish log
CREATE TABLE IF NOT EXISTS tool_refurbish_log (
    id INT AUTO_INCREMENT PRIMARY KEY,
    tool_id INT NOT NULL,
    refurbished_by VARCHAR(255),
    strokes_before INT,
    strokes_after INT DEFAULT 0,
    refurbish_date DATE NOT NULL,
    notes TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (tool_id) REFERENCES tools_extended(id) ON DELETE CASCADE,
    INDEX idx_tool_id (tool_id),
    INDEX idx_refurbish_date (refurbish_date)
);

-- ============================================
-- ALERTS TABLE
-- ============================================

CREATE TABLE IF NOT EXISTS system_alerts (
    id INT AUTO_INCREMENT PRIMARY KEY,
    alert_type ENUM('Machine Maintenance', 'Tool Stroke', 'Warning', 'Critical', 'Info') DEFAULT 'Info',
    related_id INT,
    related_type VARCHAR(50), -- 'machine' or 'tool'
    title VARCHAR(255) NOT NULL,
    message TEXT,
    severity ENUM('Low', 'Medium', 'High', 'Critical') DEFAULT 'Medium',
    is_read BOOLEAN DEFAULT FALSE,
    is_active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    resolved_at TIMESTAMP NULL,
    INDEX idx_active (is_active),
    INDEX idx_created_at (created_at),
    INDEX idx_severity (severity)
);

-- ============================================
-- SAMPLE DATA (OPTIONAL)
-- ============================================

-- INSERT INTO machines (machine_name, machine_code, purchase_date, last_maintenance_date) 
-- VALUES 
-- ('CNC Machine 001', 'CNC-001', '2023-01-15', CURDATE()),
-- ('Lathe Machine 001', 'LATHE-001', '2023-06-20', CURDATE()),
-- ('Drill Press 001', 'DRILL-001', '2022-11-10', CURDATE());

-- INSERT INTO tools_extended (tool_name, tool_code, max_life_strokes) 
-- VALUES 
-- ('Drill Bit 10mm', 'DB-10MM', 100000),
-- ('End Mill 8mm', 'EM-8MM', 150000),
-- ('Tap M8', 'TAP-M8', 50000);
