<?php
// Protect this page and disable caching of authenticated content
include 'session_check.php';

// Only Administrator role can access this page
if (($_SESSION['role'] ?? '') !== 'Administrator') {
    header("Location: index.php");
    exit();
}

$pageFlashNotice = $_SESSION['access_denied_notice'] ?? '';
unset($_SESSION['access_denied_notice']);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Administrator Dashboard - User Management</title>
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background-color: #edf2f7;
            background-image:
                linear-gradient(30deg, rgba(148, 163, 184, 0.06) 12%, transparent 12.5%, transparent 87%, rgba(148, 163, 184, 0.06) 87.5%, rgba(148, 163, 184, 0.06)),
                linear-gradient(150deg, rgba(148, 163, 184, 0.06) 12%, transparent 12.5%, transparent 87%, rgba(148, 163, 184, 0.06) 87.5%, rgba(148, 163, 184, 0.06)),
                linear-gradient(90deg, rgba(255, 255, 255, 0.85), rgba(255, 255, 255, 0.85));
            background-size: 44px 76px, 44px 76px, auto;
            background-position: 0 0, 22px 38px, 0 0;
            color: #333;
        }

        /* ====== HEADER ====== */
        .header {
            margin-left: 290px;
            padding: 22px 34px;
            display: flex;
            justify-content: space-between;
            align-items: center;
            border-bottom: 1px solid rgba(15, 23, 42, 0.08);
            background: rgba(241, 245, 249, 0.7);
            backdrop-filter: blur(2px);
            transition: margin-left 0.3s ease;
        }

        .header-title-wrap {
            display: flex;
            align-items: center;
            gap: 18px;
        }

        .header h1 {
            font-size: 28px;
            font-weight: 800;
            letter-spacing: 0.3px;
            text-transform: none;
            color: #1e293b;
            display: flex;
            align-items: center;
            gap: 14px;
            line-height: 1.1;
        }

        .header-page-icon {
            color: #334155;
            font-size: 26px;
        }

        .header-actions {
            display: flex;
            align-items: center;
        }

        .header-notify-btn {
            width: 42px;
            height: 42px;
            border: 1px solid rgba(15, 23, 42, 0.15);
            border-radius: 50%;
            background: #ffffff;
            color: #334155;
            cursor: pointer;
            transition: all 0.2s ease;
        }

        .header-notify-btn:hover {
            transform: translateY(-1px);
            box-shadow: 0 8px 18px rgba(15, 23, 42, 0.15);
        }

        .logout-btn {
            background-color: #f39c12;
            color: white;
            border: none;
            padding: 10px 20px;
            border-radius: 4px;
            cursor: pointer;
            font-weight: 600;
            transition: 0.3s;
        }

        .logout-btn:hover {
            background-color: #e67e22;
        }

        /* ====== SIDEBAR NAVIGATION ====== */
        .sidebar-nav {
            width: 290px;
            height: 100vh;
            background: linear-gradient(180deg, #003f43 0%, #01343a 48%, #052f35 100%);
            position: fixed;
            left: 0;
            top: 0;
            display: flex;
            flex-direction: column;
            color: white;
            box-shadow: 8px 0 30px rgba(2, 6, 23, 0.25);
            z-index: 1000;
            transform: translateX(0);
            transition: transform 0.3s ease;
        }

        .sidebar-nav.collapsed {
            transform: translateX(-290px);
        }

        .sidebar-logo {
            padding: 22px 24px;
            background: #003136;
            display: flex;
            align-items: center;
            gap: 14px;
            border-bottom: 1px solid rgba(255, 255, 255, 0.12);
        }

        .sidebar-logo-icon {
            width: 44px;
            height: 44px;
            border-radius: 12px;
            display: inline-flex;
            align-items: center;
            justify-content: center;
            background: rgba(245, 158, 11, 0.16);
            color: #f59e0b;
            font-size: 21px;
            border: 1px solid rgba(245, 158, 11, 0.25);
        }

        .sidebar-logo h2 {
            margin: 0;
            font-size: 34px;
            letter-spacing: 0.8px;
            color: #f8fafc;
            font-weight: 700;
            line-height: 1;
        }

        .logo-divider {
            width: 84px;
            height: 2px;
            margin: 8px 0 6px;
            background: rgba(255, 255, 255, 0.65);
        }

        .user-badge {
            font-size: 14px;
            color: #f59e0b;
            text-transform: uppercase;
            font-weight: 700;
            letter-spacing: 0.9px;
        }

        .sidebar-menu {
            list-style: none;
            padding: 14px 0;
            margin: 0;
            flex-grow: 1;
            overflow-y: auto;
        }

        .menu-label {
            padding: 14px 26px 8px;
            font-size: 11px;
            color: #8ea6ae;
            font-weight: bold;
            letter-spacing: 1.2px;
            text-transform: uppercase;
        }

        .sidebar-menu li a {
            display: flex;
            align-items: center;
            gap: 10px;
            padding: 14px 24px;
            color: #d4e4e7;
            text-decoration: none;
            font-size: 15px;
            font-weight: 500;
            border-left: 4px solid transparent;
            transition: all 0.24s ease;
        }

        .sidebar-menu li a:hover {
            background: linear-gradient(90deg, rgba(245, 158, 11, 0.08) 0%, rgba(16, 185, 129, 0.18) 100%);
            color: #ffffff;
            border-left-color: rgba(245, 158, 11, 0.8);
        }

        .sidebar-menu li a.active {
            background: linear-gradient(90deg, rgba(245, 158, 11, 0.25) 0%, rgba(15, 118, 110, 0.6) 100%);
            color: #ffffff;
            border-left: 4px solid #f59e0b;
            font-weight: 600;
        }

        .sidebar-menu li a .icon {
            width: 22px;
            display: inline-flex;
            justify-content: center;
            font-size: 16px;
        }

        .sidebar-footer {
            padding: 14px 18px 20px;
            background: #042b30;
            border-top: 1px solid rgba(255, 255, 255, 0.12);
        }

        .sidebar-footer .logout-btn {
            width: 100%;
            padding: 13px;
            background: #c73a2a;
            color: white;
            border: none;
            border-radius: 10px;
            font-weight: bold;
            font-size: 14px;
            cursor: pointer;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 8px;
            transition: background 0.3s ease;
        }

        .sidebar-footer .logout-btn:hover {
            background-color: #e74c3c;
        }

        /* Toggle Button */
        .toggle-sidebar-btn {
            width: 44px;
            height: 44px;
            background: #ffffff;
            border: 1px solid rgba(15, 23, 42, 0.14);
            color: #0f4b4e;
            font-size: 20px;
            cursor: pointer;
            border-radius: 10px;
            transition: 0.25s ease;
        }

        .toggle-sidebar-btn:hover {
            transform: translateY(-1px);
            box-shadow: 0 8px 18px rgba(15, 23, 42, 0.13);
        }

        .sidebar-nav.collapsed ~ .header {
            margin-left: 0;
        }

        /* ====== MAIN CONTAINER ====== */
        .container {
            display: flex;
            gap: 30px;
            padding: 40px;
            max-width: 100%;
            margin: 0;
            margin-left: 290px;
            transition: margin-left 0.3s ease;
        }

        .sidebar-nav.collapsed ~ .container {
            margin-left: 0;
        }

        /* ====== LEFT SIDEBAR ====== */
        .sidebar {
            flex: 0 0 350px;
        }

        .form-section {
            background: white;
            border-radius: 8px;
            padding: 25px;
            box-shadow: 0 2px 8px rgba(0, 0, 0, 0.08);
            margin-bottom: 25px;
            border-top: 4px solid #f39c12;
        }

        .form-section h3 {
            font-size: 16px;
            font-weight: 700;
            color: #222;
            margin-bottom: 20px;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }

        .form-group {
            margin-bottom: 18px;
        }

        .form-group label {
            display: block;
            font-size: 12px;
            font-weight: 600;
            color: #666;
            margin-bottom: 8px;
            text-transform: uppercase;
            letter-spacing: 0.3px;
        }

        .form-group input,
        .form-group select {
            width: 100%;
            padding: 12px 15px;
            border: 1px solid #ddd;
            border-radius: 4px;
            font-size: 14px;
            transition: 0.3s;
            font-family: 'Segoe UI', sans-serif;
        }

        .form-group input:focus,
        .form-group select:focus {
            outline: none;
            border-color: #f39c12;
            box-shadow: 0 0 0 3px rgba(243, 156, 18, 0.1);
        }

        /* Username Input Special Styling */
        .form-group input[type="text"]::placeholder,
        .form-group input[type="password"]::placeholder {
            color: #999;
        }

        .form-group select {
            cursor: pointer;
            appearance: none;
            background-image: url("data:image/svg+xml;charset=US-ASCII,%3Csvg%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20width%3D%2216%27%20height%3D%2716%27%20viewBox%3D%270%200%2016%2016%27%3E%3Cpath%20fill%3D%22%23333%22%20d%3D%27M4.5%206L8%2010l3.5-4z%27/%3E%3C/svg%3E");
            background-repeat: no-repeat;
            background-position: right 10px center;
            background-size: 14px;
            padding-right: 35px;
        }

        .btn-primary {
            width: 100%;
            padding: 13px 20px;
            background-color: #f39c12;
            color: white;
            border: none;
            border-radius: 4px;
            font-size: 14px;
            font-weight: 700;
            text-transform: uppercase;
            letter-spacing: 0.5px;
            cursor: pointer;
            transition: 0.3s;
            margin-top: 5px;
        }

        .btn-primary:hover {
            background-color: #e67e22;
            box-shadow: 0 4px 12px rgba(243, 156, 18, 0.3);
        }

        .btn-primary:active {
            transform: scale(0.98);
        }

        /* ====== RIGHT CONTENT AREA ====== */
        .content {
            flex: 1;
        }

        .table-section {
            background: white;
            border-radius: 8px;
            padding: 25px;
            box-shadow: 0 2px 8px rgba(0, 0, 0, 0.08);
            margin-bottom: 25px;
            border-top: 4px solid #f39c12;
        }

        .table-section h3 {
            font-size: 16px;
            font-weight: 700;
            color: #222;
            margin-bottom: 20px;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }

        /* ====== TABLE STYLES ====== */
        table {
            width: 100%;
            border-collapse: collapse;
            font-size: 14px;
        }

        table thead {
            background-color: #f9f9f9;
            border-bottom: 2px solid #eee;
        }

        table th {
            padding: 15px 12px;
            text-align: left;
            font-weight: 700;
            color: #333;
            text-transform: uppercase;
            font-size: 12px;
            letter-spacing: 0.3px;
            border-right: 1px solid #eee;
        }

        table th:last-child {
            border-right: none;
        }

        table tbody tr {
            border-bottom: 1px solid #eee;
            transition: 0.2s;
        }

        table tbody tr:hover {
            background-color: #f9f9f9;
        }

        table td {
            padding: 14px 12px;
            border-right: 1px solid #eee;
        }

        table td:last-child {
            border-right: none;
        }

        /* Row Number Column */
        table td:first-child {
            color: #999;
            font-weight: 600;
            width: 40px;
        }

        /* Username Column */
        table td:nth-child(2) {
            font-weight: 600;
            color: #222;
        }

        /* Activity table uses different column ordering */
        #activityTable td:first-child {
            width: auto;
            color: #222;
            font-weight: 700;
        }

        #activityTable td:nth-child(2),
        #activityTable td:nth-child(4) {
            color: #334155;
            font-weight: 600;
        }

        #activityTable td:nth-child(5) {
            min-width: 150px;
        }

        /* Role Badge */
        .role-badge {
            display: inline-block;
            padding: 6px 12px;
            border-radius: 20px;
            font-size: 12px;
            font-weight: 600;
            text-transform: capitalize;
        }

        .role-admin {
            background-color: #f39c12;
            color: white;
        }

        .role-controller {
            background-color: #3498db;
            color: white;
        }

        .role-user {
            background-color: #95a5a6;
            color: white;
        }

        /* Action Buttons */
        .btn-action {
            display: inline-block;
            padding: 8px 15px;
            border-radius: 4px;
            font-size: 12px;
            font-weight: 600;
            text-transform: uppercase;
            border: 2px solid;
            cursor: pointer;
            transition: 0.3s;
            white-space: nowrap;
        }

        .btn-change-password {
            border-color: #3498db;
            color: #3498db;
            background: white;
        }

        .btn-change-password:hover {
            background-color: #3498db;
            color: white;
        }

        .btn-delete {
            border-color: #e74c3c;
            color: #e74c3c;
            background: white;
        }

        .btn-delete:hover {
            background-color: #e74c3c;
            color: white;
        }

        .btn-delete:disabled {
            border-color: #bdc3c7;
            color: #bdc3c7;
            cursor: not-allowed;
            opacity: 0.5;
        }

        .btn-view-details {
            border-color: #16a085;
            color: #16a085;
            background: white;
        }

        .btn-view-details:hover {
            background-color: #16a085;
            color: white;
        }

        /* Empty State */
        .empty-state {
            text-align: center;
            padding: 40px 20px;
            color: #999;
        }

        .empty-state i {
            font-size: 48px;
            color: #ddd;
            margin-bottom: 15px;
        }

        /* Show/Hide Toggle */
        .toggle-show-hide {
            display: inline-flex;
            align-items: center;
            gap: 8px;
            cursor: pointer;
            color: #f39c12;
            font-weight: 600;
            font-size: 12px;
        }

        .toggle-show-hide i {
            font-size: 14px;
        }

        /* Helper Text */
        .note-text {
            font-size: 12px;
            color: #999;
            margin-top: 8px;
            font-style: italic;
        }

        .page-notice-host {
            margin-left: 290px;
            padding: 0 34px 8px;
            transition: margin-left 0.3s ease;
        }

        .sidebar-nav.collapsed ~ .page-notice-host {
            margin-left: 0;
        }

        .page-notice {
            border-radius: 8px;
            padding: 10px 14px;
            font-size: 13px;
            font-weight: 600;
            margin-top: 8px;
            border: 1px solid transparent;
            animation: fadeInNotice 0.2s ease;
        }

        .page-notice.error {
            background: #fff1f2;
            color: #b42318;
            border-color: #fecdd3;
        }

        .page-notice.success {
            background: #ecfdf3;
            color: #067647;
            border-color: #abefc6;
        }

        .page-notice.warning {
            background: #fffaeb;
            color: #b54708;
            border-color: #fedf89;
        }

        @keyframes fadeInNotice {
            from { opacity: 0; transform: translateY(-4px); }
            to { opacity: 1; transform: translateY(0); }
        }

        /* Responsive Design */
        @media (max-width: 1200px) {
            .container {
                flex-direction: column;
                margin-left: 0;
                padding: 24px;
            }

            .sidebar {
                flex: 1;
                width: 100%;
            }

            .content {
                flex: 1;
            }

            .header {
                margin-left: 0;
            }

            .page-notice-host {
                margin-left: 0;
                padding: 0 24px 8px;
            }
        }

        @media (max-width: 1024px) {
            .sidebar-nav {
                transform: translateX(-290px);
                z-index: 1100;
            }

            .sidebar-nav.mobile-open {
                transform: translateX(0);
            }

            .header {
                margin-left: 0;
                padding: 16px 20px;
            }

            .page-notice-host {
                margin-left: 0;
                padding: 0 20px 8px;
            }

            .header h1 {
                font-size: 22px;
            }

            .container {
                margin-left: 0;
                padding: 20px;
                gap: 20px;
            }

            .form-section,
            .table-section {
                padding: 18px;
            }
        }

        @media (max-width: 768px) {
            .header {
                padding: 14px 14px;
            }

            .header h1 {
                font-size: 18px;
                letter-spacing: 0.3px;
                text-transform: none;
            }

            .toggle-sidebar-btn {
                margin-right: 8px;
                font-size: 20px;
                padding: 4px 8px;
            }

            .container {
                padding: 12px;
                gap: 14px;
            }

            .page-notice-host {
                padding: 0 12px 8px;
            }

            .form-section,
            .table-section {
                padding: 14px;
                border-radius: 10px;
            }

            .form-group input,
            .form-group select,
            .btn-primary {
                padding: 11px 12px;
                font-size: 14px;
            }

            #usersTable,
            #activityTable {
                min-width: 720px;
            }

            .permission-row {
                gap: 8px;
            }

            .permission-label {
                font-size: 13px;
            }
        }

        /* Loading Spinner */
        .spinner {
            display: inline-block;
            width: 12px;
            height: 12px;
            border: 2px solid #f3f3f3;
            border-top: 2px solid #f39c12;
            border-radius: 50%;
            animation: spin 0.8s linear infinite;
            margin-right: 8px;
        }

        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }

        /* ====== TOGGLE SWITCH STYLES ====== */
        .permission-row {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 12px 0;
            border-bottom: 1px solid #eee;
        }

        .permission-row:last-child {
            border-bottom: none;
        }

        .permission-label {
            font-weight: 600;
            color: #333;
            flex: 1;
            font-size: 14px;
        }

        .toggle-switch {
            position: relative;
            display: inline-block;
            width: 50px;
            height: 26px;
            margin-left: 10px;
        }

        .toggle-switch input {
            opacity: 0;
            width: 0;
            height: 0;
        }

        .slider {
            position: absolute;
            cursor: pointer;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background-color: #ccc;
            transition: 0.4s;
            border-radius: 26px;
        }

        .slider:before {
            position: absolute;
            content: "";
            height: 20px;
            width: 20px;
            left: 3px;
            bottom: 3px;
            background-color: white;
            transition: 0.4s;
            border-radius: 50%;
        }

        input:checked + .slider {
            background-color: #f39c12;
        }

        input:checked + .slider:before {
            transform: translateX(24px);
        }

        .permission-status {
            font-size: 12px;
            font-weight: 600;
            margin-left: 10px;
            min-width: 50px;
            text-align: center;
        }

        .status-allowed {
            color: #27ae60;
        }

        .status-blocked {
            color: #e74c3c;
        }
    </style>
</head>
<body>
    <!-- SIDEBAR NAVIGATION -->
    <div class="sidebar-nav" id="adminSidebar">
        <div class="sidebar-logo">
            <div class="sidebar-logo-icon">
                <i class="fas fa-industry"></i>
            </div>
            <div class="logo-text">
                <h2>MEKTRON</h2>
                <div class="logo-divider"></div>
                <span class="user-badge">ADMINISTRATOR</span>
            </div>
        </div>
        
        <ul class="sidebar-menu">
            <li class="menu-label">MAIN MENU</li>
            <li><a href="index.php"><span class="icon"><i class="fas fa-house"></i></span> Dashboard</a></li>
            
            <li class="menu-label">ADMINISTRATION</li>
            <li><a href="admin_dashboard.php" class="active"><span class="icon"><i class="fas fa-gear"></i></span> Admin Dashboard</a></li>
            <li><a href="register_emp.php"><span class="icon"><i class="fas fa-user-plus"></i></span> Register Staff</a></li>
            <li><a href="admin_fields.php"><span class="icon"><i class="fas fa-screwdriver-wrench"></i></span> Manage Fields</a></li>
           
            <li><a href="maintenance.php"><span class="icon"><i class="fas fa-toolbox"></i></span> Maintenance</a></li>
            
            <li class="menu-label">OPERATIONS</li>
            <li><a href="all_employees.php"><span class="icon"><i class="fas fa-list-check"></i></span> Staff List</a></li>
            <li><a href="search_emp.php"><span class="icon"><i class="fas fa-magnifying-glass"></i></span> Search</a></li>
            
            <li class="menu-label">PRODUCTION LOGS</li>
            <li><a href="entry_log.php"><span class="icon"><i class="fas fa-pen"></i></span> New Entry</a></li>
            <li><a href="view_log.php"><span class="icon"><i class="fas fa-chart-column"></i></span> View Reports</a></li>
            
            <li class="menu-label">ANALYTICS</li>
            <li><a href="analytics.php"><span class="icon"><i class="fas fa-chart-line"></i></span> Analytics Dashboard</a></li>
            <li><a href="analytics_advanced.php"><span class="icon"><i class="fas fa-chart-pie"></i></span> Advanced Analytics</a></li>
        </ul>

        <div class="sidebar-footer">
            <button onclick="handleLogout()" class="logout-btn">
                <span class="icon"><i class="fas fa-right-from-bracket"></i></span> Logout
            </button>
        </div>
    </div>

    <!-- HEADER -->
    <div class="header">
        <div class="header-title-wrap">
            <button class="toggle-sidebar-btn" onclick="toggleSidebar()" aria-label="Toggle sidebar">
                <i class="fas fa-bars"></i>
            </button>
            <h1><i class="fas fa-user-shield header-page-icon"></i> Administrator Dashboard</h1>
        </div>
        <div class="header-actions">
            <button class="header-notify-btn notification-btn" type="button" title="Notifications" onclick="toggleNotificationPopup()">
                <i class="fas fa-bell"></i>
            </button>
        </div>
    </div>
    <div id="pageNoticeHost" class="page-notice-host"></div>

    <!-- MAIN CONTAINER -->
    <div class="container">
        <!-- LEFT SIDEBAR -->
        <div class="sidebar">
            <!-- CREATE NEW USER FORM -->
            <div class="form-section">
                <h3>Create New User</h3>
                <form id="createUserForm" onsubmit="handleCreateUser(event)">
                    <div class="form-group">
                        <label for="newUsername">New Username</label>
                        <input type="text" id="newUsername" name="username" placeholder="Enter username" required>
                    </div>
                    <div class="form-group">
                        <label for="newPassword">Password</label>
                        <input type="password" id="newPassword" name="password" placeholder="Enter password" required>
                    </div>
                    <div class="form-group">
                        <label for="newRole">Role</label>
                        <select id="newRole" name="role" required>
                            <option value="">-- Select Role</option>
                            <option value="Administrator">Administrator</option>
                            <option value="Supervisor">Supervisor</option>
                            <option value="Controller">Controller</option>

                        </select>
                    </div>
                    <button type="submit" class="btn-primary">Create User</button>
                </form>
            </div>

            <!-- CHANGE OWN PASSWORD FORM -->
            <div class="form-section">
                <h3>Change Own Password</h3>
                <form id="changePasswordForm" onsubmit="handleChangePassword(event)">
                    <div class="form-group">
                        <label for="currentPasswordInput">Current Password</label>
                        <input type="password" id="currentPasswordInput" name="current_password" placeholder="Enter current password" required>
                    </div>
                    <div class="form-group">
                        <label for="newPasswordChange">New Password</label>
                        <input type="password" id="newPasswordChange" name="new_password" placeholder="Enter new password" required>
                    </div>
                    <div class="form-group">
                        <label for="confirmPassword">Confirm New Password</label>
                        <input type="password" id="confirmPassword" name="confirm_password" placeholder="Confirm new password" required>
                    </div>
                    <button type="submit" class="btn-primary">Update Password</button>
                </form>
                <p class="note-text">Passwords must match for confirmation.</p>
            </div>
        </div>

        <!-- RIGHT CONTENT AREA -->
        <div class="content">
            <!-- MANAGE EXISTING USERS TABLE -->
            <div class="table-section">
                <h3>Manage Existing Users</h3>
                <div style="overflow-x: auto;">
                    <table id="usersTable">
                        <thead>
                            <tr>
                                <th>#</th>
                                <th>Username</th>
                                <th>Role</th>
                                <th>Created By</th>
                                <th>Creation Date/Time</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody id="usersTableBody">
                            <tr>
                                <td colspan="6" class="empty-state">
                                    <i class="fas fa-spinner fa-spin"></i> Loading users...
                                </td>
                            </tr>
                        </tbody>
                    </table>
                </div>
            </div>

            <!-- USER ACTIVITY LOG TABLE -->
            <div class="table-section">
                <h3>User Activity Log</h3>
                <p class="note-text">Grouped by username to reduce repeated rows. Click Details to view each record.</p>
                <div style="overflow-x: auto;">
                    <table id="activityTable">
                        <thead>
                            <tr>
                                <th>Username</th>
                                <th>Total Entries</th>
                                <th>Latest Activity</th>
                                <th>Total Records</th>
                                <th>Details</th>
                            </tr>
                        </thead>
                        <tbody id="activityTableBody">
                            <tr>
                                <td colspan="5" class="empty-state">
                                    <i class="fas fa-spinner fa-spin"></i> Loading activity log...
                                </td>
                            </tr>
                        </tbody>
                    </table>
                </div>
            </div>

            <!-- MANAGE PAGE ACCESS (Role-based permissions) -->
            <div class="table-section">
                <h3>Manage Page Access</h3>
                <p class="note-text">Toggle ON/OFF which pages each role can access. Administrator always has full access.</p>
                
                <div id="permissionsContainer">
                    <!-- Permissions will be loaded here via JavaScript -->
                    <div style="text-align: center; padding: 40px;">
                        <i class="fas fa-spinner fa-spin" style="font-size: 24px; color: #f39c12;"></i>
                        <p>Loading permissions...</p>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- CHANGE PASSWORD MODAL (for managed users) -->
    <div id="changePasswordModal" style="display: none; position: fixed; top: 0; left: 0; width: 100%; height: 100%; background: rgba(0,0,0,0.5); z-index: 9999;">
        <div style="position: absolute; top: 50%; left: 50%; transform: translate(-50%, -50%); background: white; padding: 30px; border-radius: 8px; width: 90%; max-width: 400px;">
            <h3 style="margin-bottom: 20px;">Change User Password</h3>
            <div class="form-group">
                <label>Username: <span id="modalUsername" style="font-weight: 700;"></span></label>
            </div>
            <div class="form-group">
                <label for="modalNewPassword">New Password</label>
                <input type="password" id="modalNewPassword" placeholder="Enter new password" required>
            </div>
            <div style="display: flex; gap: 10px;">
                <button class="btn-primary" style="flex: 1;" onclick="submitChangePassword()">Update Password</button>
                <button style="flex: 1; padding: 13px 20px; background: #bdc3c7; border: none; border-radius: 4px; cursor: pointer;" onclick="closePasswordModal()">Cancel</button>
            </div>
        </div>
    </div>

    <script>
        // ============== GLOBAL VARIABLES ==============
        let currentUserId = null;
        let allUsers = [];
        let currentUserRole = null;
        let activitySummaryData = [];
        let pageNoticeTimer = null;
        const initialPageNotice = <?php echo json_encode($pageFlashNotice); ?>;

        // ============== PAGE INITIALIZATION ==============
        document.addEventListener('DOMContentLoaded', () => {
            checkSession();
            loadUsers();
            loadActivityLog();
            loadPermissions();
            if (initialPageNotice) {
                showPageNotice(initialPageNotice, 'warning', false);
            }
        });

        // Handle browser back/forward cache restores after logout
        window.addEventListener('pageshow', function(event) {
            if (event.persisted) {
                fetch('api.php?action=check_session', { cache: 'no-store' })
                    .then(res => res.json())
                    .then(data => {
                        if (!data.logged_in) {
                            window.location.replace('login.php');
                        }
                    })
                    .catch(() => {
                        window.location.replace('login.php');
                    });
            }
        });

        // ============== SESSION CHECK ==============
        function checkSession() {
            fetch('api.php?action=check_session')
                .then(res => res.json())
                .then(data => {
                    if (!data.logged_in) {
                        window.location.replace('login.php');
                    } else {
                        currentUserRole = data.role;
                        // Only "Administrator" role can access this page
                        if (data.role !== 'Administrator') {
                            alert('Access Denied! Only Administrator can access this page.');
                            window.location.replace('index.php');
                        }
                    }
                });
        }

        // ============== LOAD ALL USERS ==============
        function loadUsers() {
            fetch('api.php?action=get_all_users')
                .then(res => res.json())
                .then(data => {
                    if (data.success) {
                        allUsers = data.users;
                        renderUsersTable(data.users);
                    } else {
                        console.error('Failed to load users:', data.error);
                        showPageNotice(data.error || 'Failed to load users', 'error');
                        document.getElementById('usersTableBody').innerHTML = `
                            <tr><td colspan="6" class="empty-state">Error loading users</td></tr>
                        `;
                    }
                })
                .catch(err => {
                    console.error('Error:', err);
                    showPageNotice('Network issue while loading users', 'error');
                    document.getElementById('usersTableBody').innerHTML = `
                        <tr><td colspan="6" class="empty-state">Error loading users</td></tr>
                    `;
                });
        }

        // ============== RENDER USERS TABLE ==============
        function renderUsersTable(users) {
            if (users.length === 0) {
                document.getElementById('usersTableBody').innerHTML = `
                    <tr><td colspan="6" class="empty-state">No users found</td></tr>
                `;
                return;
            }

            const html = users.map((user, index) => {
                const roleClass = `role-${user.role.toLowerCase()}`;
                const canDelete = user.role.toLowerCase() !== 'administrator'; // Can't delete admins
                
                return `
                    <tr>
                        <td>${index + 1}</td>
                        <td>${escapeHtml(user.username)}</td>
                        <td><span class="role-badge ${roleClass}">${user.role}</span></td>
                        <td>${user.created_by || 'Self'}</td>
                        <td>${formatDateTime(user.created_at)}</td>
                        <td>
                            <button class="btn-action btn-change-password" onclick="openPasswordModal(${user.id}, '${escapeHtml(user.username)}')">Change Password</button>
                            <button class="btn-action btn-delete" ${!canDelete ? 'disabled' : ''} onclick="${canDelete ? `deleteUser(${user.id}, '${escapeHtml(user.username)}')` : ''}" title="${!canDelete ? 'Administrative accounts cannot be deleted' : ''}">Delete</button>
                        </td>
                    </tr>
                `;
            }).join('');

            document.getElementById('usersTableBody').innerHTML = html;
        }

        // ============== CREATE NEW USER ==============
        function handleCreateUser(event) {
            event.preventDefault();
            
            const username = document.getElementById('newUsername').value.trim();
            const password = document.getElementById('newPassword').value;
            const role = document.getElementById('newRole').value;

            if (!username || !password || !role) {
                Swal.fire('Validation Error', 'All fields are required', 'warning');
                return;
            }

            const formData = new FormData();
            formData.append('action', 'create_user');
            formData.append('username', username);
            formData.append('password', password);
            formData.append('role', role);

            fetch('api.php', {
                method: 'POST',
                body: formData
            })
            .then(res => res.json())
            .then(data => {
                if (data.success) {
                    Swal.fire('Success', `User "${username}" created successfully!`, 'success');
                    document.getElementById('createUserForm').reset();
                    loadUsers(); // Refresh users table
                } else {
                    Swal.fire('Error', data.error || 'Failed to create user', 'error');
                }
            })
            .catch(err => {
                console.error('Error:', err);
                Swal.fire('Error', 'An error occurred', 'error');
            });
        }

        // ============== DELETE USER ==============
        function deleteUser(userId, username) {
            Swal.fire({
                title: 'Delete User?',
                text: `Are you sure you want to delete "${username}"? This action cannot be undone.`,
                icon: 'warning',
                showCancelButton: true,
                confirmButtonColor: '#e74c3c',
                cancelButtonColor: '#95a5a6',
                confirmButtonText: 'Yes, delete'
            }).then(result => {
                if (result.isConfirmed) {
                    const formData = new FormData();
                    formData.append('action', 'delete_user');
                    formData.append('user_id', userId);

                    fetch('api.php', {
                        method: 'POST',
                        body: formData
                    })
                    .then(res => res.json())
                    .then(data => {
                        if (data.success) {
                            Swal.fire('Deleted', `User "${username}" has been deleted.`, 'success');
                            loadUsers();
                        } else {
                            Swal.fire('Error', data.error || 'Failed to delete user', 'error');
                        }
                    })
                    .catch(err => {
                        console.error('Error:', err);
                        Swal.fire('Error', 'An error occurred', 'error');
                    });
                }
            });
        }

        // ============== CHANGE PASSWORD MODAL ==============
        function openPasswordModal(userId, username) {
            currentUserId = userId;
            document.getElementById('modalUsername').textContent = username;
            document.getElementById('modalNewPassword').value = '';
            document.getElementById('changePasswordModal').style.display = 'block';
        }

        function closePasswordModal() {
            document.getElementById('changePasswordModal').style.display = 'none';
            currentUserId = null;
        }

        function submitChangePassword() {
            const newPassword = document.getElementById('modalNewPassword').value.trim();

            if (!newPassword) {
                Swal.fire('Validation Error', 'Please enter a new password', 'warning');
                return;
            }

            const formData = new FormData();
            formData.append('action', 'change_user_password');
            formData.append('user_id', currentUserId);
            formData.append('new_password', newPassword);

            fetch('api.php', {
                method: 'POST',
                body: formData
            })
            .then(res => res.json())
            .then(data => {
                if (data.success) {
                    Swal.fire('Success', 'Password updated successfully!', 'success');
                    closePasswordModal();
                    loadUsers();
                } else {
                    Swal.fire('Error', data.error || 'Failed to update password', 'error');
                }
            })
            .catch(err => {
                console.error('Error:', err);
                Swal.fire('Error', 'An error occurred', 'error');
            });
        }

        // ============== CHANGE OWN PASSWORD ==============
        function handleChangePassword(event) {
            event.preventDefault();

            const currentPassword = document.getElementById('currentPasswordInput').value;
            const newPassword = document.getElementById('newPasswordChange').value;
            const confirmPassword = document.getElementById('confirmPassword').value;

            if (newPassword !== confirmPassword) {
                Swal.fire('Error', 'Passwords do not match', 'error');
                return;
            }

            const formData = new FormData();
            formData.append('action', 'change_own_password');
            formData.append('current_password', currentPassword);
            formData.append('new_password', newPassword);

            fetch('api.php', {
                method: 'POST',
                body: formData
            })
            .then(res => res.json())
            .then(data => {
                if (data.success) {
                    Swal.fire('Success', 'Your password has been updated!', 'success');
                    document.getElementById('changePasswordForm').reset();
                } else {
                    Swal.fire('Error', data.error || 'Failed to change password', 'error');
                }
            })
            .catch(err => {
                console.error('Error:', err);
                Swal.fire('Error', 'An error occurred', 'error');
            });
        }

        // ============== LOAD ACTIVITY LOG ==============
        function loadActivityLog() {
            fetch('api.php?action=get_activity_log')
                .then(res => res.json())
                .then(data => {
                    if (data.success) {
                        renderActivityLog(data.logs);
                    } else {
                        console.error('Failed to load activity log:', data.error);
                        showPageNotice(data.error || 'Failed to load activity log', 'error');
                        document.getElementById('activityTableBody').innerHTML = `
                            <tr><td colspan="5" class="empty-state">Error loading activity log</td></tr>
                        `;
                    }
                })
                .catch(err => {
                    console.error('Error:', err);
                    showPageNotice('Network issue while loading activity log', 'error');
                    document.getElementById('activityTableBody').innerHTML = `
                        <tr><td colspan="5" class="empty-state">Error loading activity log</td></tr>
                    `;
                });
        }

        // ============== RENDER ACTIVITY LOG ==============
        function renderActivityLog(logs) {
            if (!Array.isArray(logs) || logs.length === 0) {
                document.getElementById('activityTableBody').innerHTML = `
                    <tr><td colspan="5" class="empty-state">No activity records found</td></tr>
                `;
                return;
            }

            const groupedByUser = new Map();
            logs.forEach((log) => {
                const username = String(log.username || 'Unknown').trim() || 'Unknown';
                const groupKey = username.toLowerCase();
                const recordCount = parseInt(log.record_count, 10) || 0;
                const entry = {
                    entryId: Number(log.entry_id) || 0,
                    employeeData: String(log.employee_name || 'N/A'),
                    productionDate: extractProductionDate(log.details),
                    entryTime: String(log.entry_time || ''),
                    recordCount
                };

                if (!groupedByUser.has(groupKey)) {
                    groupedByUser.set(groupKey, {
                        username,
                        entries: [],
                        totalRecords: 0,
                        latestTime: ''
                    });
                }

                const group = groupedByUser.get(groupKey);
                group.entries.push(entry);
                group.totalRecords += recordCount;
                if (!group.latestTime || new Date(entry.entryTime) > new Date(group.latestTime)) {
                    group.latestTime = entry.entryTime;
                }
            });

            activitySummaryData = Array.from(groupedByUser.values())
                .map((group) => {
                    group.entries.sort((a, b) => new Date(b.entryTime) - new Date(a.entryTime));
                    return group;
                })
                .sort((a, b) => new Date(b.latestTime) - new Date(a.latestTime));

            const html = activitySummaryData.map((group, index) => `
                <tr>
                    <td>${escapeHtml(group.username)}</td>
                    <td>${group.entries.length}</td>
                    <td>${formatDateTime(group.latestTime)}</td>
                    <td>${group.totalRecords}</td>
                    <td>
                        <button class="btn-action btn-view-details" onclick="showActivityDetails(${index})">
                            Details (${group.entries.length})
                        </button>
                    </td>
                </tr>
            `).join('');

            document.getElementById('activityTableBody').innerHTML = html;
        }

        function showActivityDetails(groupIndex) {
            const group = activitySummaryData[groupIndex];
            if (!group) return;

            const detailsRows = group.entries.map((entry) => {
                const formattedId = entry.entryId > 0 ? `E${String(entry.entryId).padStart(3, '0')}` : '-';
                const productionDateText = entry.productionDate ? escapeHtml(entry.productionDate) : '-';
                return `
                    <tr>
                        <td style="padding:8px; border:1px solid #e5e7eb;">${formattedId}</td>
                        <td style="padding:8px; border:1px solid #e5e7eb;">${escapeHtml(entry.employeeData)}</td>
                        <td style="padding:8px; border:1px solid #e5e7eb;">${productionDateText}</td>
                        <td style="padding:8px; border:1px solid #e5e7eb;">${formatDateTime(entry.entryTime)}</td>
                        <td style="padding:8px; border:1px solid #e5e7eb; text-align:right;">${entry.recordCount}</td>
                    </tr>
                `;
            }).join('');

            Swal.fire({
                title: `${group.username} - Activity Details`,
                html: `
                    <div style="text-align:left; margin-bottom:10px; color:#475569;">
                        Total Entries: <b>${group.entries.length}</b> | Total Records: <b>${group.totalRecords}</b>
                    </div>
                    <div style="max-height:420px; overflow:auto; border:1px solid #e5e7eb; border-radius:6px;">
                        <table style="width:100%; border-collapse:collapse; font-size:13px;">
                            <thead style="position:sticky; top:0; background:#f8fafc;">
                                <tr>
                                    <th style="padding:8px; border:1px solid #e5e7eb; text-align:left;">Entry ID</th>
                                    <th style="padding:8px; border:1px solid #e5e7eb; text-align:left;">Employee Data</th>
                                    <th style="padding:8px; border:1px solid #e5e7eb; text-align:left;">Log Date</th>
                                    <th style="padding:8px; border:1px solid #e5e7eb; text-align:left;">Entry Time</th>
                                    <th style="padding:8px; border:1px solid #e5e7eb; text-align:right;">Records</th>
                                </tr>
                            </thead>
                            <tbody>${detailsRows}</tbody>
                        </table>
                    </div>
                `,
                width: 960,
                confirmButtonText: 'Close',
                confirmButtonColor: '#f39c12'
            });
        }

        // ============== LOAD ROLE PERMISSIONS ==============
        function loadPermissions() {
            fetch('api.php?action=get_role_permissions')
                .then(res => res.json())
                .then(data => {
                    if (data.success) {
                        renderPermissions(data.permissions);
                    } else {
                        console.error('Failed to load permissions:', data.error);
                        showPageNotice(data.error || 'Failed to load role permissions', 'error');
                        document.getElementById('permissionsContainer').innerHTML = `<p class="empty-state">Error loading permissions</p>`;
                    }
                })
                .catch(err => {
                    console.error('Error:', err);
                    showPageNotice('Network issue while loading role permissions', 'error');
                    document.getElementById('permissionsContainer').innerHTML = `<p class="empty-state">Error loading permissions</p>`;
                });
        }

        // ============== RENDER ROLE PERMISSIONS ==============
        function renderPermissions(permissions) {
            let html = '';
            
            Object.keys(permissions).sort().forEach(role => {
                const pages = permissions[role];
                html += `
                    <div style="margin-bottom: 30px;">
                        <h4 style="margin: 0 0 15px 0; color: #333; font-size: 14px; text-transform: uppercase; letter-spacing: 0.5px;">
                            ${role} Role - Page Access Control
                        </h4>
                        <div style="background: #f9f9f9; padding: 15px; border-radius: 6px;">
                `;
                
                pages.forEach(perm => {
                    if (perm.page_name === 'machines_tools_admin') return;

                    const statusClass = perm.is_allowed ? 'status-allowed' : 'status-blocked';
                    const statusText = perm.is_allowed ? 'ALLOWED' : 'BLOCKED';
                    
                    html += `
                        <div class="permission-row">
                            <span class="permission-label">${escapeHtml(perm.page_label)}</span>
                            <label class="toggle-switch">
                                <input type="checkbox" 
                                       ${perm.is_allowed ? 'checked' : ''} 
                                       onchange="updatePermission('${role}', '${perm.page_name}', this.checked)">
                                <span class="slider"></span>
                            </label>
                            <span class="permission-status ${statusClass}">${statusText}</span>
                        </div>
                    `;
                });
                
                html += `
                        </div>
                    </div>
                `;
            });
            
            if (Object.keys(permissions).length === 0) {
                html = '<p class="empty-state">No permissions found</p>';
            }
            
            document.getElementById('permissionsContainer').innerHTML = html;
        }

        // ============== UPDATE PERMISSION ==============
        function updatePermission(role, pageName, isAllowed) {
            const formData = new FormData();
            formData.append('action', 'update_role_permission');
            formData.append('role', role);
            formData.append('page_name', pageName);
            formData.append('is_allowed', isAllowed ? 1 : 0);

            fetch('api.php', {
                method: 'POST',
                body: formData
            })
            .then(res => res.json())
            .then(data => {
                if (data.success) {
                    // Reload permissions to show updated status
                    loadPermissions();
                    Swal.fire('Success', `Permission updated for ${role}`, 'success');
                    showPageNotice(`Permission updated for ${role}`, 'success');
                } else {
                    Swal.fire('Error', data.error || 'Failed to update permission', 'error');
                    showPageNotice(data.error || 'Failed to update permission', 'error');
                    // Reload to reset toggle
                    loadPermissions();
                }
            })
            .catch(err => {
                console.error('Error:', err);
                Swal.fire('Error', 'An error occurred', 'error');
                showPageNotice('Network issue while updating permission', 'error');
                loadPermissions();
            });
        }

        // ============== UTILITY FUNCTIONS ==============
        function showPageNotice(message, type = 'error', autoHide = true) {
            const host = document.getElementById('pageNoticeHost');
            if (!host || !message) return;

            if (pageNoticeTimer) {
                clearTimeout(pageNoticeTimer);
                pageNoticeTimer = null;
            }

            const safeType = ['error', 'success', 'warning'].includes(type) ? type : 'error';
            host.innerHTML = `<div class="page-notice ${safeType}">${escapeHtml(String(message))}</div>`;

            if (autoHide) {
                pageNoticeTimer = setTimeout(() => {
                    host.innerHTML = '';
                }, 5000);
            }
        }

        function escapeHtml(text) {
            const div = document.createElement('div');
            div.textContent = text;
            return div.innerHTML;
        }

        function formatDateTime(dateString) {
            if (!dateString) return '-';
            const date = new Date(dateString);
            if (Number.isNaN(date.getTime())) return '-';
            return date.toLocaleDateString() + ' ' + date.toLocaleTimeString();
        }

        function extractProductionDate(detailsText) {
            const normalized = String(detailsText || '');
            const dateMatch = normalized.match(/Log Date:\s*([0-9]{4}-[0-9]{2}-[0-9]{2})/i);
            return dateMatch ? dateMatch[1] : '';
        }

        function handleLogout() {
            fetch('api.php?action=logout', { method: 'POST' })
                .then(res => res.json())
                .then(data => {
                    window.location.replace('login.php');
                })
                .catch(() => {
                    window.location.replace('login.php');
                });
        }

        // Close modal when clicking outside
        document.getElementById('changePasswordModal')?.addEventListener('click', function(e) {
            if (e.target === this) {
                closePasswordModal();
            }
        });

        // ============== TOGGLE SIDEBAR ==============
        function toggleSidebar() {
            const sidebar = document.getElementById('adminSidebar');
            if (sidebar) {
                if (window.innerWidth <= 1024) {
                    sidebar.classList.toggle('mobile-open');
                } else {
                    sidebar.classList.toggle('collapsed');
                }
            }
        }

        // Keep sidebar state sane while resizing between desktop/mobile
        window.addEventListener('resize', function() {
            const sidebar = document.getElementById('adminSidebar');
            if (!sidebar) return;
            if (window.innerWidth > 1024) {
                sidebar.classList.remove('mobile-open');
            }
        });
    </script>
    <?php if (isset($_SESSION) && !empty($_SESSION)) { ?>
    <?php include 'notification_ui.html'; ?>
    <?php } ?>
</body>
</html>

