<?php
// Protect this page with session check
include 'session_check.php';
?>
<!DOCTYPE html>
<html lang="en">

<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Manage Fields</title>
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link rel="stylesheet" href="style.css">
    <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>

    <style>
        /* --- GENERAL LAYOUT --- */
        body {
            display: flex;
            margin: 0;
            padding: 0;
            min-height: 100vh;
            font-family: 'Segoe UI', sans-serif;
        }

        .main-content {
            margin-left: 260px;
            width: calc(100% - 260px);
            padding: 40px;
        }

        /* --- HEADER & BUTTONS --- */
        .header-flex {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 25px;
        }

        .header-flex h2 {
            margin: 0;
            color: #333;
            font-weight: 700;
            font-size: 24px;
        }

        .btn-save-order {
            background: #00a65a;
            color: white;
            border: none;
            padding: 10px 20px;
            border-radius: 4px;
            font-weight: bold;
            cursor: pointer;
            display: flex;
            align-items: center;
            gap: 8px;
            transition: 0.3s;
        }

        /* --- ADD FIELD BOX --- */
        .add-box {
            background: white;
            padding: 20px;
            border-radius: 5px;
            box-shadow: 0 2px 5px rgba(0, 0, 0, 0.05);
            margin-bottom: 25px;
            display: flex;
            gap: 15px;
            align-items: center;
            flex-wrap: wrap;
        }

        .form-input {
            flex: 1;
            padding: 10px;
            border: 1px solid #ddd;
            border-radius: 4px;
        }

        .form-select {
            padding: 10px;
            border: 1px solid #ddd;
            border-radius: 4px;
            min-width: 220px;
            cursor: pointer;
        }

        .btn-add {
            background: #3c8dbc;
            color: white;
            border: none;
            padding: 10px 20px;
            border-radius: 4px;
            cursor: pointer;
            font-weight: bold;
            transition: 0.3s;
        }

        /* --- FIELD LIST --- */
        .field-list {
            list-style: none;
            padding: 0;
            margin: 0;
        }

        .field-item {
            background: white;
            padding: 15px 25px;
            margin-bottom: 12px;
            border-radius: 5px;
            box-shadow: 0 2px 5px rgba(0, 0, 0, 0.03);
            display: flex;
            align-items: center;
            justify-content: space-between;
            border-left: 5px solid #d2d6de;
            transition: transform 0.2s, background 0.2s;
            cursor: grab;
        }

        .field-item.dragging {
            opacity: 0.5;
            background: #e3f2fd;
            border: 2px dashed #3c8dbc;
        }

        .field-item[data-type="dropdown"] {
            border-left-color: #00a65a;
        }

        .field-info {
            display: flex;
            align-items: center;
            gap: 15px;
        }

        .field-label {
            font-weight: 600;
            font-size: 16px;
            color: #444;
        }

        .drag-handle {
            color: #aaa;
            cursor: grab;
            font-size: 18px;
            margin-right: 5px;
        }

        /* --- BADGES --- */
        .badge {
            padding: 5px 10px;
            border-radius: 4px;
            font-size: 11px;
            font-weight: bold;
            text-transform: uppercase;
        }

        .badge-text {
            background: #eee;
            color: #666;
        }

        .badge-main {
            background: #d4edda;
            color: #155724;
            border: 1px solid #c3e6cb;
        }

        .badge-link {
            background: #d1ecf1;
            color: #0c5460;
            border: 1px solid #bee5eb;
        }

        /* --- CONTROLS --- */
        .controls {
            display: flex;
            align-items: center;
            gap: 12px;
        }

        .btn-data {
            background: #00c0ef;
            color: white;
            border: none;
            padding: 6px 12px;
            border-radius: 4px;
            cursor: pointer;
            font-size: 13px;
            font-weight: bold;
            display: flex;
            align-items: center;
            gap: 5px;
        }

        .btn-edit {
            background: #f39c12;
            color: white;
            border: none;
            padding: 8px 12px;
            border-radius: 4px;
            cursor: pointer;
        }

        .btn-edit:hover {
            background: #e08e0b;
        }

        .btn-del {
            background: #dd4b39;
            color: white;
            border: none;
            padding: 8px 12px;
            border-radius: 4px;
            cursor: pointer;
        }

        /* --- TOGGLE SWITCH --- */
        .switch {
            position: relative;
            display: inline-block;
            width: 40px;
            height: 22px;
        }

        .switch input {
            opacity: 0;
            width: 0;
            height: 0;
        }

        .slider {
            position: absolute;
            cursor: pointer;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background-color: #ccc;
            transition: .4s;
            border-radius: 34px;
        }

        .slider:before {
            position: absolute;
            content: "";
            height: 16px;
            width: 16px;
            left: 3px;
            bottom: 3px;
            background-color: white;
            transition: .4s;
            border-radius: 50%;
        }

        input:checked+.slider {
            background-color: #00a65a;
        }

        input:checked+.slider:before {
            transform: translateX(18px);
        }

        /* --- MODALS --- */
        .modal {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0, 0, 0, 0.5);
            justify-content: center;
            align-items: center;
            z-index: 2000;
            overflow-y: auto;
            padding: 20px 0;
        }

        /* DATA MODAL - Base Layer */
        .data-modal {
            z-index: 2000;
        }

        /* EDIT MODAL - Higher Layer */
        .edit-modal {
            z-index: 2100;
        }

        .modal-content {
            background: white;
            width: 500px;
            max-width: 90%;
            padding: 25px;
            border-radius: 8px;
            box-shadow: 0 10px 40px rgba(0, 0, 0, 0.3);
            margin: auto;
            flex-shrink: 0;
        }

        .opt-list {
            max-height: 250px;
            overflow-y: auto;
            margin-top: 15px;
            border-top: 1px solid #eee;
        }

        .opt-item {
            padding: 10px;
            border-bottom: 1px solid #eee;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }

        .filter-box {
            background: #fff8e1;
            border: 1px solid #ffe082;
            padding: 15px;
            border-radius: 4px;
            margin-bottom: 15px;
            display: none;
        }

        .edit-group {
            margin-bottom: 15px;
        }

        .edit-group label {
            display: block;
            margin-bottom: 5px;
            font-weight: bold;
            color: #555;
        }

        .edit-group input,
        .edit-group select {
            width: 100%;
            box-sizing: border-box;
        }

        /* ===== MODERN NOTIFICATION STYLES ===== */
        .notification-container {
            position: fixed;
            top: 20px;
            right: 20px;
            z-index: 9999;
            max-width: 400px;
        }

        .notification {
            background: white;
            padding: 16px 20px;
            margin-bottom: 10px;
            border-radius: 8px;
            box-shadow: 0 4px 20px rgba(0, 0, 0, 0.15);
            display: flex;
            align-items: center;
            gap: 12px;
            animation: slideIn 0.3s ease-out;
            border-left: 4px solid #00a65a;
            font-family: 'Segoe UI', sans-serif;
        }
        /* ... (Rest of CSS remains same) ... */
    </style>
</head>

<body>

    <script src="sidebar.js"></script>

    <div id="notificationContainer" class="notification-container"></div>

    <div class="main-content">

        <div class="header-flex">
            <h2>Form Fields Configuration</h2>
            <button class="btn-save-order" onclick="saveOrder()">
                <i class="fas fa-save"></i> Save Order
            </button>
        </div>

        <div class="add-box">
            <input type="text" id="newLabel" class="form-input" placeholder="Field Label (e.g. M/C No)">

            <select id="newType" class="form-select" onchange="toggleParentSelect()">
                <option value="text">Text Input</option>
                <option value="main_cat" style="font-weight:bold; color:#00a65a;">Main Category (Dropdown)</option>
                <option value="sub_cat" style="font-weight:bold; color:#00c0ef;">Dependent Dropdown (Linked)</option>
                <option value="date">Date Picker</option>
            </select>

            <select id="parentFieldSelect" class="form-select" style="display:none;">
                <option value="">Select Parent Field...</option>
            </select>

            <button class="btn-add" onclick="addNew()"><i class="fas fa-plus"></i> Add Field</button>
        </div>

        <ul id="fieldList" class="field-list">
            <li style="text-align:center; padding:20px; color:#999;">Loading fields...</li>
        </ul>

    </div>

    <div id="dataModal" class="modal data-modal">
        <div class="modal-content">
            <div style="display:flex; justify-content:space-between; margin-bottom:15px;">
                <h3 style="margin:0;">Manage Dropdown Data</h3>
                <span onclick="closeModal('dataModal')" style="cursor:pointer; font-size:24px;">&times;</span>
            </div>
            <p id="modalTitle" style="color:#666; font-size:14px; margin-top:0;"></p>

            <div id="toolHelpBox" style="background:#e8f4f8; border: 2px solid #00a65a; padding: 15px; border-radius: 6px; margin-bottom: 15px; display:none;">
                <div style="color:#00a65a; font-weight:bold; margin-bottom:8px;">
                    <i class="fas fa-info-circle"></i> Tool Norm Configuration
                </div>
                <div style="color:#333; font-size:13px; line-height:1.6;">
                    <strong>For TOOL fields:</strong> Set the norm/rate value for each tool below. This rate will be used to automatically calculate production norms in the entry form.<br>
                    <strong>Example:</strong> If "Drilling" tool has norm rate of 5, and duration is 2 hours → Norms = 2 × 5 = 10
                </div>
            </div>

            <div id="parentFilterSection" class="filter-box">
                <div style="font-weight:bold; color:#856404; margin-bottom:5px;"><i class="fas fa-filter"></i> Step 1: Filter by Parent</div>
                <select id="parentOptionFilter" class="form-select" style="width:100%;" onchange="loadOptions()"></select>
            </div>

<div style="background:#f9f9f9; padding:15px; border-radius:5px; margin-bottom:15px; border: 1px solid #e0e0e0;">
                <div style="font-weight:bold; color:#333; margin-bottom:10px; font-size:14px;">
                    <i class="fas fa-plus-circle"></i> Add New Option
                </div>
                <div style="display:flex; gap:10px; align-items:flex-end; flex-wrap:wrap;">
                    
                    <div style="flex:1; min-width: 200px;">
                        <label style="font-size:12px; color:#666; font-weight:bold; display:block; margin-bottom:4px;">Option Name</label>
                        <input type="text" id="newOption" class="form-input" placeholder="Name (e.g. MC-01 or Drill A)">
                    </div>
                    
                    <div style="width:100px;">
                        <label style="font-size:12px; color:#666; font-weight:bold; display:block; margin-bottom:4px;">Norm/Rate</label>
                        <input type="number" id="newOptionNorm" class="form-input" placeholder="Qty/Hr" step="0.1">
                    </div>

                    <div style="width:140px;">
                        <label id="lifeLimitLabel" style="font-size:12px; color:#007bff; font-weight:bold; display:block; margin-bottom:4px;">Life Limit</label>
                        <input type="number" id="newLifeLimit" class="form-input" placeholder="Months / Strokes">
                    </div>

                    <button onclick="addOption()" class="btn-save-order" style="padding: 8px 15px; white-space:nowrap;">
                        <i class="fas fa-plus"></i> Add
                    </button>
                </div>
            </div>
            <div style="font-weight:bold; color:#333; margin-bottom:10px; font-size:14px;">
                <i class="fas fa-list"></i> Existing Options
            </div>
            <div id="optionList" class="opt-list"></div>
        </div>
    </div>

    <div id="editModal" class="modal edit-modal">
        <div class="modal-content">
            <div style="display:flex; justify-content:space-between; margin-bottom:15px;">
                <h3 style="margin:0;">Edit Field</h3>
                <span onclick="closeModal('editModal')" style="cursor:pointer; font-size:24px;">&times;</span>
            </div>

            <input type="hidden" id="editFieldId">

            <div class="edit-group">
                <label>Field Label</label>
                <input type="text" id="editLabel" class="form-input">
            </div>

            <div class="edit-group">
                <label>Field Type</label>
                <select id="editType" class="form-select" onchange="toggleEditParentSelect()">
                    <option value="text">Text Input</option>
                    <option value="date">Date Picker</option>
                    <option value="dropdown">Dropdown / Category</option>
                </select>
            </div>

            <div class="edit-group" id="editParentDiv" style="display:none;">
                <label>Parent Field (For Linked Dropdowns)</label>
                <select id="editParentSelect" class="form-select">
                    <option value="">No Parent (Main Category)</option>
                </select>
            </div>

            <button onclick="saveEdit()" class="btn-save-order" style="width:100%; justify-content:center;">
                <i class="fas fa-save"></i> Save Changes
            </button>
        </div>
    </div>

    <script>
        let currentFieldId = null;
        let currentParentFieldId = null;
        let allFieldsCache = []; // Store fields for edit modal usage

        // ===== MODERN CENTERED MODAL SYSTEM =====
        function showModal(title, message = '', type = 'alert', callback = null) {
            let overlay = document.getElementById('modalOverlay');
            if (!overlay) {
                overlay = document.createElement('div');
                overlay.id = 'modalOverlay';
                overlay.className = 'modal-overlay';
                document.body.appendChild(overlay);

                if (!document.querySelector('style[data-modals]')) {
                    const style = document.createElement('style');
                    style.setAttribute('data-modals', 'true');
                    style.textContent = `
                        .modal-overlay { position: fixed; top: 0; left: 0; width: 100%; height: 100%; background: rgba(0, 0, 0, 0.55); display: flex; align-items: center; justify-content: center; padding: 20px; z-index: 10000; animation: fadeIn 0.3s cubic-bezier(0.4, 0, 0.2, 1); backdrop-filter: blur(2px); }
                        @keyframes fadeIn { from { opacity: 0; } to { opacity: 1; } }
                        @keyframes zoomIn { from { transform: scale(0.85) translateY(-30px); opacity: 0; } to { transform: scale(1) translateY(0); opacity: 1; } }
                        .modal-content { overflow: auto; animation: zoomIn 0.35s cubic-bezier(0.34, 1.56, 0.64, 1); font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif; display: flex; flex-direction: column; }
                        .modal-header { padding: 32px 32px 24px; border-bottom: 1px solid #f0f0f0; display: flex; align-items: flex-start; gap: 16px; flex-shrink: 0; }
                        .modal-icon { font-size: 28px; flex-shrink: 0; width: 44px; height: 44px; display: flex; align-items: center; justify-content: center; border-radius: 50%; margin-top: 2px; }
                        .modal-icon.success { color: #00a65a; background: rgba(0, 166, 90, 0.1); }
                        .modal-icon.error { color: #dd4b39; background: rgba(221, 75, 57, 0.1); }
                        .modal-icon.warning { color: #f39c12; background: rgba(243, 156, 18, 0.1); }
                        .modal-icon.info { color: #3c8dbc; background: rgba(60, 141, 188, 0.1); }
                        .modal-title { font-size: 20px; font-weight: 700; color: #1a1a1a; margin: 0; line-height: 1.4; flex: 1; }
                        .modal-body { padding: 24px 32px; color: #555; font-size: 14px; line-height: 1.7; flex: 1; overflow-y: auto; }
                        .modal-input { width: 100%; padding: 12px 16px; border: 2px solid #e0e0e0; border-radius: 8px; font-size: 14px; font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif; box-sizing: border-box; margin-top: 12px; transition: all 0.2s cubic-bezier(0.4, 0, 0.2, 1); background: #fafafa; }
                        .modal-input:focus { outline: none; border-color: #3c8dbc; background: white; box-shadow: 0 0 0 4px rgba(60, 141, 188, 0.15); }
                        .modal-footer { padding: 24px 32px; border-top: 1px solid #f0f0f0; display: flex; gap: 12px; justify-content: flex-end; background: #fafafa; flex-shrink: 0; }
                        .modal-btn { padding: 12px 32px; border: none; border-radius: 8px; font-size: 14px; font-weight: 600; cursor: pointer; transition: all 0.25s cubic-bezier(0.4, 0, 0.2, 1); min-width: 110px; display: inline-flex; align-items: center; justify-content: center; white-space: nowrap; }
                        .modal-btn-ok { background: linear-gradient(135deg, #00a65a 0%, #008c47 100%); color: white; box-shadow: 0 4px 15px rgba(0, 166, 90, 0.25); }
                        .modal-btn-ok:hover { transform: translateY(-2px); box-shadow: 0 6px 25px rgba(0, 166, 90, 0.35); }
                        .modal-btn-ok:active { transform: translateY(0); }
                        .modal-btn-cancel { background: #f5f5f5; color: #555; box-shadow: 0 2px 8px rgba(0, 0, 0, 0.08); }
                        .modal-btn-cancel:hover { background: #e9e9e9; transform: translateY(-2px); box-shadow: 0 4px 15px rgba(0, 0, 0, 0.12); }
                        .modal-btn-cancel:active { transform: translateY(0); }
                        .modal-btn-delete { background: linear-gradient(135deg, #dd4b39 0%, #c93222 100%); color: white; box-shadow: 0 4px 15px rgba(221, 75, 57, 0.25); }
                        .modal-btn-delete:hover { transform: translateY(-2px); box-shadow: 0 6px 25px rgba(221, 75, 57, 0.35); }
                        .modal-btn-delete:active { transform: translateY(0); }
                        @media (max-width: 600px) { .modal-overlay { padding: 16px; } .modal-content { max-width: 100%; max-height: 90vh; border-radius: 14px; } .modal-header { padding: 24px 20px 18px; gap: 12px; } .modal-icon { width: 40px; height: 40px; } .modal-title { font-size: 18px; } .modal-body { padding: 20px; } .modal-footer { padding: 20px; } .modal-btn { flex: 1; min-width: auto; padding: 11px 20px; } }
                    `;
                    document.head.appendChild(style);
                }
            }

            const icons = {
                success: 'fas fa-check-circle',
                error: 'fas fa-exclamation-circle',
                warning: 'fas fa-warning',
                info: 'fas fa-info-circle'
            };

            const modal = document.createElement('div');
            modal.className = 'modal-content';

            let buttonsHTML = '';
            if (type === 'alert') {
                buttonsHTML = `<button class="modal-btn modal-btn-ok" onclick="this.closest('.modal-overlay').remove()">OK</button>`;
            } else if (type === 'confirm') {
                buttonsHTML = `<button class="modal-btn modal-btn-delete" onclick="window.modalResult = true; this.closest('.modal-overlay').remove()">Confirm</button>
                               <button class="modal-btn modal-btn-cancel" onclick="window.modalResult = false; this.closest('.modal-overlay').remove()">Cancel</button>`;
            }

            modal.innerHTML = `
                <div class="modal-header">
                    <div class="modal-icon ${type}"><i class="fas ${icons[type]}"></i></div>
                    <h2 class="modal-title">${title}</h2>
                </div>
                <div class="modal-body">${message}</div>
                ${buttonsHTML ? `<div class="modal-footer">${buttonsHTML}</div>` : ''}
            `;

            overlay.innerHTML = '';
            overlay.appendChild(modal);
            overlay.onclick = (e) => {
                if (e.target === overlay) overlay.remove();
            };

            if (type === 'confirm' && callback) {
                const checkInterval = setInterval(() => {
                    if (overlay.parentElement === null) {
                        clearInterval(checkInterval);
                        callback(window.modalResult || false);
                    }
                }, 100);
            }
        }

        // ===== MODERN PROMPT MODAL =====
        function showPrompt(title, message = '', defaultValue = '', callback = null) {
            // Initialize modal overlay if needed (styles auto-injected by showModal)
            let overlay = document.getElementById('modalOverlay');
            if (!overlay) {
                overlay = document.createElement('div');
                overlay.id = 'modalOverlay';
                overlay.className = 'modal-overlay';
                document.body.appendChild(overlay);
                // Ensure styles exist by creating them if not present
                if (!document.querySelector('style[data-modals]')) {
                    const style = document.createElement('style');
                    style.setAttribute('data-modals', 'true');
                    style.textContent = `
                        .modal-overlay { position: fixed; top: 0; left: 0; width: 100%; height: 100%; background: rgba(0, 0, 0, 0.55); display: flex; align-items: center; justify-content: center; padding: 20px; z-index: 10000; animation: fadeIn 0.3s cubic-bezier(0.4, 0, 0.2, 1); backdrop-filter: blur(2px); }
                        @keyframes fadeIn { from { opacity: 0; } to { opacity: 1; } }
                        @keyframes zoomIn { from { transform: scale(0.85) translateY(-30px); opacity: 0; } to { transform: scale(1) translateY(0); opacity: 1; } }
                        .modal-content { background: white; border-radius: 8px; box-shadow: 0 25px 80px rgba(0, 0, 0, 0.35); max-width: 540px; width: 100%; max-height: 85vh; overflow: auto; animation: zoomIn 0.35s cubic-bezier(0.34, 1.56, 0.64, 1); font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif; display: flex; flex-direction: column; }
                        .modal-header { padding: 32px 32px 24px; border-bottom: 1px solid #f0f0f0; display: flex; align-items: flex-start; gap: 16px; flex-shrink: 0; }
                        .modal-icon { font-size: 28px; flex-shrink: 0; width: 44px; height: 44px; display: flex; align-items: center; justify-content: center; border-radius: 50%; margin-top: 2px; }
                        .modal-icon.success { color: #00a65a; background: rgba(0, 166, 90, 0.1); }
                        .modal-icon.error { color: #dd4b39; background: rgba(221, 75, 57, 0.1); }
                        .modal-icon.warning { color: #f39c12; background: rgba(243, 156, 18, 0.1); }
                        .modal-icon.info { color: #3c8dbc; background: rgba(60, 141, 188, 0.1); }
                        .modal-title { font-size: 20px; font-weight: 700; color: #1a1a1a; margin: 0; line-height: 1.4; flex: 1; }
                        .modal-body { padding: 24px 32px; color: #555; font-size: 14px; line-height: 1.7; flex: 1; overflow-y: auto; }
                        .modal-input { width: 100%; padding: 12px 16px; border: 2px solid #e0e0e0; border-radius: 8px; font-size: 14px; font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif; box-sizing: border-box; margin-top: 12px; transition: all 0.2s cubic-bezier(0.4, 0, 0.2, 1); background: #fafafa; }
                        .modal-input:focus { outline: none; border-color: #3c8dbc; background: white; box-shadow: 0 0 0 4px rgba(60, 141, 188, 0.15); }
                        .modal-footer { padding: 24px 32px; border-top: 1px solid #f0f0f0; display: flex; gap: 12px; justify-content: flex-end; background: #fafafa; flex-shrink: 0; }
                        .modal-btn { padding: 12px 32px; border: none; border-radius: 8px; font-size: 14px; font-weight: 600; cursor: pointer; transition: all 0.25s cubic-bezier(0.4, 0, 0.2, 1); min-width: 110px; display: inline-flex; align-items: center; justify-content: center; white-space: nowrap; }
                        .modal-btn-ok { background: linear-gradient(135deg, #00a65a 0%, #008c47 100%); color: white; box-shadow: 0 4px 15px rgba(0, 166, 90, 0.25); }
                        .modal-btn-ok:hover { transform: translateY(-2px); box-shadow: 0 6px 25px rgba(0, 166, 90, 0.35); }
                        .modal-btn-ok:active { transform: translateY(0); }
                        .modal-btn-cancel { background: #f5f5f5; color: #555; box-shadow: 0 2px 8px rgba(0, 0, 0, 0.08); }
                        .modal-btn-cancel:hover { background: #e9e9e9; transform: translateY(-2px); box-shadow: 0 4px 15px rgba(0, 0, 0, 0.12); }
                        .modal-btn-cancel:active { transform: translateY(0); }
                        .modal-btn-delete { background: linear-gradient(135deg, #dd4b39 0%, #c93222 100%); color: white; box-shadow: 0 4px 15px rgba(221, 75, 57, 0.25); }
                        .modal-btn-delete:hover { transform: translateY(-2px); box-shadow: 0 6px 25px rgba(221, 75, 57, 0.35); }
                        .modal-btn-delete:active { transform: translateY(0); }
                        @media (max-width: 600px) { .modal-overlay { padding: 16px; } .modal-content { max-width: 100%; max-height: 90vh; border-radius: 14px; } .modal-header { padding: 24px 20px 18px; gap: 12px; } .modal-icon { width: 40px; height: 40px; } .modal-title { font-size: 18px; } .modal-body { padding: 20px; } .modal-footer { padding: 20px; } .modal-btn { flex: 1; min-width: auto; padding: 11px 20px; } }
                    `;
                    document.head.appendChild(style);
                }
            }

            const modal = document.createElement('div');
            modal.className = 'modal-content';
            const inputId = 'promptInput_' + Date.now();

            modal.innerHTML = `
                <div class="modal-header">
                    <div class="modal-icon info"><i class="fas fa-edit"></i></div>
                    <h2 class="modal-title">${title}</h2>
                </div>
                <div class="modal-body">
                    ${message}
                    <input type="text" id="${inputId}" class="modal-input" value="${defaultValue || ''}" placeholder="">
                </div>
                <div class="modal-footer">
                    <button class="modal-btn modal-btn-cancel" onclick="window.promptResult = null; this.closest('.modal-overlay').remove()">Cancel</button>
                    <button class="modal-btn modal-btn-ok" onclick="window.promptResult = document.getElementById('${inputId}').value; this.closest('.modal-overlay').remove()">OK</button>
                </div>
            `;

            overlay.innerHTML = '';
            overlay.style.display = 'flex';
            document.body.style.overflow = 'hidden';
            overlay.appendChild(modal);
            overlay.onclick = (e) => {
                if (e.target === overlay) {
                    window.promptResult = null;
                    document.body.style.overflow = '';
                    overlay.remove();
                }
            };

            setTimeout(() => document.getElementById(inputId)?.focus(), 100);

            if (callback) {
                const checkInterval = setInterval(() => {
                    if (overlay.parentElement === null) {
                        clearInterval(checkInterval);
                        document.body.style.overflow = '';
                        callback(window.promptResult);
                    }
                }, 100);
            }
        }

        document.addEventListener("DOMContentLoaded", function() {
            loadFields();

            // Auto-capitalize first letter in Option Name input
            const newOptionInput = document.getElementById('newOption');
            if (newOptionInput) {
                newOptionInput.addEventListener('input', function() {
                    if (this.value.length > 0) {
                        this.value = this.value.charAt(0).toUpperCase() + this.value.slice(1);
                    }
                });
            }

            // Drag listeners
            const list = document.getElementById('fieldList');
            list.addEventListener('dragover', function(e) {
                e.preventDefault();
                const afterElement = getDragAfterElement(list, e.clientY);
                const draggable = document.querySelector('.dragging');
                if (draggable) {
                    if (afterElement == null) list.appendChild(draggable);
                    else list.insertBefore(draggable, afterElement);
                }
            });
        });

        // --- 1. LOAD FIELDS (UPDATED WITH LIFE BUTTON) ---
        async function loadFields() {
            try {
                const res = await fetch('api.php?action=get_custom_fields');
                const data = await res.json();
                allFieldsCache = data; // Cache data for Edit Modal

                const list = document.getElementById('fieldList');
                const parentSel = document.getElementById('parentFieldSelect');

                list.innerHTML = '';
                parentSel.innerHTML = '<option value="">Select Parent Field...</option>';

                data.forEach(f => {
                    const isDrop = f.field_type === 'dropdown';

                    // Populate Parent Dropdown for "Add New" form
                    if (isDrop) {
                        parentSel.innerHTML += `<option value="${f.id}">${f.field_label}</option>`;
                    }

                    // Determine Badge & Data Button
                    let badgeClass = 'badge-text';
                    let badgeText = 'Text Input';
                    let dataBtn = '';
                    
                    // --- NEW: LIFE BUTTON LOGIC ---
                  // --- NEW: LIFE BUTTON LOGIC ---
            let lifeBtn = ''; 
            const labelLower = f.field_label.toLowerCase();

            // Check if this is a Machine field -> Send to MACHINE view
            if (labelLower.includes('m/c') || labelLower.includes('machine')) {
                lifeBtn = `<a href="maintenance.php?view=machine" target="_blank" class="btn-data" style="background:#605ca8; text-decoration:none; margin-right:5px; display:inline-flex; align-items:center; gap:5px;" title="Manage Machine Life">
                                <i class="fas fa-heartbeat"></i> Life
                           </a>`;
            } 
            // Check if this is a Tool field -> Send to TOOL view
            else if (labelLower.includes('tool')) {
                lifeBtn = `<a href="maintenance.php?view=tool" target="_blank" class="btn-data" style="background:#d81b60; text-decoration:none; margin-right:5px; display:inline-flex; align-items:center; gap:5px;" title="Manage Tool Strokes">
                                <i class="fas fa-hammer"></i> Life
                           </a>`;
            }

                    if (isDrop) {
                        if (f.parent_field_id) {
                            const parentName = data.find(p => p.id == f.parent_field_id)?.field_label || 'Parent';
                            badgeClass = 'badge-link';
                            badgeText = `LINKED TO: ${parentName}`;
                        } else {
                            badgeClass = 'badge-main';
                            badgeText = `MAIN CATEGORY`;
                        }
                        dataBtn = `<button class="btn-data" onclick="openDataModal(${f.id}, '${f.field_label}', ${f.parent_field_id})"><i class="fas fa-list"></i> Data</button>`;
                    } else if (f.field_type === 'date') {
                        badgeText = 'DATE PICKER';
                    }

                    // Render List Item
                    list.innerHTML += `
                        <li class="field-item" draggable="true" data-id="${f.id}" data-type="${isDrop ? 'dropdown' : 'text'}">
                            <div class="field-info">
                                <i class="fas fa-grip-vertical drag-handle"></i>
                                <span class="field-label">${f.field_label}</span>
                                <span class="badge ${badgeClass}">${badgeText}</span>
                            </div>
                            <div class="controls">
                                ${lifeBtn}
                                ${dataBtn}
                                
                                <label class="switch">
                                    <input type="checkbox" ${f.is_active == 1 ? 'checked' : ''} onchange="toggleField(${f.id}, this.checked)">
                                    <span class="slider"></span>
                                </label>
                                
                                <button class="btn-edit" onclick="openEditModal(${f.id})">
                                    <i class="fas fa-pencil-alt"></i>
                                </button>

                                <button class="btn-del" onclick="deleteField(${f.id})">
                                    <i class="fas fa-trash"></i>
                                </button>
                            </div>
                        </li>`;
                });

                attachDragListeners();

            } catch (e) {
                console.error(e);
                list.innerHTML = '<li style="text-align:center; padding:20px; color:red;">Error loading fields</li>';
            }
        }

        // --- DRAG FUNCTIONS ---
        function attachDragListeners() {
            const items = document.querySelectorAll('.field-item');
            items.forEach(item => {
                item.addEventListener('dragstart', () => {
                    item.classList.add('dragging');
                });
                item.addEventListener('dragend', () => {
                    item.classList.remove('dragging');
                });
            });
        }

        function getDragAfterElement(container, y) {
            const draggableElements = [...container.querySelectorAll('.field-item:not(.dragging)')];
            return draggableElements.reduce((closest, child) => {
                const box = child.getBoundingClientRect();
                const offset = y - box.top - box.height / 2;
                if (offset < 0 && offset > closest.offset) return {
                    offset: offset,
                    element: child
                };
                else return closest;
            }, {
                offset: Number.NEGATIVE_INFINITY
            }).element;
        }

        // --- ACTION FUNCTIONS ---
        async function saveOrder() {
            const items = document.querySelectorAll('.field-item');
            const fd = new FormData();
            items.forEach(item => fd.append('order[]', item.getAttribute('data-id')));
            const res = await fetch('api.php?action=update_field_order', {
                method: 'POST',
                body: fd
            });
            if ((await res.json()).success) showModal('✓ Success', 'Field order saved successfully!', 'success');
        }

        function toggleParentSelect() {
            const type = document.getElementById('newType').value;
            document.getElementById('parentFieldSelect').style.display = (type === 'sub_cat') ? 'block' : 'none';
        }

        async function addNew() {
            const lbl = document.getElementById('newLabel').value;
            const uiType = document.getElementById('newType').value;
            const parentId = document.getElementById('parentFieldSelect').value;

            if (!lbl) {
                showModal('⚠ Required', 'Please enter a field label', 'warning');
                return;
            }
            const fd = new FormData();
            fd.append('label', lbl);

            if (uiType === 'main_cat') {
                fd.append('type', 'dropdown');
            } else if (uiType === 'sub_cat') {
                fd.append('type', 'dropdown');
                if (!parentId) {
                    showModal('⚠ Required', 'Please select a parent field for linked dropdown', 'warning');
                    return;
                }
                fd.append('parent_id', parentId);
            } else {
                fd.append('type', uiType);
            }

            await fetch('api.php?action=add_custom_field', {
                method: 'POST',
                body: fd
            });
            document.getElementById('newLabel').value = '';
            showModal('✓ Success', 'New field added successfully!', 'success');
            loadFields();
        }

        function toggleField(id, status) {
            const fd = new FormData();
            fd.append('id', id);
            fd.append('status', status ? 1 : 0);
            fetch('api.php?action=toggle_field', {
                method: 'POST',
                body: fd
            });
        }

        async function deleteField(id) {
            showModal('⚠ Confirm Delete', 'Are you sure you want to delete this field?', 'confirm', (result) => {
                if (result) {
                    const fd = new FormData();
                    fd.append('id', id);
                    fetch('api.php?action=delete_custom_field', {
                        method: 'POST',
                        body: fd
                    }).then(() => {
                        showModal('✓ Deleted', 'Field has been deleted', 'success');
                        loadFields();
                    });
                }
            });
        }

        // --- DATA MODAL FUNCTIONS ---
async function openDataModal(id, label, parentId) {
            currentFieldId = id;
            currentParentFieldId = parentId;
            document.getElementById('modalTitle').innerText = "Adding options for: " + label;
            
            const dataModal = document.getElementById('dataModal');
            dataModal.style.display = 'flex';
            dataModal.style.zIndex = '2000';
            document.body.style.overflow = 'hidden';

            // --- NEW: DYNAMIC LABEL LOGIC ---
            const lbl = label.toLowerCase();
            const lifeLabel = document.getElementById('lifeLimitLabel');
            const lifeInput = document.getElementById('newLifeLimit');

            // Default State
            lifeLabel.innerText = "Life Limit (Optional)";
            lifeInput.placeholder = "0 if none";

            // If it is a Machine Field
            if (lbl.includes('m/c') || lbl.includes('machine')) {
                lifeLabel.innerText = "Maint. Interval (Months)";
                lifeInput.placeholder = "e.g. 3";
            } 
            // If it is a Tool Field
            else if (lbl.includes('tool')) {
                lifeLabel.innerText = "Max Stroke Life";
                lifeInput.placeholder = "e.g. 100000";
            }
            // --------------------------------

            // Show help box if this is a TOOL field
            const isToolField = lbl.includes('tool');
            document.getElementById('toolHelpBox').style.display = isToolField ? 'block' : 'none';

            const filterBox = document.getElementById('parentFilterSection');
            const parentSel = document.getElementById('parentOptionFilter');

            if (parentId) {
                filterBox.style.display = 'block';
                const res = await fetch(`api.php?action=get_options&field_id=${parentId}`);
                const parentOpts = await res.json();
                parentSel.innerHTML = '<option value="">-- Select Parent Option --</option>';
                parentOpts.forEach(p => parentSel.innerHTML += `<option value="${p.id}">${p.label}</option>`);
                document.getElementById('optionList').innerHTML = '<div style="padding:10px;text-align:center;">Select parent option first.</div>';
            } else {
                filterBox.style.display = 'none';
                loadOptions();
            }
        }
        function closeModal(modalId) {
            const modal = document.getElementById(modalId);
            if (modal) {
                modal.style.display = 'none';
                document.body.style.overflow = '';
                // Reset z-index for data modal
                if (modalId === 'dataModal') {
                    modal.style.zIndex = '2000';
                }
                // Ensure edit modal is hidden
                if (modalId === 'editModal') {
                    modal.style.zIndex = '2100';
                }
            }
        }

        async function loadOptions() {
            let url = `api.php?action=get_options&field_id=${currentFieldId}`;
            if (currentParentFieldId) {
                const parentOptId = document.getElementById('parentOptionFilter').value;
                if (!parentOptId) return;
                url += `&parent_opt_id=${parentOptId}`;
            }
            const res = await fetch(url);
            const opts = await res.json();
            const box = document.getElementById('optionList');
            box.innerHTML = '';
            if (opts.length === 0) {
                box.innerHTML = '<div style="text-align:center;color:#999;padding:20px;"><i class="fas fa-inbox"></i><br>No options added yet</div>';
            }
            opts.forEach(o => {
                const normDisplay = o.norm_value ? `<strong>${o.norm_value}</strong>` : '<span style="color:#d9534f;">Not set</span>';
                box.innerHTML += `
                    <div class="opt-item" style="padding:12px; display:flex; justify-content:space-between; align-items:center; background:#f9f9f9; margin-bottom:8px; border-radius:4px; border: 1px solid #e0e0e0;">
                        <div style="flex:1;">
                            <div style="font-weight:bold; color:#333;">${o.label}</div>
                            <div style="font-size:12px; color:#666; margin-top:3px;">
                                <i class="fas fa-tachometer-alt"></i> Norm/Rate: ${normDisplay} per hour
                            </div>
                        </div>
                        <div style="display:flex; gap:8px; align-items:center;">
                            <button class="btn-data" onclick="editOptionNorm(${o.id}, '${o.label}', ${o.norm_value || 0})" style="background:#f39c12; padding:6px 12px; font-size:12px;">
                                <i class="fas fa-pencil"></i> Edit Rate
                            </button>
                            <button style="background:#dd4b39; color:white; border:none; padding:6px 10px; border-radius:4px; cursor:pointer; font-size:12px;" onclick="deleteOption(${o.id})">
                                <i class="fas fa-trash"></i>
                            </button>
                        </div>
                    </div>`;
            });
        }

async function addOption() {
            let val = document.getElementById('newOption').value.trim();
            const norm = document.getElementById('newOptionNorm').value;
            const life = document.getElementById('newLifeLimit').value || 0; // Get the Life Limit value

            // Capitalize first letter
            if (val.length > 0) {
                val = val.charAt(0).toUpperCase() + val.slice(1);
            }

            if (!val) {
                showModal('⚠ Required', 'Please enter an option name', 'warning');
                return;
            }
            // Norm is optional if you want, but strictly enforced here
            if (!norm) { 
               // showModal('⚠ Required', 'Please enter a norm/rate value', 'warning');
               // return;
            }

            const fd = new FormData();
            fd.append('field_id', currentFieldId);
            fd.append('label', val);
            fd.append('norm_value', norm || 0);
            fd.append('life_limit', life); // SEND TO SERVER

            if (currentParentFieldId) {
                const parentOptId = document.getElementById('parentOptionFilter').value;
                if (!parentOptId) {
                    showModal('⚠ Required', 'Please select a parent option first', 'warning');
                    return;
                }
                fd.append('parent_opt_id', parentOptId);
            }

            await fetch('api.php?action=add_option', {
                method: 'POST',
                body: fd
            });

            // Clear inputs
            document.getElementById('newOption').value = '';
            document.getElementById('newOptionNorm').value = '';
            document.getElementById('newLifeLimit').value = '';
            
            showModal('✓ Added', `Option "${val}" added successfully!`, 'success');
            loadOptions();
        }

        async function deleteOption(oid) {
            showModal('⚠ Confirm Delete', 'Are you sure you want to delete this option?', 'confirm', (result) => {
                if (result) {
                    const fd = new FormData();
                    fd.append('id', oid);
                    fetch('api.php?action=delete_option', {
                        method: 'POST',
                        body: fd
                    }).then(() => {
                        showModal('✓ Deleted', 'Option has been removed', 'success');
                        loadOptions();
                    });
                }
            });
        }

        // NEW: Edit Option Norm Value
        function editOptionNorm(optionId, label, currentNorm) {
            showPrompt(
                `Edit Norm/Rate for "${label}":`,
                `<strong>Current: ${currentNorm || 'Not set'}</strong>`,
                currentNorm || '',
                (newNorm) => {
                    if (newNorm === null) return; // User cancelled

                    if (newNorm === '') {
                        showModal('⚠ Error', 'Norm value cannot be empty', 'error');
                        return;
                    }

                    if (isNaN(newNorm)) {
                        showModal('⚠ Error', 'Please enter a valid number', 'error');
                        return;
                    }

                    const fd = new FormData();
                    fd.append('option_id', optionId);
                    fd.append('norm_value', parseFloat(newNorm));

                    fetch('api.php?action=update_option_norm', {
                            method: 'POST',
                            body: fd
                        })
                        .then(res => res.json())
                        .then(data => {
                            if (data.success) {
                                showModal('✓ Updated', `Norm rate updated to ${newNorm}`, 'success');
                                loadOptions(); // Refresh list
                            } else {
                                showModal('✗ Error', data.error || 'Unknown error occurred', 'error');
                            }
                        })
                        .catch(e => {
                            console.error(e);
                            showModal('✗ Error', 'Failed to update norm value', 'error');
                        });
                }
            );
        }

        // --- ✅ NEW EDIT MODAL FUNCTIONS ---

        function openEditModal(id) {
            // Find the field data from cache
            const field = allFieldsCache.find(f => f.id == id);
            if (!field) return;

            // Populate Modal
            document.getElementById('editFieldId').value = field.id;
            document.getElementById('editLabel').value = field.field_label;
            document.getElementById('editType').value = field.field_type;

            // Handle Parent Logic
            const parentDiv = document.getElementById('editParentDiv');
            const parentSel = document.getElementById('editParentSelect');

            // Re-populate parent dropdown (excluding itself)
            parentSel.innerHTML = '<option value="">No Parent (Main Category)</option>';
            allFieldsCache.forEach(f => {
                if (f.field_type === 'dropdown' && f.id != id) {
                    parentSel.innerHTML += `<option value="${f.id}">${f.field_label}</option>`;
                }
            });

            if (field.field_type === 'dropdown') {
                parentDiv.style.display = 'block';
                parentSel.value = field.parent_field_id || '';
            } else {
                parentDiv.style.display = 'none';
            }

            const editModal = document.getElementById('editModal');
            editModal.style.display = 'flex';
            editModal.style.zIndex = '2100';
            document.body.style.overflow = 'hidden';
        }

        function toggleEditParentSelect() {
            const type = document.getElementById('editType').value;
            document.getElementById('editParentDiv').style.display = (type === 'dropdown') ? 'block' : 'none';
        }

        async function saveEdit() {
            const id = document.getElementById('editFieldId').value;
            const label = document.getElementById('editLabel').value;
            const type = document.getElementById('editType').value;
            const parentId = document.getElementById('editParentSelect').value;

            if (!label) {
                showModal('⚠ Required', 'Field label cannot be empty', 'warning');
                return;
            }

            const fd = new FormData();
            fd.append('id', id);
            fd.append('label', label);
            fd.append('type', type);
            if (type === 'dropdown') fd.append('parent_id', parentId);

            try {
                const res = await fetch('api.php?action=update_custom_field', {
                    method: 'POST',
                    body: fd
                });
                const data = await res.json();

                if (data.success) {
                    showModal('✓ Success', 'Field updated successfully!', 'success');
                    document.getElementById('editModal').style.display = 'none';
                    loadFields(); // Refresh list to show changes
                } else {
                    showModal('✗ Error', data.error || 'Failed to update field', 'error');
                }
            } catch (e) {
                console.error(e);
                showModal('✗ Error', 'An error occurred while updating', 'error');
            }
        }
    </script>
</body>

</html>