<?php
// Protect this page with session check
include 'session_check.php';
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Mektron - Advanced Analytics Dashboard</title>
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <script src="https://cdn.jsdelivr.net/npm/chart.js@3.9.1/dist/chart.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
    <link rel="stylesheet" href="style.css">
    
    <style>
        body {
            background: #f4f6f9;
            font-family: 'Segoe UI', sans-serif;
            margin: 0;
            overflow-x: hidden;
        }

        .main-content {
            margin-left: 0;
            padding: 25px;
            width: 100%;
            box-sizing: border-box;
            min-height: 100vh;
            transition: margin-left 0.3s ease;
        }

        .sidebar {
            position: fixed;
            top: 0;
            left: -260px;
            width: 260px;
            height: 100%;
            background: #013B3B;
            transition: left 0.3s ease;
            z-index: 1000;
            overflow-y: auto;
        }

        .sidebar.active { left: 0; }
        .main-content.shifted { margin-left: 260px; }

        .analytics-container {
            background: white;
            padding: 30px;
            border-radius: 12px;
            box-shadow: 0 4px 15px rgba(0,0,0,0.05);
        }

        .analytics-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 30px;
            border-bottom: 2px solid #eee;
            padding-bottom: 20px;
        }

        .analytics-header h1 {
            margin: 0;
            color: #2c3e50;
            font-size: 28px;
        }

        .btn-toggle-sidebar {
            background: #013B3B;
            color: white;
            border: none;
            padding: 10px 15px;
            border-radius: 6px;
            cursor: pointer;
            font-size: 16px;
            transition: 0.3s;
            display: flex;
            align-items: center;
            gap: 8px;
        }

        .btn-toggle-sidebar:hover { background: #00564f; }

        /* ===== HAMBURGER MENU BUTTON ===== */
        .menu-toggle-btn {
            background: none;
            border: none;
            font-size: 24px;
            color: #013B3B;
            cursor: pointer;
            margin-right: 15px;
            padding: 5px;
            display: inline-flex;
            align-items: center;
            justify-content: center;
            border-radius: 4px;
            transition: 0.2s;
        }
        .menu-toggle-btn:hover { background: #e0e0e0; }

        /* ===== OVERLAY FOR SIDEBAR ===== */
        .overlay {
            display: none;
            position: fixed;
            top: 0; left: 0; width: 100%; height: 100%;
            background: rgba(0,0,0,0.5);
            z-index: 999;
        }
        .overlay.active { display: block; }

        /* ===== HIERARCHICAL CONTROLS ===== */
        .hierarchical-section {
            background: #f9f9f9;
            padding: 25px;
            border-radius: 8px;
            margin-bottom: 30px;
            border-left: 5px solid #1abc9c;
        }

        .hierarchy-title {
            font-weight: 700;
            color: #2c3e50;
            margin-bottom: 20px;
            font-size: 15px;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }

        .hierarchy-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(280px, 1fr));
            gap: 20px;
            margin-bottom: 20px;
        }

        .hierarchy-step {
            background: white;
            padding: 20px;
            border-radius: 8px;
            border: 2px solid #e0e0e0;
            transition: 0.3s;
        }

        .hierarchy-step.active {
            border-color: #1abc9c;
            background: #f0fffe;
        }

        .step-label {
            display: block;
            font-weight: 600;
            color: #555;
            margin-bottom: 10px;
            font-size: 13px;
            text-transform: uppercase;
        }

        .hierarchy-step select {
            width: 100%;
            padding: 12px;
            border: 1px solid #ddd;
            border-radius: 6px;
            font-size: 14px;
            box-sizing: border-box;
            transition: 0.3s;
        }

        .hierarchy-step select:focus {
            border-color: #1abc9c;
            outline: none;
            box-shadow: 0 0 0 3px rgba(26, 188, 156, 0.1);
        }

        .step-indicator {
            display: inline-block;
            background: #1abc9c;
            color: white;
            width: 30px;
            height: 30px;
            border-radius: 50%;
            text-align: center;
            line-height: 30px;
            font-weight: 700;
            margin-bottom: 10px;
            font-size: 14px;
        }

        .step-arrow {
            text-align: center;
            color: #1abc9c;
            font-size: 24px;
            margin: 0 10px;
            align-self: flex-end;
            margin-bottom: 10px;
        }

        .dependent-fields-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(280px, 1fr));
            gap: 20px;
            margin-bottom: 20px;
        }

        .dependent-field-selector {
            background: white;
            padding: 20px;
            border-radius: 8px;
            border: 2px solid #e0e0e0;
            cursor: pointer;
            transition: 0.3s;
        }

        .dependent-field-selector:hover {
            border-color: #1abc9c;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }

        .dependent-field-selector.selected {
            border-color: #1abc9c;
            background: #f0fffe;
        }

        .field-selector-label {
            display: block;
            font-weight: 600;
            color: #2c3e50;
            margin-bottom: 8px;
            font-size: 14px;
        }

        .field-selector-checkbox {
            width: 18px;
            height: 18px;
            cursor: pointer;
            margin-right: 8px;
            accent-color: #1abc9c;
        }

        /* ===== STATS CARDS ===== */
        .machine-stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(180px, 1fr));
            gap: 15px;
            margin-bottom: 25px;
        }

        .machine-stat-card {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 20px;
            border-radius: 10px;
            box-shadow: 0 4px 15px rgba(0,0,0,0.1);
        }

        .stat-label { font-size: 12px; opacity: 0.9; text-transform: uppercase; }
        .stat-value { font-size: 26px; font-weight: 800; margin-top: 8px; }

        .stat-card-ops { background: linear-gradient(135deg, #f093fb 0%, #f5576c 100%); }
        .stat-card-emp { background: linear-gradient(135deg, #4facfe 0%, #00f2fe 100%); }
        .stat-card-days { background: linear-gradient(135deg, #43e97b 0%, #38f9d7 100%); }

        /* ===== CHARTS & TABLES SECTION ===== */
        .field-analysis-section {
            margin-top: 30px;
            padding-top: 30px;
            border-top: 2px solid #eee;
        }

        .chart-container {
            background: linear-gradient(160deg, #ffffff 0%, #f8fbff 65%, #f4fff8 100%);
            padding: 25px;
            border-radius: 10px;
            box-shadow: 0 8px 24px rgba(15, 23, 42, 0.08);
            border: 1px solid rgba(102, 126, 234, 0.12);
            border-top: 4px solid #1abc9c;
            transition: all 0.3s ease;
            position: relative;
            overflow: hidden;
        }

        .chart-container::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            height: 4px;
            background: linear-gradient(90deg, #1abc9c, #16a085);
            opacity: 0;
            transition: opacity 0.3s ease;
        }

        .chart-container:hover {
            box-shadow: 0 8px 25px rgba(0,0,0,0.12);
            transform: translateY(-3px);
        }

        .chart-container:hover::before {
            opacity: 1;
        }

        .chart-title {
            font-size: 15px;
            font-weight: 700;
            color: #2c3e50;
            margin-bottom: 20px;
            display: flex;
            align-items: center;
            gap: 10px;
            padding-bottom: 15px;
            border-bottom: 2px solid #f0f0f0;
        }

        .chart-title i {
            font-size: 18px;
            color: #1abc9c;
        }

        .details-table {
            width: 100%;
            border-collapse: collapse;
            margin-top: 15px;
        }

        .details-table thead {
            background: #f0f0f0;
            border-bottom: 2px solid #ddd;
        }

        .details-table th {
            padding: 12px;
            text-align: left;
            font-weight: 600;
            color: #555;
            font-size: 12px;
            text-transform: uppercase;
        }

        .details-table td {
            padding: 12px;
            border-bottom: 1px solid #eee;
            font-size: 14px;
            color: #666;
        }

        .details-table tbody tr:hover {
            background: #f9f9f9;
        }

        .badge {
            display: inline-block;
            padding: 5px 12px;
            background: #e3f2fd;
            color: #1976d2;
            border-radius: 20px;
            font-size: 12px;
            font-weight: 600;
        }

        .badge-teal {
            background: #e0f7f6;
            color: #1abc9c;
        }

        .no-data {
            color: #999;
            font-style: italic;
            text-align: center;
            padding: 14px 10px;
        }

        .loading-spinner {
            text-align: center;
            padding: 40px;
            color: #999;
        }

        .spinner {
            border: 4px solid #f3f3f3;
            border-top: 4px solid #1abc9c;
            border-radius: 50%;
            width: 40px;
            height: 40px;
            animation: spin 1s linear infinite;
            margin: 0 auto 20px;
        }

        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }

        /* ===== RESPONSIVE ===== */
        @media (max-width: 1024px) {
            .hierarchy-grid { grid-template-columns: 1fr; }
        }

        @media (max-width: 768px) {
            .main-content { padding: 15px; }
            .analytics-container { padding: 20px; }
            .hierarchy-grid { grid-template-columns: 1fr; }
            .hierarchical-section { padding: 15px; }
        }

        .progress-bar {
            width: 100%;
            height: 6px;
            background: #ddd;
            border-radius: 3px;
            overflow: hidden;
            margin-top: 8px;
        }

        .progress-fill {
            height: 100%;
            background: #1abc9c;
            transition: width 0.3s;
        }

        /* ===== PRODUCTION TREND SECTION ===== */
        .production-trend-section {
            background: linear-gradient(135deg, rgba(102, 126, 234, 0.05) 0%, rgba(118, 75, 162, 0.05) 100%);
            padding: 30px;
            border-radius: 12px;
            margin-top: 30px;
            border: 2px solid rgba(26, 188, 156, 0.2);
            box-shadow: 0 8px 30px rgba(0, 0, 0, 0.08);
        }

        .trend-header {
            display: flex;
            align-items: center;
            gap: 15px;
            margin-bottom: 25px;
            padding-bottom: 15px;
            border-bottom: 3px solid #1abc9c;
        }

        .trend-header h2 {
            margin: 0;
            color: #2c3e50;
            font-size: 20px;
            font-weight: 700;
        }

        .trend-icon {
            font-size: 28px;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            background-clip: text;
        }

        .trend-charts-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(340px, 1fr));
            gap: 25px;
            margin-top: 20px;
        }

        .trend-chart-container {
            background: linear-gradient(165deg, #ffffff 0%, #f8fbff 55%, #f5fff9 100%);
            padding: 25px;
            border-radius: 12px;
            box-shadow: 0 10px 26px rgba(15, 23, 42, 0.1);
            border: 1px solid rgba(102, 126, 234, 0.14);
            border-left: 5px solid #667eea;
            transition: all 0.3s ease;
            position: relative;
            min-height: 280px;
            height: auto;
            display: flex;
            flex-direction: column;
        }

        .trend-chart-container:hover {
            transform: translateY(-5px);
            box-shadow: 0 8px 25px rgba(0, 0, 0, 0.12);
        }

        .trend-chart-container:nth-child(2) {
            border-left-color: #f093fb;
        }

        .trend-chart-container:nth-child(3) {
            border-left-color: #4facfe;
        }

        .trend-chart-title {
            font-size: 15px;
            font-weight: 700;
            color: #2c3e50;
            margin-bottom: 15px;
            display: flex;
            align-items: center;
            gap: 10px;
            flex-shrink: 0;
        }

        .trend-chart-container canvas {
            max-width: 100%;
            height: auto !important;
            flex-grow: 1;
        }

        .trend-chart-canvas {
            max-height: 300px;
            min-height: 220px;
        }

        .trend-chart-container.is-empty {
            display: none !important;
        }

        .all-fields-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(320px, 1fr));
            gap: 20px;
            align-items: start;
        }

        .trend-metrics {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(150px, 1fr));
            gap: 15px;
            margin-top: 25px;
        }

        .metric-card {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 18px;
            border-radius: 10px;
            text-align: center;
            box-shadow: 0 4px 15px rgba(102, 126, 234, 0.3);
            animation: slideUp 0.5s ease-out;
        }

        .metric-card.metric-ops {
            background: linear-gradient(135deg, #f093fb 0%, #f5576c 100%);
            box-shadow: 0 4px 15px rgba(240, 147, 251, 0.3);
        }

        .metric-card.metric-emp {
            background: linear-gradient(135deg, #4facfe 0%, #00f2fe 100%);
            box-shadow: 0 4px 15px rgba(79, 172, 254, 0.3);
        }

        .metric-card.metric-success {
            background: linear-gradient(135deg, #43e97b 0%, #38f9d7 100%);
            box-shadow: 0 4px 15px rgba(67, 233, 123, 0.3);
        }

        .metric-label {
            font-size: 12px;
            opacity: 0.9;
            text-transform: uppercase;
            letter-spacing: 0.5px;
            margin-bottom: 8px;
        }

        .metric-value {
            font-size: 28px;
            font-weight: 800;
            display: block;
        }

        @keyframes slideUp {
            from {
                opacity: 0;
                transform: translateY(20px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }

        /* ===== RESPONSIVE ===== */
        @media (max-width: 1200px) {
            .trend-charts-grid {
                grid-template-columns: 1fr;
            }
        }

        @media (max-width: 768px) {
            .production-trend-section {
                padding: 20px;
            }

            .trend-header h2 {
                font-size: 18px;
            }

            .trend-metrics {
                grid-template-columns: repeat(2, 1fr);
            }

            .metric-value {
                font-size: 22px;
            }
        }
    </style>
</head>

<body>

<!-- OVERLAY -->
<div id="overlay" class="overlay" onclick="toggleSidebar()"></div>

<!-- SIDEBAR -->
<script src="sidebar.js"></script>

<div class="main-content">
    <div class="analytics-container">
        
        <!-- HEADER -->
        <div class="analytics-header">
            <div style="display: flex; align-items: center;">
                <button class="menu-toggle-btn" onclick="toggleSidebar()">
                    <i class="fas fa-bars"></i>
                </button>
                <h1><i class="fas fa-chart-bar"></i> Advanced Analytics Dashboard</h1>
            </div>
        </div>

        <!-- HIERARCHICAL FIELD SELECTION -->
        <div class="hierarchical-section">
            <div class="hierarchy-title">🔗 Personalized Analytics by Machine</div>
            
            <div class="hierarchy-grid">
                <!-- STEP 1: Select Machine Field -->
                <div class="hierarchy-step active">
                    <div style="display: flex; align-items: center; gap: 10px; margin-bottom: 15px;">
                        <div class="step-indicator">1</div>
                        <span class="step-label">Select Machine Field</span>
                    </div>
                    <select id="machineFieldSelect" onchange="loadMachines()">
                        <option value="">-- Choose Machine Field --</option>
                    </select>
                </div>

                <!-- STEP 2: Select Machine -->
                <div class="hierarchy-step" id="machineSelectStep" style="display:none;">
                    <div style="display: flex; align-items: center; gap: 10px; margin-bottom: 15px;">
                        <div class="step-indicator">2</div>
                        <span class="step-label">Select Specific Machine</span>
                    </div>
                    <select id="machineValueSelect" onchange="loadMachineStats()">
                        <option value="">-- Choose Machine --</option>
                    </select>
                </div>

                <!-- STEP 3: View Machine Stats -->
                <div class="hierarchy-step" id="machineStatsStep" style="display:none;">
                    <div style="display: flex; align-items: center; gap: 10px; margin-bottom: 15px;">
                        <div class="step-indicator">3</div>
                        <span class="step-label">Machine Statistics</span>
                    </div>
                    <div id="quickStats" style="font-size: 13px; color: #666;">
                        <p style="margin: 5px 0;"><i class="fas fa-cube"></i> Total Production: <strong id="statsOps">-</strong></p>
                        <p style="margin: 5px 0;"><i class="fas fa-users"></i> Employees: <strong id="statsEmp">-</strong></p>
                        <p style="margin: 5px 0;"><i class="fas fa-calendar"></i> Days: <strong id="statsDays">-</strong></p>
                    </div>
                </div>
            </div>

            <!-- DEPENDENT FIELDS SELECTION -->
            <div id="dependentFieldsSection" style="display:none; margin-top: 25px;">
                <div class="hierarchy-title">📊 Select Fields to Analyze</div>
                <div class="dependent-fields-grid" id="dependentFieldsContainer">
                    <!-- Will be populated by JavaScript -->
                </div>
            </div>
        </div>

        <!-- MACHINE STATISTICS CARDS -->
        <div id="machineStatsCardsSection" style="display:none;">
            <div style="margin-bottom: 25px;">
                <h3 style="margin: 0 0 15px 0; color: #2c3e50; font-size: 16px; font-weight: 700;">
                    <i class="fas fa-tachometer-alt"></i> Machine: <span id="selectedMachineName">-</span>
                </h3>
            </div>

            <div class="machine-stats-grid">
                <div class="machine-stat-card">
                    <div class="stat-label">Total Production</div>
                    <div class="stat-value" id="totalOps">-</div>
                </div>
                <div class="machine-stat-card stat-card-emp">
                    <div class="stat-label">Employees</div>
                    <div class="stat-value" id="empCount">-</div>
                </div>
                <div class="machine-stat-card stat-card-ops">
                    <div class="stat-label">Working Days</div>
                    <div class="stat-value" id="workDays">-</div>
                </div>
                <div class="machine-stat-card stat-card-days">
                    <div class="stat-label">Avg/Day</div>
                    <div class="stat-value" id="avgPerDay">-</div>
                </div>
            </div>
        </div>

        <!-- FIELD ANALYSIS SECTION -->
        <div id="fieldAnalysisSection" class="field-analysis-section" style="display:none;">
            <h3 style="margin: 0 0 25px 0; color: #2c3e50; font-size: 18px; font-weight: 700; border-bottom: 2px solid #eee; padding-bottom: 15px;">
                <i class="fas fa-bar-chart"></i> Field-Level Analysis (Selected Fields)
            </h3>
            
            <div id="fieldAnalysisContainer">
                <!-- Will be populated with field analysis charts -->
            </div>
        </div>

        <!-- ALL FIELDS COMPREHENSIVE ANALYTICS -->
        <div id="allFieldsSection" class="production-trend-section" style="display:none; margin-top: 40px; border: 2px solid #e0f7f6;">
            <div class="trend-header">
                <i class="fas fa-th trend-icon" style="font-size: 32px;"></i>
                <div>
                    <h2 style="margin: 0; color: #2c3e50; font-size: 22px; font-weight: 700;">📊 All Fields Comprehensive Analysis</h2>
                    <p style="margin: 5px 0 0 0; color: #999; font-size: 13px;">Visual representation of all available database fields with interactive charts</p>
                </div>
            </div>



            <!-- All Fields Charts Container -->
            <div id="allFieldsChartsContainer" class="all-fields-grid">
                <!-- Will be populated with all fields charts -->
            </div>
        </div>

        <!-- EMPLOYEES FOR THIS MACHINE -->
        <div id="employeeSection" style="display:none; margin-top: 30px;">
            <div class="chart-container">
                <div class="chart-title">
                    <i class="fas fa-user-tie"></i> Employees Working on This Machine
                </div>
                <table class="details-table">
                    <thead>
                        <tr>
                            <th>Employee ID</th>
                            <th>Employee Name</th>
                            <th>Production Qty</th>
                            <th>Percentage</th>
                        </tr>
                    </thead>
                    <tbody id="employeeTableBody">
                        <tr><td colspan="4" class="no-data">No employee data</td></tr>
                    </tbody>
                </table>
            </div>
        </div>

        <!-- PRODUCTION TREND ANALYTICS SECTION -->
        <div id="productionTrendSection" class="production-trend-section" style="display:none;">
            <div class="trend-header">
                <i class="fas fa-chart-line trend-icon"></i>
                <div>
                    <h2>📈 Production Trend Analytics</h2>
                    <p style="margin: 5px 0 0 0; color: #999; font-size: 13px;">Date-wise production patterns and employee activity</p>
                </div>
            </div>

            <!-- Key Metrics -->
            <div class="trend-metrics" id="trendMetricsContainer">
                <!-- Will be populated by JavaScript -->
            </div>

            <!-- Trend Charts -->
            <div class="trend-charts-grid">
                <!-- Hours Trend -->
                <div class="trend-chart-container">
                    <div class="trend-chart-title">
                        <i class="fas fa-clock" style="color: #667eea;"></i> Daily Hours Trend
                    </div>
                    <div style="flex-grow: 1; position: relative;">
                        <canvas id="operationsTrendChart" class="trend-chart-canvas"></canvas>
                    </div>
                </div>

                <!-- Quantity Trend -->
                <div class="trend-chart-container">
                    <div class="trend-chart-title">
                        <i class="fas fa-box" style="color: #f093fb;"></i> Daily Quantity Trend
                    </div>
                    <div style="flex-grow: 1; position: relative;">
                        <canvas id="quantityTrendChart" class="trend-chart-canvas"></canvas>
                    </div>
                </div>

                <!-- Employees Trend -->
                <div class="trend-chart-container">
                    <div class="trend-chart-title">
                        <i class="fas fa-users" style="color: #4facfe;"></i> Active Employees Trend
                    </div>
                    <div style="flex-grow: 1; position: relative;">
                        <canvas id="employeesTrendChart" class="trend-chart-canvas"></canvas>
                    </div>
                </div>

                <!-- Combined Overview -->
                <div class="trend-chart-container">
                    <div class="trend-chart-title">
                        <i class="fas fa-chart-area" style="color: #1abc9c;"></i> Combined Production Overview
                    </div>
                    <div style="flex-grow: 1; position: relative;">
                        <canvas id="combinedTrendChart" class="trend-chart-canvas"></canvas>
                    </div>
                </div>
            </div>
        </div>

        <!-- ===== COMPREHENSIVE VISUAL DASHBOARD ===== -->
        <div id="visualDashboardSection" class="production-trend-section" style="display:none; margin-top: 40px; border: 2px solid #e0f7f6; background: linear-gradient(135deg, rgba(26, 188, 156, 0.05) 0%, rgba(22, 160, 133, 0.05) 100%);">
            <div class="trend-header">
                <i class="fas fa-chart-bar trend-icon" style="font-size: 32px;"></i>
                <div>
                    <h2 style="margin: 0; color: #2c3e50; font-size: 22px; font-weight: 700;">📊 Comprehensive Visual Analytics</h2>
                    <p style="margin: 5px 0 0 0; color: #999; font-size: 13px;">Machine, Employee, and Activity distribution with interactive charts</p>
                </div>
            </div>

            <!-- Visual Charts Grid -->
            <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(340px, 1fr)); gap: 25px; margin-top: 25px;">
                
                <!-- Machine Distribution Bar Chart -->
                <div class="trend-chart-container">
                    <div class="trend-chart-title">
                        <i class="fas fa-industry"></i> Machine Distribution (Bar Chart)
                    </div>
                    <div style="flex-grow: 1; position: relative;">
                        <canvas id="machineDistributionChart"></canvas>
                    </div>
                </div>

                <!-- Machine Distribution Horizontal Bar -->
                <div class="trend-chart-container">
                    <div class="trend-chart-title">
                        <i class="fas fa-bars"></i> Machine Comparison (Horizontal)
                    </div>
                    <div style="flex-grow: 1; position: relative;">
                        <canvas id="machineComparisonChart"></canvas>
                    </div>
                </div>

                <!-- Top Employees Bar Chart -->
                <div class="trend-chart-container">
                    <div class="trend-chart-title">
                        <i class="fas fa-users"></i> Top Employees by Activity
                    </div>
                    <div style="flex-grow: 1; position: relative;">
                        <canvas id="topEmployeesChart"></canvas>
                    </div>
                </div>

                <!-- Employee Activity Pie Chart -->
                <div class="trend-chart-container">
                    <div class="trend-chart-title">
                        <i class="fas fa-pie-chart"></i> Employee Activity Distribution
                    </div>
                    <div style="flex-grow: 1; position: relative;">
                        <canvas id="employeeActivityPieChart"></canvas>
                    </div>
                </div>

                <!-- Machine Assignment Matrix -->
                <div class="trend-chart-container" style="grid-column: span 2;">
                    <div class="trend-chart-title">
                        <i class="fas fa-network-wired"></i> Employee-Machine Assignment Network
                    </div>
                    <div style="flex-grow: 1; position: relative;">
                        <canvas id="machineAssignmentChart"></canvas>
                    </div>
                </div>

                <!-- Bubble Chart - Production vs Employees -->
                <div class="trend-chart-container">
                    <div class="trend-chart-title">
                        <i class="fas fa-circle"></i> Production Efficiency Bubble
                    </div>
                    <div style="flex-grow: 1; position: relative;">
                        <canvas id="bubbleEfficiencyChart"></canvas>
                    </div>
                </div>

                <!-- Radar Chart - Multi-Metric Comparison -->
                <div class="trend-chart-container">
                    <div class="trend-chart-title">
                        <i class="fas fa-spider"></i> Multi-Metric Performance Radar
                    </div>
                    <div style="flex-grow: 1; position: relative;">
                        <canvas id="radarMetricsChart"></canvas>
                    </div>
                </div>

            </div>

            <!-- Summary Statistics -->
            <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(150px, 1fr)); gap: 15px; margin-top: 30px;">
                <div class="metric-card" style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); padding: 20px; border-radius: 10px; color: white; text-align: center;">
                    <div style="font-size: 12px; opacity: 0.9; text-transform: uppercase; margin-bottom: 8px;">Total Machines</div>
                    <div style="font-size: 32px; font-weight: 800;" id="totalMachinesMetric">0</div>
                </div>
                <div class="metric-card" style="background: linear-gradient(135deg, #f093fb 0%, #f5576c 100%); padding: 20px; border-radius: 10px; color: white; text-align: center;">
                    <div style="font-size: 12px; opacity: 0.9; text-transform: uppercase; margin-bottom: 8px;">Total Employees</div>
                    <div style="font-size: 32px; font-weight: 800;" id="totalEmployeesMetric">0</div>
                </div>
                <div class="metric-card" style="background: linear-gradient(135deg, #4facfe 0%, #00f2fe 100%); padding: 20px; border-radius: 10px; color: white; text-align: center;">
                    <div style="font-size: 12px; opacity: 0.9; text-transform: uppercase; margin-bottom: 8px;">Avg Production</div>
                    <div style="font-size: 32px; font-weight: 800;" id="avgOperationsMetric">0</div>
                </div>
                <div class="metric-card" style="background: linear-gradient(135deg, #43e97b 0%, #38f9d7 100%); padding: 20px; border-radius: 10px; color: white; text-align: center;">
                    <div style="font-size: 12px; opacity: 0.9; text-transform: uppercase; margin-bottom: 8px;">Efficiency %</div>
                    <div style="font-size: 32px; font-weight: 800;" id="efficiencyMetric">0%</div>
                </div>
            </div>
        </div>

    </div>
</div>

<script>
    // ===== CHART INSTANCES =====
    let fieldCharts = {};
    let trendCharts = {};

    // ===== COLOR PALETTES ===== 
    const colorPalettes = {
        operations: {
            line: 'rgba(102, 126, 234, 1)',
            fill: 'rgba(102, 126, 234, 0.1)',
            point: 'rgba(102, 126, 234, 1)'
        },
        quantity: {
            line: 'rgba(240, 147, 251, 1)',
            fill: 'rgba(240, 147, 251, 0.1)',
            point: 'rgba(240, 147, 251, 1)'
        },
        employees: {
            line: 'rgba(79, 172, 254, 1)',
            fill: 'rgba(79, 172, 254, 0.1)',
            point: 'rgba(79, 172, 254, 1)'
        },
        success: {
            line: 'rgba(67, 233, 123, 1)',
            fill: 'rgba(67, 233, 123, 0.1)',
            point: 'rgba(67, 233, 123, 1)'
        }
    };

    // ===== INTERACTIVE BAR THEME =====
    const barColorPairs = [
        ['#22d3ee', '#0ea5e9'],
        ['#34d399', '#22c55e'],
        ['#60a5fa', '#3b82f6'],
        ['#f472b6', '#ec4899'],
        ['#f59e0b', '#ef4444'],
        ['#a78bfa', '#7c3aed'],
        ['#fb7185', '#e11d48'],
        ['#2dd4bf', '#14b8a6']
    ];

    function hexToRgba(hex, alpha = 1) {
        if (typeof hex !== 'string' || !hex.startsWith('#')) return `rgba(59,130,246,${alpha})`;
        const sanitized = hex.replace('#', '');
        const full = sanitized.length === 3
            ? sanitized.split('').map(ch => ch + ch).join('')
            : sanitized;
        const r = parseInt(full.substring(0, 2), 16);
        const g = parseInt(full.substring(2, 4), 16);
        const b = parseInt(full.substring(4, 6), 16);
        return `rgba(${r},${g},${b},${alpha})`;
    }

    function getBarColorPair(datasetIndex, valueIndex) {
        return barColorPairs[(datasetIndex + valueIndex) % barColorPairs.length];
    }

    const interactiveBarThemePlugin = {
        id: 'interactiveBarThemeAdvanced',
        beforeUpdate(chart) {
            if (chart.config.type !== 'bar') return;
            const datasets = chart.data?.datasets || [];

            datasets.forEach((dataset, datasetIndex) => {
                const values = Array.isArray(dataset.data) ? dataset.data : [];
                const backgrounds = [];
                const borders = [];
                const hoverBackgrounds = [];

                values.forEach((_, valueIndex) => {
                    const [base, accent] = getBarColorPair(datasetIndex, valueIndex);
                    backgrounds.push(hexToRgba(base, 0.8));
                    borders.push(accent);
                    hoverBackgrounds.push(hexToRgba(accent, 0.95));
                });

                dataset.backgroundColor = backgrounds;
                dataset.borderColor = borders;
                dataset.hoverBackgroundColor = hoverBackgrounds;
                dataset.borderWidth = 1.5;
                dataset.borderRadius = 10;
                dataset.borderSkipped = false;
                dataset.hoverBorderWidth = 2;
                dataset.maxBarThickness = dataset.maxBarThickness || 34;
                dataset.barPercentage = dataset.barPercentage ?? 0.72;
                dataset.categoryPercentage = dataset.categoryPercentage ?? 0.66;
            });
        },
        beforeDatasetDraw(chart, args) {
            const meta = chart.getDatasetMeta(args.index);
            if (meta.type !== 'bar') return;
            const ds = chart.data.datasets[args.index] || {};
            const color = Array.isArray(ds.borderColor) ? ds.borderColor[0] : ds.borderColor;
            const glow = typeof color === 'string' ? hexToRgba(color, 0.45) : 'rgba(14,165,233,0.45)';

            chart.ctx.save();
            chart.ctx.shadowColor = glow;
            chart.ctx.shadowBlur = 12;
            chart.ctx.shadowOffsetX = 0;
            chart.ctx.shadowOffsetY = 2;
        },
        afterDatasetDraw(chart, args) {
            const meta = chart.getDatasetMeta(args.index);
            if (meta.type !== 'bar') return;
            chart.ctx.restore();
        }
    };

    Chart.register(interactiveBarThemePlugin);
    Chart.defaults.animation.duration = 950;
    Chart.defaults.animation.easing = 'easeOutQuart';
    Chart.defaults.plugins.legend.labels.usePointStyle = true;
    Chart.defaults.plugins.legend.labels.boxWidth = 10;
    Chart.defaults.plugins.legend.labels.padding = 14;

    function setCardVisibilityByCanvasId(canvasId, isVisible) {
        const canvas = document.getElementById(canvasId);
        if (!canvas) return;
        const card = canvas.closest('.trend-chart-container');
        if (!card) return;
        card.classList.toggle('is-empty', !isVisible);
        card.style.display = isVisible ? 'flex' : 'none';
    }

    function resetAnalyticsSections() {
        const sections = [
            'machineStatsCardsSection',
            'fieldAnalysisSection',
            'employeeSection',
            'productionTrendSection',
            'visualDashboardSection',
            'allFieldsSection'
        ];
        sections.forEach(id => {
            const el = document.getElementById(id);
            if (el) el.style.display = 'none';
        });

        const fieldContainer = document.getElementById('fieldAnalysisContainer');
        if (fieldContainer) fieldContainer.innerHTML = '';
    }

    // ===== TOGGLE SIDEBAR =====
    // Sidebar toggle is handled by sidebar.js
    // toggleSidebar() function is provided by sidebar.js

    // ===== LOAD MACHINE FIELDS =====
    async function loadMachineFields() {
        try {
            const response = await fetch('api.php?action=get_all_fields');
            const result = await response.json();
            
            if (result.success) {
                const select = document.getElementById('machineFieldSelect');
                select.innerHTML = '<option value="">-- Choose Machine Field --</option>';
                
                result.parent_fields.forEach(field => {
                    const option = document.createElement('option');
                    option.value = field.id;
                    option.setAttribute('data-field-name', field.field_name);
                    option.textContent = field.field_label;
                    select.appendChild(option);
                });
            }
        } catch (error) {
            console.error('Error loading fields:', error);
        }
    }

    // ===== LOAD MACHINES FOR SELECTED FIELD =====
    async function loadMachines() {
        const fieldSelect = document.getElementById('machineFieldSelect');
        const fieldId = fieldSelect.value;
        const fieldName = fieldSelect.options[fieldSelect.selectedIndex].getAttribute('data-field-name');
        
        if (!fieldId || !fieldName) {
            document.getElementById('machineSelectStep').style.display = 'none';
            document.getElementById('dependentFieldsSection').style.display = 'none';
            resetAnalyticsSections();
            return;
        }

        try {
            // Load dependent fields
            const depResponse = await fetch(`api.php?action=get_dependent_fields&parent_field_id=${fieldId}`);
            const depResult = await depResponse.json();
            
            if (depResult.success) {
                buildDependentFieldSelectors(depResult.dependent_fields, fieldName);
            }

            // Load machines for this field
            const response = await fetch(`api.php?action=get_machine_summary&machine_field=${fieldName}`);
            const result = await response.json();
            
            if (result.success) {
                const select = document.getElementById('machineValueSelect');
                select.innerHTML = '<option value="">-- Choose Machine --</option>';
                
                result.machines.forEach(machine => {
                    const option = document.createElement('option');
                    option.value = machine.machine_name;
                    const qty = Number(machine.total_production_qty ?? machine.total_operations ?? 0).toLocaleString(undefined, { maximumFractionDigits: 2 });
                    const hrs = Number(machine.total_hours ?? 0).toLocaleString(undefined, { maximumFractionDigits: 2 });
                    option.textContent = `${machine.machine_name} (${qty} qty, ${hrs} h)`;
                    select.appendChild(option);
                });

                document.getElementById('machineSelectStep').style.display = 'block';
                document.getElementById('dependentFieldsSection').style.display = 'block';
            }
        } catch (error) {
            console.error('Error loading machines:', error);
        }
    }

    // ===== BUILD DEPENDENT FIELD SELECTORS =====
    function buildDependentFieldSelectors(fields, machineFieldName) {
        const container = document.getElementById('dependentFieldsContainer');
        
        if (!fields || fields.length === 0) {
            // Show trend line graph for selected machine instead of empty message
            const machineValue = document.getElementById('machineValueSelect').value;
            displayMachineTrendGraph(machineValue, machineFieldName, container);
            return;
        }

        container.innerHTML = fields.map(field => `
            <div class="dependent-field-selector" data-field-name="${field.field_name}" onclick="toggleFieldSelection(this, '${field.field_name}', '${field.field_label}', '${machineFieldName}')">
                <div style="display: flex; align-items: center; gap: 10px;">
                    <input type="checkbox" class="field-selector-checkbox" onclick="event.stopPropagation()">
                    <div style="flex: 1;">
                        <div class="field-selector-label">${field.field_label}</div>
                        <div style="font-size: 12px; color: #999;">${field.field_type}</div>
                    </div>
                </div>
            </div>
        `).join('');
    }

    // ===== DISPLAY MACHINE TREND GRAPH =====
    async function displayMachineTrendGraph(machineValue, machineFieldName, container) {
        try {
            const response = await fetch(`api.php?action=get_production_trend_by_date&machine_field=${machineFieldName}&machine_value=${machineValue}`);
            const result = await response.json();
            
            if (result.success && result.data && result.data.length > 0) {
                const chartHtml = `
                    <div style="width: 100%; margin-top: 20px;">
                        <div style="background: #f9f9f9; padding: 20px; border-radius: 8px;">
                            <h3 style="margin-top: 0; color: #2c3e50; font-size: 16px;">📊 Production Trend for ${machineValue}</h3>
                            <canvas id="machineNoFieldsTrendChart"></canvas>
                        </div>
                    </div>
                `;
                
                container.innerHTML = chartHtml;
                
                // Create line chart
                const dates = result.data.map(item => item.date);
                const productionQty = result.data.map(item => Number(item.total_quantity ?? item.total_operations ?? 0));
                
                const ctx = document.getElementById('machineNoFieldsTrendChart').getContext('2d');
                
                new Chart(ctx, {
                    type: 'line',
                    data: {
                        labels: dates,
                        datasets: [{
                            label: `Production Qty for ${machineValue}`,
                            data: productionQty,
                            borderColor: '#1abc9c',
                            backgroundColor: 'rgba(26, 188, 156, 0.1)',
                            borderWidth: 3,
                            fill: true,
                            tension: 0.4,
                            pointRadius: 5,
                            pointBackgroundColor: '#1abc9c',
                            pointBorderColor: '#fff',
                            pointBorderWidth: 2,
                            pointHoverRadius: 7
                        }]
                    },
                    options: {
                        responsive: true,
                        maintainAspectRatio: true,
                        plugins: {
                            legend: {
                                display: true,
                                labels: {
                                    usePointStyle: true,
                                    padding: 15,
                                    font: { size: 13, weight: 500 }
                                }
                            },
                            tooltip: {
                                backgroundColor: 'rgba(0,0,0,0.8)',
                                padding: 12,
                                titleFont: { size: 14, weight: 'bold' },
                                bodyFont: { size: 13 },
                                cornerRadius: 6
                            }
                        },
                        scales: {
                            y: {
                                beginAtZero: true,
                                title: { display: true, text: 'Production Qty' },
                                ticks: { font: { size: 12 } }
                            },
                            x: {
                                ticks: { font: { size: 12 } }
                            }
                        }
                    }
                });
            } else {
                container.innerHTML = '<p class="no-data">No trend data available for this machine</p>';
            }
        } catch (error) {
            console.error('Error loading trend graph:', error);
            container.innerHTML = '<p class="no-data">Error loading trend data</p>';
        }
    }

    // ===== TOGGLE FIELD SELECTION =====
    function toggleFieldSelection(element, fieldName, fieldLabel, machineFieldName) {
        const checkbox = element.querySelector('.field-selector-checkbox');
        checkbox.checked = !checkbox.checked;
        element.classList.toggle('selected', checkbox.checked);
        
        if (checkbox.checked) {
            const machineValue = document.getElementById('machineValueSelect').value;
            if (machineValue) {
                loadFieldAnalysis(fieldName, fieldLabel, machineFieldName, machineValue);
            }
        } else {
            // Remove chart if it exists
            if (fieldCharts[fieldName]) {
                fieldCharts[fieldName].destroy();
                delete fieldCharts[fieldName];
            }
            // Remove section
            const section = document.getElementById(`field-section-${fieldName}`);
            if (section) section.remove();
        }
    }

    // ===== LOAD MACHINE STATISTICS =====
    async function loadMachineStats() {
        const machineValue = document.getElementById('machineValueSelect').value;
        const fieldId = document.getElementById('machineFieldSelect').value;
        const machineFieldName = document.getElementById('machineFieldSelect').options[document.getElementById('machineFieldSelect').selectedIndex].getAttribute('data-field-name');
        
        if (!machineValue || !machineFieldName) {
            resetAnalyticsSections();
            return;
        }

        try {
            const response = await fetch(`api.php?action=get_machine_summary&machine_field=${machineFieldName}`);
            const result = await response.json();
            
            if (result.success) {
                const machine = result.machines.find(m => m.machine_name === machineValue);
                
                if (machine) {
                    const totalProduction = Number(machine.total_production_qty ?? machine.total_operations ?? 0);
                    const totalHours = Number(machine.total_hours ?? 0);
                    document.getElementById('selectedMachineName').textContent = machineValue;
                    document.getElementById('totalOps').textContent = totalProduction.toLocaleString(undefined, { maximumFractionDigits: 2 });
                    document.getElementById('empCount').textContent = machine.employee_count;
                    document.getElementById('workDays').textContent = machine.working_days;
                    document.getElementById('statsOps').textContent = totalProduction.toLocaleString(undefined, { maximumFractionDigits: 2 });
                    document.getElementById('statsEmp').textContent = machine.employee_count;
                    document.getElementById('statsDays').textContent = machine.working_days;
                    
                    const avgPerDay = (totalProduction / (machine.working_days || 1)).toFixed(1);
                    document.getElementById('avgPerDay').textContent = avgPerDay;
                    
                    document.getElementById('machineStatsCardsSection').style.display = 'block';
                    document.getElementById('fieldAnalysisSection').style.display = 'block';
                    document.getElementById('employeeSection').style.display = 'none';
                    
                    // Load employees and production trends
                    loadEmployeesForMachine(machineFieldName, machineValue);
                    loadProductionTrend(machineFieldName, machineValue);
                    createVisualDashboard(machineFieldName, machineValue);
                    loadAllFieldsAnalytics(machineFieldName, machineValue);
                    
                    // Load selected field analyses
                    loadSelectedFieldAnalyses(machineFieldName, machineValue);
                } else {
                    resetAnalyticsSections();
                }
            }
        } catch (error) {
            console.error('Error loading machine stats:', error);
            resetAnalyticsSections();
        }
    }

    // ===== LOAD FIELD ANALYSIS =====
    async function loadFieldAnalysis(fieldName, fieldLabel, machineFieldName, machineValue) {
        try {
            const response = await fetch(`api.php?action=get_machine_specific_stats&machine_field=${machineFieldName}&machine_value=${machineValue}&dependent_field=${fieldName}`);
            const result = await response.json();
            
            if (result.success) {
                const totalMetric = Number(result.total_production_qty ?? result.total_records ?? 0);
                buildFieldAnalysisCard(fieldName, fieldLabel, result.data, totalMetric);
            }
        } catch (error) {
            console.error('Error loading field analysis:', error);
        }
    }

    // ===== BUILD FIELD ANALYSIS CARD =====
    function buildFieldAnalysisCard(fieldName, fieldLabel, data, totalRecords) {
        const container = document.getElementById('fieldAnalysisContainer');
        
        let sectionId = `field-section-${fieldName}`;
        let section = document.getElementById(sectionId);
        
        if (!section) {
            section = document.createElement('div');
            section.id = sectionId;
            section.style.marginBottom = '25px';
            container.appendChild(section);
        }

        const html = `
            <!-- BAR CHART -->
            <div class="chart-container">
                <div class="chart-title">
                    <i class="fas fa-bar-chart"></i> ${fieldLabel} Distribution (Bar Chart)
                </div>
                <canvas id="chart-bar-${fieldName}" style="max-height: 300px;"></canvas>
            </div>
            
            <!-- LINE CHART TREND -->
            <div class="chart-container">
                <div class="chart-title">
                    <i class="fas fa-chart-line"></i> ${fieldLabel} Trend (Line Chart)
                </div>
                <canvas id="chart-line-${fieldName}" style="max-height: 300px;"></canvas>
            </div>

            <!-- PIE CHART -->
            <div class="chart-container">
                <div class="chart-title">
                    <i class="fas fa-pie-chart"></i> ${fieldLabel} Composition (Pie Chart)
                </div>
                <canvas id="chart-pie-${fieldName}" style="max-height: 300px;"></canvas>
            </div>
            
            <!-- DATA TABLE -->
            <div class="chart-container" style="grid-column: span 3;">
                <div class="chart-title">
                    <i class="fas fa-list"></i> ${fieldLabel} Detailed Breakdown
                </div>
                <table class="details-table">
                    <thead>
                        <tr>
                            <th>Rank</th>
                            <th>Value</th>
                            <th>Total Production</th>
                            <th>Percentage</th>
                            <th>Progress</th>
                        </tr>
                    </thead>
                    <tbody>
                        ${data.map((item, index) => {
                            const metricValue = Number(item.total_production_qty ?? item.count ?? 0);
                            const percentage = ((metricValue / (totalRecords || 1)) * 100).toFixed(1);
                            return `
                                <tr>
                                    <td><strong>#${index + 1}</strong></td>
                                    <td><strong>${item.value || 'N/A'}</strong></td>
                                    <td>${metricValue.toLocaleString(undefined, { maximumFractionDigits: 2 })}</td>
                                    <td>${percentage}%</td>
                                    <td>
                                        <div style="display: flex; align-items: center; gap: 8px;">
                                            <div style="width: 80px; height: 8px; background: #eee; border-radius: 4px; overflow: hidden;">
                                                <div style="width: ${percentage}%; height: 100%; background: linear-gradient(90deg, #1abc9c, #16a085); transition: width 0.3s;"></div>
                                            </div>
                                            <span style="font-weight: 600; color: #1abc9c;">${percentage}%</span>
                                        </div>
                                    </td>
                                </tr>
                            `;
                        }).join('')}
                    </tbody>
                </table>
            </div>
        `;
        
        section.innerHTML = html;
        
        // Build all three chart types
        buildFieldCharts(fieldName, data, fieldLabel);
    }

    // ===== BUILD MULTIPLE FIELD CHARTS (BAR, LINE, PIE) =====
    function buildFieldCharts(fieldName, data, fieldLabel) {
        const chartData = data.slice(0, 15);
        const labels = chartData.map(d => d.value || 'N/A');
        const values = chartData.map(d => Number(d.total_production_qty ?? d.count ?? 0));
        
        // Destroy existing charts
        if (fieldCharts[`bar-${fieldName}`]) fieldCharts[`bar-${fieldName}`].destroy();
        if (fieldCharts[`line-${fieldName}`]) fieldCharts[`line-${fieldName}`].destroy();
        if (fieldCharts[`pie-${fieldName}`]) fieldCharts[`pie-${fieldName}`].destroy();
        
        const colors = [
            '#1abc9c', '#16a085', '#138d75', '#0e6251', '#117a65',
            '#0d5345', '#0a4630', '#07361b', '#051810', '#1e8449',
            '#196f3d', '#145a32', '#0b3d2c', '#082e25', '#05251d'
        ];
        
        // 1. BAR CHART
        const barCtx = document.getElementById(`chart-bar-${fieldName}`);
        if (barCtx) {
            fieldCharts[`bar-${fieldName}`] = new Chart(barCtx, {
                type: 'bar',
                data: {
                    labels: labels,
                    datasets: [{
                        label: fieldLabel,
                        data: values,
                        backgroundColor: colors.slice(0, labels.length),
                        borderColor: colors.slice(0, labels.length),
                        borderWidth: 2,
                        borderRadius: 6,
                        hoverBackgroundColor: '#117a65'
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    indexAxis: labels.length > 8 ? 'y' : 'x',
                    plugins: {
                        legend: { display: true, position: 'top' },
                        tooltip: {
                            backgroundColor: 'rgba(0,0,0,0.8)',
                            padding: 12,
                            titleFont: { size: 14 },
                            bodyFont: { size: 13 }
                        }
                    },
                    scales: {
                        y: { beginAtZero: true, ticks: { font: { size: 11 } } },
                        x: { ticks: { font: { size: 11 } } }
                    }
                }
            });
        }
        
        // 2. LINE CHART (TREND)
        const lineCtx = document.getElementById(`chart-line-${fieldName}`);
        if (lineCtx) {
            fieldCharts[`line-${fieldName}`] = new Chart(lineCtx, {
                type: 'line',
                data: {
                    labels: labels,
                    datasets: [{
                        label: `${fieldLabel} Trend`,
                        data: values,
                        borderColor: '#667eea',
                        backgroundColor: 'rgba(102, 126, 234, 0.1)',
                        borderWidth: 3,
                        fill: true,
                        tension: 0.4,
                        pointRadius: 5,
                        pointBackgroundColor: '#667eea',
                        pointBorderColor: '#fff',
                        pointBorderWidth: 2,
                        pointHoverRadius: 7,
                        hoverBackgroundColor: '#764ba2'
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: { display: true, position: 'top' },
                        tooltip: {
                            backgroundColor: 'rgba(0,0,0,0.8)',
                            padding: 12,
                            titleFont: { size: 14 },
                            bodyFont: { size: 13 }
                        }
                    },
                    scales: {
                        y: { beginAtZero: true, ticks: { font: { size: 11 } } },
                        x: { ticks: { font: { size: 11 } } }
                    }
                }
            });
        }
        
        // 3. PIE CHART (COMPOSITION)
        const pieCtx = document.getElementById(`chart-pie-${fieldName}`);
        if (pieCtx) {
            fieldCharts[`pie-${fieldName}`] = new Chart(pieCtx, {
                type: 'doughnut',
                data: {
                    labels: labels,
                    datasets: [{
                        data: values,
                        backgroundColor: colors.slice(0, labels.length),
                        borderColor: '#fff',
                        borderWidth: 2,
                        hoverOffset: 10
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: {
                            display: true,
                            position: 'right',
                            labels: { font: { size: 11 }, padding: 15 }
                        },
                        tooltip: {
                            backgroundColor: 'rgba(0,0,0,0.8)',
                            padding: 12,
                            titleFont: { size: 14 },
                            bodyFont: { size: 13 },
                            callbacks: {
                                label: function(context) {
                                    const total = context.dataset.data.reduce((a, b) => a + b, 0);
                                    const percentage = ((context.parsed / total) * 100).toFixed(1);
                                    return `${context.label}: ${context.parsed} (${percentage}%)`;
                                }
                            }
                        }
                    }
                }
            });
        }
    }

    // ===== LOAD SELECTED FIELD ANALYSES =====
    function loadSelectedFieldAnalyses(machineFieldName, machineValue) {
        const checkboxes = document.querySelectorAll('.field-selector-checkbox:checked');
        const machineField = document.getElementById('machineFieldSelect');
        
        checkboxes.forEach(checkbox => {
            const parent = checkbox.closest('.dependent-field-selector');
            if (parent) {
                const fieldLabel = parent.querySelector('.field-selector-label').textContent;
                // Try to find field name from the selector
                const fieldName = parent.getAttribute('data-field-name') || 
                                 fieldLabel.toLowerCase().replace(/\s+/g, '_');
                
                // Get the actual field name
                const fieldOptions = document.querySelectorAll('.dependent-field-selector');
                fieldOptions.forEach(opt => {
                    if (opt.querySelector('.field-selector-label').textContent === fieldLabel) {
                        const actualFieldName = opt.getAttribute('data-field-name');
                        if (actualFieldName) {
                            loadFieldAnalysis(actualFieldName, fieldLabel, machineFieldName, machineValue);
                        }
                    }
                });
            }
        });
    }

    // ===== LOAD EMPLOYEES FOR MACHINE =====
    async function loadEmployeesForMachine(machineFieldName, machineValue) {
        try {
            const response = await fetch(`api.php?action=get_machine_specific_stats&machine_field=${machineFieldName}&machine_value=${machineValue}&dependent_field=emp_id`);
            const result = await response.json();

            const section = document.getElementById('employeeSection');
            const tbody = document.getElementById('employeeTableBody');
            const employees = Array.isArray(result.employees_working) ? result.employees_working : [];
            const usableEmployees = employees.filter(emp => Number(emp.total_production_qty ?? emp.count ?? 0) > 0);

            if (result.success && usableEmployees.length > 0) {
                const totalProduction = usableEmployees.reduce((sum, emp) => sum + Number(emp.total_production_qty ?? emp.count ?? 0), 0);

                tbody.innerHTML = usableEmployees.map(emp => {
                    const metric = Number(emp.total_production_qty ?? emp.count ?? 0);
                    const percentage = ((metric / (totalProduction || 1)) * 100).toFixed(1);
                    return `
                        <tr>
                            <td><span class="badge">${emp.emp_id}</span></td>
                            <td><strong>${emp.operator_name}</strong></td>
                            <td>${metric.toLocaleString(undefined, { maximumFractionDigits: 2 })}</td>
                            <td>${percentage}%</td>
                        </tr>
                    `;
                }).join('');
                if (section) section.style.display = 'block';
            } else {
                if (tbody) {
                    tbody.innerHTML = '<tr><td colspan="4" class="no-data">No employee data</td></tr>';
                }
                if (section) section.style.display = 'none';
            }
        } catch (error) {
            console.error('Error loading employees:', error);
            const section = document.getElementById('employeeSection');
            if (section) section.style.display = 'none';
        }
    }

    // ===== LOAD PRODUCTION TREND =====
    async function loadProductionTrend(machineFieldName, machineValue) {
        try {
            const response = await fetch(`api.php?action=get_production_trend_by_date&machine_field=${machineFieldName}&machine_value=${machineValue}`);
            const result = await response.json();
            
            const hasTrendData = result.success && Array.isArray(result.dates) && result.dates.length > 0;
            if (hasTrendData) {
                displayTrendMetrics(result);
                drawTrendCharts(result);
                document.getElementById('productionTrendSection').style.display = 'block';
            } else {
                document.getElementById('productionTrendSection').style.display = 'none';
            }
        } catch (error) {
            console.error('Error loading production trend:', error);
            document.getElementById('productionTrendSection').style.display = 'none';
        }
    }

    // ===== DISPLAY TREND METRICS =====
    function displayTrendMetrics(data) {
        const dates = Array.isArray(data.dates) ? data.dates : [];
        const quantities = Array.isArray(data.total_quantity) ? data.total_quantity : [];
        const hours = Array.isArray(data.total_hours) ? data.total_hours : [];
        const employees = Array.isArray(data.employees_count) ? data.employees_count : [];

        const totalQty = quantities.reduce((a, b) => a + Number(b || 0), 0);
        const totalHours = hours.reduce((a, b) => a + Number(b || 0), 0);
        const avgQty = (totalQty / (dates.length || 1)).toFixed(1);
        const avgEmp = (employees.reduce((a, b) => a + Number(b || 0), 0) / (dates.length || 1)).toFixed(1);
        
        const container = document.getElementById('trendMetricsContainer');
        container.innerHTML = `
            <div class="metric-card">
                <div class="metric-label">Total Production</div>
                <span class="metric-value">${totalQty.toFixed(2)}</span>
            </div>
            <div class="metric-card metric-ops">
                <div class="metric-label">Total Hours</div>
                <span class="metric-value">${totalHours.toFixed(2)}</span>
            </div>
            <div class="metric-card metric-emp">
                <div class="metric-label">Avg Prod/Day</div>
                <span class="metric-value">${avgQty}</span>
            </div>
            <div class="metric-card metric-success">
                <div class="metric-label">Avg Employees</div>
                <span class="metric-value">${avgEmp}</span>
            </div>
        `;
    }

    // ===== DRAW TREND CHARTS =====
    function drawTrendCharts(data) {
        // Destroy existing charts
        Object.values(trendCharts).forEach(chart => chart?.destroy?.());
        trendCharts = {};

        const commonOptions = {
            responsive: true,
            maintainAspectRatio: false,
            interaction: { mode: 'index', intersect: false },
            plugins: {
                legend: { display: true, position: 'top' },
                tooltip: {
                    backgroundColor: 'rgba(0,0,0,0.8)',
                    padding: 12,
                    displayColors: true
                }
            }
        };

        const dates = Array.isArray(data.dates) ? data.dates : [];
        const hoursSeries = Array.isArray(data.total_hours) ? data.total_hours.map(Number) : [];
        const qtySeries = Array.isArray(data.total_quantity) ? data.total_quantity.map(Number) : [];
        const empSeries = Array.isArray(data.employees_count) ? data.employees_count.map(Number) : [];

        const hasHours = hoursSeries.some(v => Number(v) > 0);
        const hasQty = qtySeries.some(v => Number(v) > 0);
        const hasEmp = empSeries.some(v => Number(v) > 0);
        const hasCombined = hasHours || hasQty || hasEmp;

        setCardVisibilityByCanvasId('operationsTrendChart', hasHours);
        setCardVisibilityByCanvasId('quantityTrendChart', hasQty);
        setCardVisibilityByCanvasId('employeesTrendChart', hasEmp);
        setCardVisibilityByCanvasId('combinedTrendChart', hasCombined);

        // 1. Hours Trend
        if (hasHours) {
        const operationsCtx = document.getElementById('operationsTrendChart').getContext('2d');
        trendCharts.operations = new Chart(operationsCtx, {
            type: 'line',
            data: {
                labels: dates,
                datasets: [{
                    label: 'Daily Hours',
                    data: hoursSeries,
                    borderColor: colorPalettes.operations.line,
                    backgroundColor: colorPalettes.operations.fill,
                    borderWidth: 3,
                    fill: true,
                    pointRadius: 5,
                    pointBackgroundColor: colorPalettes.operations.point,
                    pointBorderColor: '#fff',
                    pointBorderWidth: 2,
                    pointHoverRadius: 7,
                    tension: 0.4
                }]
            },
            options: {
                ...commonOptions,
                scales: { y: { beginAtZero: true, title: { display: true, text: 'Hours' } } }
            }
        });
        }

        // 2. Quantity Trend
        if (hasQty) {
        const quantityCtx = document.getElementById('quantityTrendChart').getContext('2d');
        trendCharts.quantity = new Chart(quantityCtx, {
            type: 'line',
            data: {
                labels: dates,
                datasets: [{
                    label: 'Daily Quantity',
                    data: qtySeries,
                    borderColor: colorPalettes.quantity.line,
                    backgroundColor: colorPalettes.quantity.fill,
                    borderWidth: 3,
                    fill: true,
                    pointRadius: 5,
                    pointBackgroundColor: colorPalettes.quantity.point,
                    pointBorderColor: '#fff',
                    pointBorderWidth: 2,
                    pointHoverRadius: 7,
                    tension: 0.4
                }]
            },
            options: {
                ...commonOptions,
                scales: { y: { beginAtZero: true, title: { display: true, text: 'Quantity (Units)' } } }
            }
        });
        }

        // 3. Employees Trend
        if (hasEmp) {
        const employeesCtx = document.getElementById('employeesTrendChart').getContext('2d');
        trendCharts.employees = new Chart(employeesCtx, {
            type: 'line',
            data: {
                labels: dates,
                datasets: [{
                    label: 'Active Employees',
                    data: empSeries,
                    borderColor: colorPalettes.employees.line,
                    backgroundColor: colorPalettes.employees.fill,
                    borderWidth: 3,
                    fill: true,
                    pointRadius: 5,
                    pointBackgroundColor: colorPalettes.employees.point,
                    pointBorderColor: '#fff',
                    pointBorderWidth: 2,
                    pointHoverRadius: 7,
                    tension: 0.4,
                    borderDash: [5, 5]
                }]
            },
            options: {
                ...commonOptions,
                scales: { y: { beginAtZero: true, title: { display: true, text: 'Employee Count' } } }
            }
        });
        }

        // 4. Combined Overview (Multi-axis)
        if (hasCombined) {
        const combinedCtx = document.getElementById('combinedTrendChart').getContext('2d');
        trendCharts.combined = new Chart(combinedCtx, {
            type: 'line',
            data: {
                labels: dates,
                datasets: [
                    {
                        label: 'Hours',
                        data: hoursSeries,
                        borderColor: colorPalettes.operations.line,
                        backgroundColor: colorPalettes.operations.fill,
                        borderWidth: 2,
                        fill: false,
                        pointRadius: 4,
                        pointBackgroundColor: colorPalettes.operations.point,
                        tension: 0.4,
                        yAxisID: 'y'
                    },
                    {
                        label: 'Quantity',
                        data: qtySeries,
                        borderColor: colorPalettes.quantity.line,
                        backgroundColor: colorPalettes.quantity.fill,
                        borderWidth: 2,
                        fill: false,
                        pointRadius: 4,
                        pointBackgroundColor: colorPalettes.quantity.point,
                        tension: 0.4,
                        yAxisID: 'y1'
                    },
                    {
                        label: 'Employees',
                        data: empSeries,
                        borderColor: colorPalettes.employees.line,
                        backgroundColor: colorPalettes.employees.fill,
                        borderWidth: 2,
                        fill: false,
                        pointRadius: 4,
                        pointBackgroundColor: colorPalettes.employees.point,
                        tension: 0.4,
                        borderDash: [5, 5]
                    }
                ]
            },
            options: {
                ...commonOptions,
                scales: {
                    y: {
                        type: 'linear',
                        display: true,
                        position: 'left',
                        title: { display: true, text: 'Hours / Employees' }
                    },
                    y1: {
                        type: 'linear',
                        display: true,
                        position: 'right',
                        title: { display: true, text: 'Quantity' },
                        grid: { drawOnChartArea: false }
                    }
                }
            }
        });
        }
    }

    // ===== CREATE VISUAL DASHBOARD WITH ALL CHARTS =====
    async function createVisualDashboard(machineFieldName, machineValue) {
        try {
            // Get machine stats for dashboard
            const statsResponse = await fetch(`api.php?action=get_machine_summary&machine_field=${machineFieldName}`);
            const statsResult = await statsResponse.json();

            // Get employee data
            const empResponse = await fetch(`api.php?action=get_machine_specific_stats&machine_field=${machineFieldName}&machine_value=${machineValue}&dependent_field=emp_id`);
            const empResult = await empResponse.json();

            const machines = Array.isArray(statsResult.machines) ? statsResult.machines : [];
            const employees = Array.isArray(empResult.employees_working) ? empResult.employees_working : [];

            const hasMachineValues = machines.some(m => Number(m.total_production_qty ?? m.total_operations ?? 0) > 0 || Number(m.employee_count || 0) > 0);
            const hasEmployeeValues = employees.some(e => Number(e.total_production_qty ?? e.total_operations ?? e.count ?? 0) > 0);

            if (!hasMachineValues && !hasEmployeeValues) {
                document.getElementById('visualDashboardSection').style.display = 'none';
                return;
            }

            // Show visual dashboard
            document.getElementById('visualDashboardSection').style.display = 'block';

            // Draw all charts
            drawMachineDistributionCharts(machines);
            drawEmployeeCharts(employees);
            updateDashboardMetrics(statsResult, empResult);

        } catch (error) {
            console.error('Error creating visual dashboard:', error);
            document.getElementById('visualDashboardSection').style.display = 'none';
        }
    }

    // ===== DRAW MACHINE DISTRIBUTION CHARTS =====
    function drawMachineDistributionCharts(machines) {
        const machineList = Array.isArray(machines) ? machines : [];
        const hasUsableMachineData = machineList.some(m => Number(m.total_production_qty ?? m.total_operations ?? 0) > 0 || Number(m.employee_count || 0) > 0);

        if (!hasUsableMachineData) {
            setCardVisibilityByCanvasId('machineDistributionChart', false);
            setCardVisibilityByCanvasId('machineComparisonChart', false);
            if (window.machineDistributionChartInstance) {
                window.machineDistributionChartInstance.destroy();
                window.machineDistributionChartInstance = null;
            }
            if (window.machineComparisonChartInstance) {
                window.machineComparisonChartInstance.destroy();
                window.machineComparisonChartInstance = null;
            }
            return;
        }

        setCardVisibilityByCanvasId('machineDistributionChart', true);
        setCardVisibilityByCanvasId('machineComparisonChart', true);

        const machineNames = machineList.map(m => m.machine_name || m.machine_value || 'N/A');
        const machineOps = machineList.map(m => Number(m.total_production_qty ?? m.total_operations ?? 0));
        const machineHours = machineList.map(m => Number(m.total_hours ?? 0));
        const machineEmps = machineList.map(m => m.employee_count);

        const colors = ['#667eea', '#764ba2', '#f093fb', '#f5576c', '#4facfe', '#00f2fe', '#43e97b', '#38f9d7', '#1abc9c', '#16a085'];

        // 1. Vertical Bar Chart
        const barCtx = document.getElementById('machineDistributionChart');
        if (barCtx && window.machineDistributionChartInstance) {
            window.machineDistributionChartInstance.destroy();
        }
        if (barCtx) {
            window.machineDistributionChartInstance = new Chart(barCtx, {
                type: 'bar',
                data: {
                    labels: machineNames,
                    datasets: [{
                        label: 'Total Production',
                        data: machineOps,
                        backgroundColor: colors.slice(0, machineNames.length),
                        borderColor: colors.slice(0, machineNames.length),
                        borderWidth: 2,
                        borderRadius: 6,
                        hoverBackgroundColor: '#1abc9c'
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: { display: true, position: 'top' },
                        tooltip: { backgroundColor: 'rgba(0,0,0,0.8)', padding: 12 }
                    },
                    scales: { y: { beginAtZero: true } }
                }
            });
        }

        // 2. Horizontal Bar Chart
        const hBarCtx = document.getElementById('machineComparisonChart');
        if (hBarCtx && window.machineComparisonChartInstance) {
            window.machineComparisonChartInstance.destroy();
        }
        if (hBarCtx) {
            window.machineComparisonChartInstance = new Chart(hBarCtx, {
                type: 'bar',
                data: {
                    labels: machineNames,
                    datasets: [
                        {
                            label: 'Production',
                            data: machineOps,
                            backgroundColor: 'rgba(102, 126, 234, 0.8)',
                            borderColor: '#667eea',
                            borderWidth: 2
                        },
                        {
                            label: 'Hours',
                            data: machineHours,
                            backgroundColor: 'rgba(67, 233, 123, 0.8)',
                            borderColor: '#43e97b',
                            borderWidth: 2
                        },
                        {
                            label: 'Employees',
                            data: machineEmps,
                            backgroundColor: 'rgba(240, 147, 251, 0.8)',
                            borderColor: '#f093fb',
                            borderWidth: 2
                        }
                    ]
                },
                options: {
                    indexAxis: 'y',
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: { display: true, position: 'top' },
                        tooltip: { backgroundColor: 'rgba(0,0,0,0.8)', padding: 12 }
                    },
                    scales: { x: { beginAtZero: true } }
                }
            });
        }
    }

    // ===== DRAW EMPLOYEE CHARTS =====
    function drawEmployeeCharts(employees) {
        const employeeList = Array.isArray(employees) ? employees : [];
        const hasUsableEmployeeData = employeeList.some(e => Number(e.total_production_qty ?? e.total_operations ?? e.count ?? 0) > 0);

        if (!hasUsableEmployeeData) {
            ['topEmployeesChart', 'employeeActivityPieChart', 'machineAssignmentChart', 'bubbleEfficiencyChart', 'radarMetricsChart']
                .forEach(id => setCardVisibilityByCanvasId(id, false));

            if (window.topEmployeesChartInstance) { window.topEmployeesChartInstance.destroy(); window.topEmployeesChartInstance = null; }
            if (window.employeeActivityPieChartInstance) { window.employeeActivityPieChartInstance.destroy(); window.employeeActivityPieChartInstance = null; }
            if (window.machineAssignmentChartInstance) { window.machineAssignmentChartInstance.destroy(); window.machineAssignmentChartInstance = null; }
            if (window.bubbleEfficiencyChartInstance) { window.bubbleEfficiencyChartInstance.destroy(); window.bubbleEfficiencyChartInstance = null; }
            if (window.radarMetricsChartInstance) { window.radarMetricsChartInstance.destroy(); window.radarMetricsChartInstance = null; }
            return;
        }

        ['topEmployeesChart', 'employeeActivityPieChart', 'machineAssignmentChart', 'bubbleEfficiencyChart', 'radarMetricsChart']
            .forEach(id => setCardVisibilityByCanvasId(id, true));

        // Sort by production quantity
        const sorted = [...employeeList].sort((a, b) => Number(b.total_production_qty ?? b.total_operations ?? b.count ?? 0) - Number(a.total_production_qty ?? a.total_operations ?? a.count ?? 0)).slice(0, 10);
        const empNames = sorted.map(e => e.operator_name || `EMP${e.emp_id}`);
        const empOps = sorted.map(e => Number(e.total_production_qty ?? e.total_operations ?? e.count ?? 0));

        const colors = ['#667eea', '#764ba2', '#f093fb', '#f5576c', '#4facfe', '#00f2fe', '#43e97b', '#38f9d7', '#1abc9c', '#16a085'];

        // 1. Top Employees Bar Chart
        const topEmpCtx = document.getElementById('topEmployeesChart');
        if (topEmpCtx && window.topEmployeesChartInstance) {
            window.topEmployeesChartInstance.destroy();
        }
        if (topEmpCtx) {
            window.topEmployeesChartInstance = new Chart(topEmpCtx, {
                type: 'bar',
                data: {
                    labels: empNames,
                    datasets: [{
                        label: 'Production Qty',
                        data: empOps,
                        backgroundColor: colors.slice(0, empNames.length),
                        borderColor: colors.slice(0, empNames.length),
                        borderWidth: 2,
                        borderRadius: 6
                    }]
                },
                options: {
                    indexAxis: empNames.length > 6 ? 'y' : 'x',
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: { display: true, position: 'top' },
                        tooltip: { backgroundColor: 'rgba(0,0,0,0.8)', padding: 12 }
                    },
                    scales: { x: { beginAtZero: true }, y: { beginAtZero: true } }
                }
            });
        }

        // 2. Employee Activity Pie Chart
        const pieCtx = document.getElementById('employeeActivityPieChart');
        if (pieCtx && window.employeeActivityPieChartInstance) {
            window.employeeActivityPieChartInstance.destroy();
        }
        if (pieCtx) {
            window.employeeActivityPieChartInstance = new Chart(pieCtx, {
                type: 'doughnut',
                data: {
                    labels: empNames,
                    datasets: [{
                        data: empOps,
                        backgroundColor: colors.slice(0, empNames.length),
                        borderColor: '#fff',
                        borderWidth: 2,
                        hoverOffset: 10
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: { position: 'right', labels: { font: { size: 11 } } },
                        tooltip: {
                            backgroundColor: 'rgba(0,0,0,0.8)',
                            callbacks: {
                                label: function(context) {
                                    const total = context.dataset.data.reduce((a, b) => a + b, 0);
                                    const percentage = ((context.parsed / total) * 100).toFixed(1);
                                    return `${context.label}: ${context.parsed} (${percentage}%)`;
                                }
                            }
                        }
                    }
                }
            });
        }

        // 3. Machine Assignment Bubble Chart
        const bubbleCtx = document.getElementById('machineAssignmentChart');
        if (bubbleCtx && window.machineAssignmentChartInstance) {
            window.machineAssignmentChartInstance.destroy();
        }
        if (bubbleCtx) {
            const bubbleData = sorted.map((e, i) => ({
                x: Number(e.working_days || 1),
                y: Number(e.total_production_qty ?? e.total_operations ?? e.count ?? 0),
                r: (Number(e.total_production_qty ?? e.total_operations ?? e.count ?? 0) / 10) + 5
            }));

            window.machineAssignmentChartInstance = new Chart(bubbleCtx, {
                type: 'bubble',
                data: {
                    datasets: [{
                        label: 'Employee Performance',
                        data: bubbleData,
                        backgroundColor: colors.slice(0, bubbleData.length).map(c => c + '80'),
                        borderColor: colors.slice(0, bubbleData.length),
                        borderWidth: 2
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: { display: true, position: 'top' },
                        tooltip: {
                            backgroundColor: 'rgba(0,0,0,0.8)',
                            callbacks: {
                                label: function(context) {
                                    return `Days: ${context.raw.x}, Qty: ${context.raw.y}`;
                                }
                            }
                        }
                    },
                    scales: {
                        x: { title: { display: true, text: 'Working Days' }, beginAtZero: true },
                        y: { title: { display: true, text: 'Total Production' }, beginAtZero: true }
                    }
                }
            });
        }

        // 4. Radar Chart - Multi-Metric Comparison
        const radarCtx = document.getElementById('radarMetricsChart');
        if (radarCtx && window.radarMetricsChartInstance) {
            window.radarMetricsChartInstance.destroy();
        }
        if (radarCtx) {
            const topEmps = sorted.slice(0, 5);
            window.radarMetricsChartInstance = new Chart(radarCtx, {
                type: 'radar',
                data: {
                    labels: ['Productivity', 'Consistency', 'Activity', 'Efficiency', 'Performance'],
                    datasets: topEmps.map((e, i) => ({
                        label: e.operator_name || `Employee ${i + 1}`,
                        data: [
                            Number(e.total_production_qty ?? e.total_operations ?? e.count ?? 0) / 10,
                            Number(e.working_days || 1) * 2,
                            (Number(e.total_production_qty ?? e.total_operations ?? e.count ?? 0) / 5),
                            (Number(e.total_production_qty ?? e.total_operations ?? e.count ?? 0) / Number(e.working_days || 1)).toFixed(0),
                            Math.min((Number(e.total_production_qty ?? e.total_operations ?? e.count ?? 0) / 50) * 100, 100)
                        ],
                        borderColor: colors[i],
                        backgroundColor: colors[i] + '20',
                        borderWidth: 2,
                        pointRadius: 4,
                        pointHoverRadius: 6
                    }))
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: { legend: { position: 'right' } },
                    scales: { r: { beginAtZero: true, max: 100 } }
                }
            });
        }

        // 5. Efficiency Bubble Chart
        const effCtx = document.getElementById('bubbleEfficiencyChart');
        if (effCtx && window.bubbleEfficiencyChartInstance) {
            window.bubbleEfficiencyChartInstance.destroy();
        }
        if (effCtx) {
            const effData = sorted.map(e => ({
                x: Number(e.working_days || 1),
                y: (Number(e.total_production_qty ?? e.total_operations ?? e.count ?? 0) / Number(e.working_days || 1)).toFixed(1),
                r: Number(e.total_production_qty ?? e.total_operations ?? e.count ?? 0) / 8
            }));

            window.bubbleEfficiencyChartInstance = new Chart(effCtx, {
                type: 'bubble',
                data: {
                    datasets: [{
                        label: 'Production per Day',
                        data: effData,
                        backgroundColor: colors.slice(0, effData.length).map(c => c + '80'),
                        borderColor: colors.slice(0, effData.length),
                        borderWidth: 2
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: { display: true },
                        tooltip: {
                            callbacks: {
                                label: function(context) {
                                    return `Avg per Day: ${context.raw.y}`;
                                }
                            }
                        }
                    },
                    scales: {
                        x: { title: { display: true, text: 'Working Days' }, beginAtZero: true },
                        y: { title: { display: true, text: 'Production per Day' }, beginAtZero: true }
                    }
                }
            });
        }
    }

    // ===== UPDATE DASHBOARD METRICS =====
    function updateDashboardMetrics(statsResult, empResult) {
        const machines = statsResult.machines || [];
        const employees = empResult.employees_working || [];

        const totalMachines = machines.length;
        const totalEmployees = employees.length;
        const totalOps = machines.reduce((sum, m) => sum + Number(m.total_production_qty ?? m.total_operations ?? 0), 0);
        const totalHours = machines.reduce((sum, m) => sum + Number(m.total_hours ?? 0), 0);
        const avgOps = totalOps > 0 ? (totalOps / (totalMachines || 1)).toFixed(2) : 0;
        const efficiency = totalHours > 0 ? ((totalOps / totalHours) * 100).toFixed(1) : 0;

        document.getElementById('totalMachinesMetric').textContent = totalMachines;
        document.getElementById('totalEmployeesMetric').textContent = totalEmployees;
        document.getElementById('avgOperationsMetric').textContent = avgOps;
        document.getElementById('efficiencyMetric').textContent = efficiency + '%';
    }

    // ===== LOAD ALL FIELDS FOR COMPREHENSIVE ANALYSIS =====
    async function loadAllFieldsAnalytics(machineFieldName, machineValue) {
        try {
            const response = await fetch(`api.php?action=get_all_fields`);
            const result = await response.json();

            if (result.success && result.all_fields && result.all_fields.length > 0) {
                document.getElementById('allFieldsSection').style.display = 'block';
                displayAllFieldsCharts(result.all_fields, machineFieldName, machineValue);
            }
        } catch (error) {
            console.error('Error loading all fields:', error);
        }
    }

    // ===== DISPLAY ALL FIELDS AS CHARTS =====
    async function displayAllFieldsCharts(allFields, machineFieldName, machineValue) {
        const container = document.getElementById('allFieldsChartsContainer');
        container.innerHTML = '';

        let fieldCount = 0;
        const colors = ['#667eea', '#764ba2', '#f093fb', '#f5576c', '#4facfe', '#00f2fe', '#43e97b', '#38f9d7', '#1abc9c', '#16a085'];

        for (const field of allFields) {
            if (fieldCount >= 12) break; // Limit to 12 fields for performance

            try {
                const response = await fetch(`api.php?action=get_machine_specific_stats&machine_field=${machineFieldName}&machine_value=${machineValue}&dependent_field=${field.field_name}`);
                const result = await response.json();

                if (result.success && result.data && result.data.length > 0) {
                    const fieldData = result.data.slice(0, 8); // Top 8 values
                    const labels = fieldData.map(d => (d.value || 'N/A').substring(0, 15));
                    const values = fieldData.map(d => Number(d.total_production_qty ?? d.count ?? 0));
                    const hasUsableData = values.some(v => Number(v) > 0);
                    if (!hasUsableData) {
                        continue;
                    }

                    // Create field card with 3 charts
                    const fieldCard = document.createElement('div');
                    fieldCard.className = 'trend-chart-container';
                    fieldCard.innerHTML = `
                        <div class="trend-chart-title">
                            <i class="fas fa-cube"></i> ${field.field_label}
                            <span style="font-size: 11px; margin-left: auto; color: #999;">${values.length} values</span>
                        </div>
                        <div style="position: relative; min-height: 220px; height: 220px;">
                            <canvas id="allfield-chart-${field.field_name}"></canvas>
                        </div>
                        <div style="margin-top: 15px; padding-top: 15px; border-top: 1px solid #eee;">
                            <small style="color: #999;">
                                ${values.reduce((a,b)=>a+b, 0).toFixed(2)} total qty | 
                                Avg: ${(values.reduce((a,b)=>a+b, 0) / values.length).toFixed(2)} | 
                                Max: ${Math.max(...values).toFixed(2)}
                            </small>
                        </div>
                    `;
                    container.appendChild(fieldCard);

                    // Draw chart after element is added
                    setTimeout(() => {
                        const ctx = document.getElementById(`allfield-chart-${field.field_name}`);
                        if (ctx) {
                            new Chart(ctx, {
                                type: 'bar',
                                data: {
                                    labels: labels,
                                    datasets: [{
                                        label: `${field.field_label} Production`,
                                        data: values,
                                        backgroundColor: colors[fieldCount % colors.length],
                                        borderColor: colors[fieldCount % colors.length],
                                        borderWidth: 2,
                                        borderRadius: 6
                                    }]
                                },
                                options: {
                                    responsive: true,
                                    maintainAspectRatio: false,
                                    indexAxis: values.length > 5 ? 'y' : 'x',
                                    plugins: {
                                        legend: { display: true },
                                        tooltip: { backgroundColor: 'rgba(0,0,0,0.8)', padding: 12 }
                                    },
                                    scales: {
                                        x: { beginAtZero: true },
                                        y: { beginAtZero: true }
                                    }
                                }
                            });
                        } else {
                            fieldCard.remove();
                            fieldCount = Math.max(0, fieldCount - 1);
                        }
                    }, 100);

                    fieldCount++;
                }
            } catch (error) {
                console.error(`Error loading field ${field.field_name}:`, error);
            }
        }

        // Show message if no fields to display
        if (fieldCount === 0) {
            container.innerHTML = '<p class="no-data">No field data available for this machine</p>';
        }
    }

    // ===== TOGGLE ALL FIELDS VIEW =====
    function toggleAllFieldsView(viewType) {
        const container = document.getElementById('allFieldsChartsContainer');
        if (viewType === 'grid') {
            container.style.gridTemplateColumns = 'repeat(auto-fit, minmax(320px, 1fr))';
        } else if (viewType === 'cards') {
            container.style.gridTemplateColumns = 'repeat(auto-fit, minmax(420px, 1fr))';
        }
    }

    // ===== INITIALIZE ON PAGE LOAD =====
    document.addEventListener('DOMContentLoaded', function() {
        loadMachineFields();
    });
</script>

</body>
</html>
