<?php
// Protect this page with session check
include 'session_check.php';
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>Daily Production Entry</title>
    <meta name="viewport" content="width=device-width, initial-scale=1.0">

    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">

    <link rel="stylesheet" href="style.css">
    <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>

    <style>
        /* ===============================
           LAYOUT & STYLE (UNCHANGED)
        =============================== */
        body {
            margin: 0;
            font-family: 'Segoe UI', sans-serif;
            background-color: #f4f6f9;
        }

        .main-content {
            margin-left: 260px;
            padding: 40px;
        }

        @media (max-width: 768px) {
            .main-content { margin-left: 0; padding: 20px; }
        }

        .container {
            background: #fff;
            padding: 40px;
            border-radius: 8px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.05);
            max-width: 900px;
            margin: auto;
        }

        h2 {
            margin-top: 0; color: #2c3e50; border-bottom: 2px solid #eee;
            padding-bottom: 15px; margin-bottom: 30px;
        }

        .highlight-box {
            background: #fff8e1; padding: 25px; border: 1px solid #ffe082;
            border-radius: 6px; margin-bottom: 30px; display: grid;
            grid-template-columns: 1fr 1fr; gap: 25px;
        }

        .form-grid { display: grid; grid-template-columns: 1fr 1fr; gap: 25px; }

        @media (max-width: 768px) {
            .highlight-box, .form-grid { grid-template-columns: 1fr; }
        }

        .form-group label {
            display: block; margin-bottom: 8px; font-weight: 600;
            color: #555; font-size: 14px;
        }

        .form-control {
            width: 100%; padding: 12px; border: 1px solid #ddd;
            border-radius: 4px; font-size: 14px; box-sizing: border-box;
        }
        .form-control:focus { border-color: #00a65a; outline: none; }
        .form-control[readonly] { background: #e9ecef; cursor: not-allowed; }

        .btn-submit {
            background: #00a65a; color: #fff; border: none; padding: 15px;
            width: 100%; font-size: 16px; font-weight: bold; border-radius: 4px;
            cursor: pointer; margin-top: 30px; transition: 0.3s;
        }
        .btn-submit:hover { background: #008d4c; }

        .status-msg {
            padding: 15px; margin-bottom: 20px; border-radius: 5px;
            font-weight: bold; display: none;
        }
        .status-success { background: #d4edda; color: #155724; border: 1px solid #c3e6cb; }
        .status-error { background: #f8d7da; color: #721c24; border: 1px solid #f5c6cb; }

        .loading-text { grid-column: 1 / -1; text-align: center; color: #777; font-style: italic; }

        /* Lunch Period Indicator */
        .lunch-period-badge {
            display: inline-block; margin-left: 10px; padding: 6px 12px;
            background: #fff3cd; color: #856404; border: 1px solid #f39c12;
            border-radius: 4px; font-size: 12px; font-weight: 600;
            animation: pulseWarning 2s infinite;
        }

        @keyframes pulseWarning {
            0%, 100% { opacity: 1; }
            50% { opacity: 0.7; }
        }

        .lunch-period-info {
            background: #fff9e6; border: 2px dashed #f39c12;
            padding: 12px 16px; border-radius: 6px; margin-bottom: 20px;
            color: #856404; font-size: 13px; display: none;
        }
        .lunch-period-info.active { display: block; }
        .lunch-period-info i { margin-right: 8px; font-size: 16px; }
    </style>
</head>

<body>

<script src="sidebar.js"></script>

<div class="main-content">
    <div class="container">
        <h2><i class="fas fa-edit"></i> Daily Production Entry</h2>

        <div id="statusMessage" class="status-msg"></div>

        <div id="lunchPeriodIndicator" class="lunch-period-info">
            <i class="fas fa-utensils"></i><strong>Lunch Deduction Applied:</strong> Shift overlaps 1:00 PM - 1:30 PM. Duration and norms are reduced by 30 minutes.
        </div>

        <form id="productionForm">
            <div class="highlight-box">
                <div class="form-group">
                    <label>Date <span style="color:red">*</span></label>
                    <input type="date" class="form-control" id="logDate" name="log_date" required>
                </div>

                <div class="form-group">
                    <label>Employee ID <span style="color:red">*</span></label>
                    <select class="form-control" id="empSelect" name="empIdSelect" required>
                        <option value="">Loading Employees...</option>
                    </select>
                </div>

                <input type="hidden" name="operator_name" id="operatorName">
            </div>

            <div id="dynamicFields" class="form-grid">
                <p class="loading-text">Connecting to database...</p>
            </div>

            <!-- Auto-Calculated Fields (Read-Only) -->
            <div class="form-group">
                <label>NC Qty (Auto-Calculated)</label>
                <input type="text" class="form-control" id="ncQtyInput" readonly placeholder="Auto-calculated">
            </div>

            <div class="form-group">
                <label>Production Efficiency (%)</label>
                <input type="text" class="form-control" id="efficiencyInput" readonly placeholder="Auto-calculated">
            </div>

            <div class="form-group">
                <label>NC (%)</label>
                <input type="text" class="form-control" id="ncPercentInput" readonly placeholder="Auto-calculated">
            </div>

            <button type="submit" class="btn-submit">Save Entry to Log</button>
        </form>
    </div>
</div>
<script>
    // ===== MODERN CENTERED MODAL SYSTEM =====
    function showModal(title, message = '', type = 'alert', callback = null) {
        let overlay = document.getElementById('modalOverlay');
        if (!overlay) {
            overlay = document.createElement('div');
            overlay.id = 'modalOverlay';
            overlay.className = 'modal-overlay';
            document.body.appendChild(overlay);
            
            if (!document.querySelector('style[data-modals]')) {
                const style = document.createElement('style');
                style.setAttribute('data-modals', 'true');
                style.textContent = `.modal-overlay { position: fixed; top: 0; left: 0; width: 100%; height: 100%; background: rgba(0, 0, 0, 0.55); display: flex; align-items: center; justify-content: center; padding: 20px; z-index: 10000; animation: fadeIn 0.3s cubic-bezier(0.4, 0, 0.2, 1); backdrop-filter: blur(2px); }
                    @keyframes fadeIn { from { opacity: 0; } to { opacity: 1; } }
                    @keyframes zoomIn { from { transform: scale(0.85) translateY(-30px); opacity: 0; } to { transform: scale(1) translateY(0); opacity: 1; } }
                    .modal-content { background: white; border-radius: 16px; box-shadow: 0 25px 80px rgba(0, 0, 0, 0.35); max-width: 540px; width: 100%; max-height: 85vh; padding: 0; overflow: auto; animation: zoomIn 0.35s cubic-bezier(0.34, 1.56, 0.64, 1); font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif; display: flex; flex-direction: column; }
                    .modal-header { padding: 32px 32px 24px; border-bottom: 1px solid #f0f0f0; display: flex; align-items: flex-start; gap: 16px; flex-shrink: 0; }
                    .modal-icon { font-size: 28px; flex-shrink: 0; width: 44px; height: 44px; display: flex; align-items: center; justify-content: center; border-radius: 50%; margin-top: 2px; }
                    .modal-icon.success { color: #00a65a; background: rgba(0, 166, 90, 0.1); }
                    .modal-icon.error { color: #dd4b39; background: rgba(221, 75, 57, 0.1); }
                    .modal-icon.warning { color: #f39c12; background: rgba(243, 156, 18, 0.1); }
                    .modal-icon.confirm { color: #2196f3; background: rgba(33, 150, 243, 0.1); }
                    .modal-title { font-size: 20px; font-weight: 700; color: #1a1a1a; margin: 0; line-height: 1.4; flex: 1; }
                    .modal-body { padding: 24px 32px; color: #555; font-size: 14px; line-height: 1.7; flex: 1; overflow-y: auto; }
                    .modal-footer { padding: 24px 32px; border-top: 1px solid #f0f0f0; display: flex; gap: 12px; justify-content: flex-end; background: #fafafa; flex-shrink: 0; }
                    .modal-btn { padding: 12px 32px; border: none; border-radius: 8px; font-size: 14px; font-weight: 600; cursor: pointer; transition: all 0.25s cubic-bezier(0.4, 0, 0.2, 1); min-width: 110px; display: inline-flex; align-items: center; justify-content: center; white-space: nowrap; }
                    .modal-btn-ok { background: linear-gradient(135deg, #00a65a 0%, #008c47 100%); color: white; box-shadow: 0 4px 15px rgba(0, 166, 90, 0.25); }
                    .modal-btn-ok:hover { transform: translateY(-2px); box-shadow: 0 6px 25px rgba(0, 166, 90, 0.35); }
                    .modal-btn-ok:active { transform: translateY(0); }
                    .modal-btn-confirm { background: linear-gradient(135deg, #2196f3 0%, #1976d2 100%); color: white; box-shadow: 0 4px 15px rgba(33, 150, 243, 0.25); }
                    .modal-btn-confirm:hover { transform: translateY(-2px); box-shadow: 0 6px 25px rgba(33, 150, 243, 0.35); }
                    .modal-btn-cancel { background: #e0e0e0; color: #333; box-shadow: 0 2px 8px rgba(0, 0, 0, 0.1); }
                    .modal-btn-cancel:hover { background: #d0d0d0; transform: translateY(-2px); }
                    @media (max-width: 600px) { .modal-overlay { padding: 16px; } .modal-content { max-width: 100%; max-height: 90vh; border-radius: 14px; } .modal-header { padding: 24px 20px 18px; gap: 12px; } .modal-icon { width: 40px; height: 40px; } .modal-title { font-size: 18px; } .modal-body { padding: 20px; } .modal-footer { padding: 20px; } .modal-btn { flex: 1; min-width: auto; padding: 11px 20px; } }`;
                document.head.appendChild(style);
            }
        }
        
        const icons = { success: 'fas fa-check-circle', error: 'fas fa-exclamation-circle', warning: 'fas fa-warning', alert: 'fas fa-info-circle', confirm: 'fas fa-question-circle' };
        const modal = document.createElement('div');
        modal.className = 'modal-content';
        
        let buttonsHTML = `<button class="modal-btn modal-btn-ok" onclick="this.closest('.modal-overlay').remove()">OK</button>`;
        
        if (type === 'confirm') {
            buttonsHTML = `
                <button class="modal-btn modal-btn-confirm" onclick="window.modalResult = true; this.closest('.modal-overlay').remove()">Yes, Continue</button>
                <button class="modal-btn modal-btn-cancel" onclick="window.modalResult = false; this.closest('.modal-overlay').remove()">Cancel</button>
            `;
        }
        
        modal.innerHTML = `<div class="modal-header"><div class="modal-icon ${type}"><i class="fas ${icons[type]}"></i></div><h2 class="modal-title">${title}</h2></div><div class="modal-body">${message}</div><div class="modal-footer">${buttonsHTML}</div>`;
        overlay.innerHTML = '';
        overlay.appendChild(modal);
        
        if (type === 'confirm') {
            window.modalResult = null;
            // Wait for user response
            const checkInterval = setInterval(() => {
                if (window.modalResult !== null) {
                    clearInterval(checkInterval);
                    if (callback) callback(window.modalResult);
                    window.modalResult = null;
                }
            }, 100);
        }
        
        overlay.onclick = (e) => { if (e.target === overlay) overlay.remove(); };
    }

    let startTimeInput = null, endTimeInput = null, durationInput = null;
    let normInput = null; // To store the "Norms" input field
    let planQtyInput = null, producedQtyInput = null, acceptedQtyInput = null;
    let ncQtyInput = null, efficiencyInput = null, ncPercentInput = null;

    // Initialize efficiency and NC% inputs
    ncQtyInput = document.getElementById('ncQtyInput');
    efficiencyInput = document.getElementById('efficiencyInput');
    ncPercentInput = document.getElementById('ncPercentInput');

    // 2. LUNCH PERIOD ADJUSTMENT - Reduce 30 mins when shift overlaps 1:00 PM to 1:30 PM
    function checkLunchPeriodAdjustment(startMin, endMin) {
        const LUNCH_START = 13 * 60;      // 1:00 PM = 13:00
        const LUNCH_END = 13 * 60 + 30;   // 1:30 PM = 13:30
        const indicator = document.getElementById('lunchPeriodIndicator');

        // Apply deduction when interval overlaps lunch slot.
        // If user starts after 1:30 PM, this condition is false.
        const overlapsSameDay = startMin < LUNCH_END && endMin > LUNCH_START;
        const overlapsNextDay = startMin < (LUNCH_END + 1440) && endMin > (LUNCH_START + 1440);
        const applyLunchDeduction = overlapsSameDay || overlapsNextDay;

        if (applyLunchDeduction) {
            if (indicator) indicator.classList.add('active');
            return 30;
        }

        if (indicator) indicator.classList.remove('active');
        return 0;
    }

    // 2. DURATION CALC - When start/end time changes, recalculate duration and norms
    function calculateDuration() {
        if (startTimeInput && endTimeInput && durationInput) {
            const s = startTimeInput.value, e = endTimeInput.value;
            if (s && e) {
                const sMin = parseInt(s.split(':')[0]) * 60 + parseInt(s.split(':')[1]);
                const eMin = parseInt(e.split(':')[0]) * 60 + parseInt(e.split(':')[1]);
                let adjustedEndMin = eMin;
                if (adjustedEndMin < sMin) adjustedEndMin += 1440;
                let diff = adjustedEndMin - sMin;

                const lunchReduction = checkLunchPeriodAdjustment(sMin, adjustedEndMin);
                if (lunchReduction > 0) {
                    diff = Math.max(0, diff - lunchReduction);
                    durationInput.style.backgroundColor = '#fff3cd';
                    durationInput.style.borderColor = '#f39c12';
                    durationInput.title = 'Lunch overlap (1:00-1:30 PM) - 30 mins auto reduced';
                } else {
                    durationInput.style.backgroundColor = '';
                    durationInput.style.borderColor = '';
                    durationInput.title = '';
                }

                const hours = Math.floor(diff / 60);
                const minutes = diff % 60;
                durationInput.value = `${hours.toString().padStart(2, '0')}:${minutes.toString().padStart(2, '0')}`;

                // Recalculate norms using adjusted duration
                calculateTotalNorms(diff);
            }
        }
    }

    // 3. âœ… MASTER CALCULATION FUNCTION - Calculates norms based on Tool Rate Ã— Duration (in hours)
    function calculateTotalNorms(durationMinutes = null) {
        if (!normInput) return; // Exit if no "Norms" field found

        // A. Get Duration in minutes (if not passed)
        if (!durationMinutes && durationInput && durationInput.value) {
            const parts = durationInput.value.split(':');
            durationMinutes = (parseInt(parts[0]) * 60) + parseInt(parts[1]);
        }

        if (!durationMinutes || durationMinutes <= 0) {
            normInput.value = ""; // Clear if no duration
            return;
        }

        // B. Find TOOL field and get its selected norm rate
        let toolNormRate = 0;
        const dropdowns = document.querySelectorAll('.dropdown-field');

        dropdowns.forEach(dd => {
            const fieldName = (dd.name || '').toLowerCase();
            // Check if this is a TOOL field
            if (fieldName.includes('tool')) {
                const selectedOption = dd.options[dd.selectedIndex];
                if (selectedOption && selectedOption.dataset.norm) {
                    toolNormRate = parseFloat(selectedOption.dataset.norm) || 0;
                }
            }
        });

        // C. Calculate Total Norms: (Duration in Hours) Ã— (Tool Norm Rate)
        if (toolNormRate > 0) {
            const durationHours = durationMinutes / 60;
            const totalNorms = Math.round(durationHours * toolNormRate);
            normInput.value = totalNorms;
            console.log(`âœ… Tool Norm Rate: ${toolNormRate}/hr Ã— Duration: ${durationHours.toFixed(2)} hours = Total Norms: ${totalNorms}`);
        } else {
            normInput.value = ""; // Empty if tool not selected or no rate
        }

        // D. Trigger efficiency and NC calculation after norms update
        calculateEfficiencyAndNC();
    }

    // 4. âœ… EFFICIENCY AND NC% CALCULATION FUNCTION (WITH PROPER VALIDATION)
    function calculateEfficiencyAndNC() {
        // A. Get values from form
        const planQty = planQtyInput ? parseFloat(planQtyInput.value) || 0 : 0;
        const producedQty = producedQtyInput ? parseFloat(producedQtyInput.value) || 0 : 0;
        const acceptedQty = acceptedQtyInput ? parseFloat(acceptedQtyInput.value) || 0 : 0;

        // === VALIDATION ===
        // Clear efficiency/NC if plan qty is not set
        if (!planQty || planQty <= 0) {
            if (efficiencyInput) efficiencyInput.value = '';
            if (ncPercentInput) ncPercentInput.value = '';
            return;
        }

        // Check for invalid inputs
        if (producedQty < 0) {
            if (efficiencyInput) efficiencyInput.value = '';
            if (ncPercentInput) ncPercentInput.value = '';
            return;
        }

        if (acceptedQty < 0 || acceptedQty > producedQty) {
            if (ncPercentInput) ncPercentInput.value = '';
            return;
        }

        // B. Calculate NC Qty first (Produced - Accepted)
        const ncQty = Math.max(producedQty - acceptedQty, 0);

        // Display NC Qty (Auto-Calculated Read-Only)
        if (ncQtyInput) {
            ncQtyInput.value = ncQty.toFixed(0);  // Show as whole number
        }

        // C. Calculate Production Efficiency: (Produced Qty / Plan Qty) Ã— 100 âœ… CORRECT FORMULA
        if (efficiencyInput && planQty > 0) {
            const efficiency = (producedQty / planQty) * 100;
            const effValue = efficiency.toFixed(2);
            
            // Color indicator for efficiency
            let displayText = effValue + '%';
            if (efficiency > 100) {
                displayText += ' ðŸ”µ Over Target';  // Blue background in CSS
            } else if (efficiency >= 85) {
                displayText += ' ðŸŸ¢';  // Green
            } else if (efficiency >= 70) {
                displayText += ' ðŸŸ ';  // Orange
            } else {
                displayText += ' ðŸ”´';  // Red
            }
            
            efficiencyInput.value = displayText;
        } else if (efficiencyInput) {
            efficiencyInput.value = '';
        }

        // D. Calculate NC%: (NC Qty / Produced Qty) Ã— 100
        if (ncPercentInput && producedQty > 0) {
            const ncPercent = (ncQty / producedQty) * 100;
            ncPercentInput.value = ncPercent.toFixed(2) + '%';
        } else if (ncPercentInput) {
            ncPercentInput.value = '';
        }
    }

    document.addEventListener("DOMContentLoaded", () => {
        document.getElementById('logDate').value = new Date().toISOString().split('T')[0];
        loadEmployees();
        generateDynamicForm();
    });

    // 1. GENERATE FORM
    async function generateDynamicForm() {
        const container = document.getElementById('dynamicFields');
        try {
            const res = await fetch('api.php?action=get_custom_fields&type=active');
            const fields = await res.json();
            container.innerHTML = '';

            if (!fields || fields.length === 0) {
                container.innerHTML = '<p class="loading-text">No active fields found.</p>';
                return;
            }

            fields.forEach(field => {
                const div = document.createElement('div');
                div.className = 'form-group';
                div.innerHTML = `<label>${field.field_label}</label>`;
                
                let input;
                const name = field.field_name.toLowerCase();

                if (field.field_type === 'dropdown') {
                    input = document.createElement('select');
                    input.className = 'form-control dropdown-field';
                    input.name = field.field_name;
                    input.dataset.fieldId = field.id;
                    input.dataset.parentId = field.parent_field_id || '';

                    if (!field.parent_field_id) loadOptions(input, field.id);
                    else {
                        input.disabled = true;
                        input.innerHTML = '<option value="">Select Parent First</option>';
                    }
                    
                    // RECALCULATE ON CHANGE
                    input.addEventListener('change', () => {
                        triggerChildUpdates(input);
                        // Always recalculate norms after any dropdown change
                        calculateTotalNorms();
                    });
                } 
                else {
                    input = document.createElement('input');
                    input.className = 'form-control';
                    input.name = field.field_name;

                if (name.includes('start_time')) {
                        input.type = 'time'; startTimeInput = input;
                        input.addEventListener('change', calculateDuration);
                    } else if (name.includes('end_time')) {
                        input.type = 'time'; endTimeInput = input;
                        input.addEventListener('change', calculateDuration);
                    } else if (name.includes('duration')) {
                        input.type = 'text'; input.readOnly = true;
                        durationInput = input; input.placeholder = "Auto-calculated";
                        input.addEventListener('change', calculateEfficiencyAndNC); // Trigger calculation when duration changes
                    }
                    // AUTO-DETECT NORMS FIELD
                    else if (name.includes('norm') || name.includes('target')) {
                        input.type = 'text';
                        input.readOnly = true;
                        input.style.backgroundColor = "#e8f0fe"; // Light blue to show it's auto
                        normInput = input;
                    }
                    // AUTO-DETECT PLAN QTY FIELD
                    else if (name.includes('plan_qty') || name.includes('planned_qty')) {
                        input.type = 'number'; input.step = '0.01';
                        planQtyInput = input;
                        input.addEventListener('input', calculateEfficiencyAndNC);
                    }
                    // AUTO-DETECT PRODUCED QTY FIELD
                    else if (name.includes('produced_qty') || name.includes('produced_quantity')) {
                        input.type = 'number';
                        producedQtyInput = input;
                        input.addEventListener('input', calculateEfficiencyAndNC);
                    }
                    // AUTO-DETECT ACCEPTED QTY FIELD
                    else if (name.includes('accepted_qty') || name.includes('accepted_quantity')) {
                        input.type = 'number';
                        acceptedQtyInput = input;
                        input.addEventListener('input', calculateEfficiencyAndNC);
                    }
                    else if (field.field_type === 'date') {
                        input.type = 'date';
                    } else {
                        input.type = 'text'; input.placeholder = "Enter " + field.field_label;
                    }
                }
                div.appendChild(input);
                container.appendChild(div);
            });
        } catch (err) {
            console.error(err);
            container.innerHTML = '<p class="loading-text" style="color:red">Error loading fields.</p>';
        }
    }

    // 2. DURATION CALC - When start/end time changes, recalculate duration and norms
    function calculateDuration() {
        if (startTimeInput && endTimeInput && durationInput) {
            const s = startTimeInput.value, e = endTimeInput.value;
            if (s && e) {
                const sMin = parseInt(s.split(':')[0]) * 60 + parseInt(s.split(':')[1]);
                const eMin = parseInt(e.split(':')[0]) * 60 + parseInt(e.split(':')[1]);
                let adjustedEndMin = eMin;
                if (adjustedEndMin < sMin) adjustedEndMin += 1440;
                let diff = adjustedEndMin - sMin;

                const lunchReduction = checkLunchPeriodAdjustment(sMin, adjustedEndMin);
                if (lunchReduction > 0) {
                    diff = Math.max(0, diff - lunchReduction);
                    durationInput.style.backgroundColor = '#fff3cd';
                    durationInput.style.borderColor = '#f39c12';
                    durationInput.title = 'Lunch overlap (1:00-1:30 PM) - 30 mins auto reduced';
                } else {
                    durationInput.style.backgroundColor = '';
                    durationInput.style.borderColor = '';
                    durationInput.title = '';
                }

                const hours = Math.floor(diff / 60);
                const minutes = diff % 60;
                durationInput.value = `${hours.toString().padStart(2, '0')}:${minutes.toString().padStart(2, '0')}`;

                // Recalculate norms using adjusted duration
                calculateTotalNorms(diff);
            }
        }
    }

    // 3. âœ… MASTER CALCULATION FUNCTION - Calculates norms based on Tool Rate Ã— Duration (in hours)
    function calculateTotalNorms(durationMinutes = null) {
        if (!normInput) return; // Exit if no "Norms" field found

        // A. Get Duration in minutes (if not passed)
        if (!durationMinutes && durationInput && durationInput.value) {
            const parts = durationInput.value.split(':');
            durationMinutes = (parseInt(parts[0]) * 60) + parseInt(parts[1]);
        }

        if (!durationMinutes || durationMinutes <= 0) {
            normInput.value = ""; // Clear if no duration
            return;
        }

        // B. Find TOOL field and get its selected norm rate
        let toolNormRate = 0;
        const dropdowns = document.querySelectorAll('.dropdown-field');

        dropdowns.forEach(dd => {
            const fieldName = (dd.name || '').toLowerCase();
            // Check if this is a TOOL field
            if (fieldName.includes('tool')) {
                const selectedOption = dd.options[dd.selectedIndex];
                if (selectedOption && selectedOption.dataset.norm) {
                    toolNormRate = parseFloat(selectedOption.dataset.norm) || 0;
                }
            }
        });

        // C. Calculate Total Norms: (Duration in Hours) Ã— (Tool Norm Rate)
        if (toolNormRate > 0) {
            const durationHours = durationMinutes / 60;
            const totalNorms = Math.round(durationHours * toolNormRate);
            normInput.value = totalNorms;
            console.log(`âœ… Tool Norm Rate: ${toolNormRate}/hr Ã— Duration: ${durationHours.toFixed(2)} hours = Total Norms: ${totalNorms}`);
        } else {
            normInput.value = ""; // Empty if tool not selected or no rate
        }

        // D. Trigger efficiency and NC calculation after norms update
        calculateEfficiencyAndNC();
    }

    // 4. LOAD OPTIONS (Updated to store Norm Value in HTML)
    async function loadOptions(el, id, pid = null) {
        let url = `api.php?action=get_options&field_id=${id}`;
        if (pid) url += `&parent_opt_id=${pid}`;
        try {
            const res = await fetch(url);
            const opts = await res.json();
            el.innerHTML = '<option value="">Select Option</option>';

            opts.forEach(o => {
                // STORE NORM VALUE HERE
                const normVal = o.norm_value || 0;
                el.innerHTML += `<option value="${o.label}" data-opt-id="${o.id}" data-norm="${normVal}">${o.label}</option>`;
            });
            el.disabled = false;
        } catch (e) { console.error(e); }
    }

    function triggerChildUpdates(parent) {
        const pid = parent.dataset.fieldId;
        const optId = parent.selectedOptions[0]?.dataset.optId;
        const fieldName = (parent.name || '').toLowerCase();
        
        document.querySelectorAll('.dropdown-field').forEach(child => {
            if (child.dataset.parentId === pid) {
                if (optId) {
                    child.innerHTML = '<option>Loading...</option>';
                    loadOptions(child, child.dataset.fieldId, optId);
                } else {
                    child.innerHTML = '<option value="">Select Parent First</option>';
                    child.disabled = true;
                }
                child.dispatchEvent(new Event('change'));
            }
        });
    }

    // 5. LOAD EMPLOYEES
    async function loadEmployees() {
        try {
            const res = await fetch('api.php?action=get_employees');
            const data = await res.json();
            const activeStaff = data.filter(e => (e.status || 'active').toLowerCase() !== 'inactive');
            activeStaff.sort((a,b) => a.name.localeCompare(b.name));

            const sel = document.getElementById('empSelect');
            sel.innerHTML = '<option value="">Select Employee</option>';
            window.empMap = {};
            activeStaff.forEach(e => {
                sel.innerHTML += `<option value="${e.emp_unique_id}">${e.name} (${e.emp_unique_id})</option>`;
                window.empMap[e.emp_unique_id] = e.name;
            });
            sel.addEventListener('change', function() {
                document.getElementById('operatorName').value = window.empMap[this.value] || '';
            });
        } catch (e) { console.error(e); }
    }

    // 6. SUBMIT (WITH VALIDATION)
    document.getElementById('productionForm').addEventListener('submit', async function(e) {
        e.preventDefault();
        const btn = this.querySelector('button');
        const msg = document.getElementById('statusMessage');
        
        // === REQUIRED FIELD VALIDATION ===
        const empSelect = document.getElementById('empSelect');
        const logDate = document.getElementById('logDate');
        
        if (!logDate.value) {
            showModal('âš  Required', 'Please select a Date', 'alert');
            logDate.focus();
            return;
        }

        if (!empSelect.value) {
            showModal('âš  Required', 'Please select an Employee ID', 'alert');
            empSelect.focus();
            return;
        }

        // === PRODUCTION QTY VALIDATION ===
        const planQty = planQtyInput ? parseFloat(planQtyInput.value) || 0 : 0;
        const producedQty = producedQtyInput ? parseFloat(producedQtyInput.value) || 0 : 0;
        const acceptedQty = acceptedQtyInput ? parseFloat(acceptedQtyInput.value) || 0 : 0;

        // Validation errors
        const errors = [];

        if (planQty <= 0) {
            errors.push('âœ— Plan Qty must be greater than 0');
        }
        if (producedQty < 0) {
            errors.push('âœ— Produced Qty cannot be negative');
        }
        if (acceptedQty < 0) {
            errors.push('âœ— Accepted Qty cannot be negative');
        }
        if (acceptedQty > producedQty) {
            errors.push('âœ— Accepted Qty cannot exceed Produced Qty');
        }

        // Show validation errors
        if (errors.length > 0) {
            showModal('âš  Validation Error', errors.join('<br>'), 'alert');
            return;
        }

        // Warning for overproduction
        if (producedQty > planQty) {
            const overage = ((producedQty / planQty) * 100).toFixed(2);
            showModal('â„¹ Overproduction Alert', 
                `Produced Qty (${producedQty}) exceeds Plan Qty (${planQty}).<br>` +
                `Efficiency: ${overage}% (Over-Target)<br><br>` +
                `Do you want to continue?`, 
                'confirm',
                (confirmed) => {
                    if (confirmed) {
                        submitForm(this, btn, msg);
                    }
                }
            );
        } else {
            submitForm(this, btn, msg);
        }
    });

    // Helper function to submit form after validation
    async function submitForm(form, btn, msg) {
        btn.disabled = true;
        btn.innerText = "Saving...";
        msg.style.display = 'none';

        const fd = new FormData(form);
        fd.append('action', 'save_log');
        
        // Extract tool_id from selected tool dropdown
        let toolId = null;
        const toolDropdowns = form.querySelectorAll('.dropdown-field');
        toolDropdowns.forEach(dd => {
            const fieldName = (dd.name || '').toLowerCase();
            if (fieldName.includes('tool') && dd.selectedIndex > 0) {
                toolId = dd.options[dd.selectedIndex]?.dataset.optId;
                if (toolId) {
                    fd.append('tool_id', toolId);
                    console.log('âœ“ Tool ID captured:', toolId);
                }
            }
        });
        
        // Get produced quantity value - ensure it's properly captured
        const producedQtyInput = document.querySelector('[name*="produced_qty"]');
        const producedQtyValue = producedQtyInput?.value;
        if (producedQtyValue && producedQtyValue > 0) {
            fd.append('produced_qty', producedQtyValue);
            console.log('âœ“ Produced Qty captured:', producedQtyValue);
        } else {
            console.warn('âš ï¸ Produced Qty not found or invalid:', producedQtyValue);
        }
        
        // Debug: Log all form data being sent
        console.log('=== FORM DATA BEING SENT ===');
        for (let [key, value] of fd.entries()) {
            console.log(`  ${key}: ${value}`);
        }

        try {
            const res = await fetch('api.php', { method: 'POST', body: fd });
            const data = await res.json();
            msg.style.display = 'block';
            
            console.log('=== API RESPONSE ===', data);
            
            if (data.success) {
                msg.className = 'status-msg status-success';
                msg.innerText = "âœ“ Entry Saved Successfully! Tool strokes updated automatically.";
                
                // Notify maintenance page to refresh immediately
                try {
                    localStorage.setItem('maintenance_refresh_trigger', new Date().toISOString());
                    console.log('âœ“ Maintenance page refresh triggered via localStorage');
                } catch(e) { 
                    console.error('localStorage error:', e);
                }
                
                form.reset();
                document.getElementById('logDate').value = new Date().toISOString().split('T')[0];
                document.getElementById('empSelect').value = '';
                document.getElementById('operatorName').value = '';
                generateDynamicForm();
            } else {
                msg.className = 'status-msg status-error';
                msg.innerText = "Error: " + (data.error || "Unknown Error");
                console.error('Save failed:', data.error);
            }
        } catch (err) {
            console.error('Network error:', err);
            msg.innerText = "Connection Failed: " + err.message;
            msg.className = 'status-msg status-error';
            msg.style.display = 'block';
        } finally {
            btn.disabled = false;
            btn.innerText = "Save Entry to Log";
        }
    }
</script>

</body>
</html>
