<?php
// Protect this page with session check
include 'session_check.php';

$pageFlashNotice = $_SESSION['access_denied_notice'] ?? '';
unset($_SESSION['access_denied_notice']);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Mektron Analytics Dashboard</title>
    <link rel="stylesheet" href="style.css">
    <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <style>
        /* Dashboard Specific Styles */
        body {
            background: linear-gradient(135deg, #f5f7fa 0%, #c3cfe2 100%);
            margin: 0;
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
        }

        .main-content {
            padding: 30px;
            min-height: 100vh;
        }

        .page-header {
            margin-bottom: 40px;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }

        .page-header h1 {
            margin: 0;
            color: #2e2626;
            font-size: 36px;
            display: flex;
            align-items: center;
            gap: 15px;
            text-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }

        /* KPI Cards with Mini Charts */
        .kpi-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(280px, 1fr));
            gap: 20px;
            margin-bottom: 40px;
        }

        .kpi-card {
            background: white;
            padding: 20px;
            border-radius: 15px;
            box-shadow: 0 10px 30px rgba(0,0,0,0.1);
            position: relative;
            overflow: hidden;
            transition: all 0.3s ease;
            border-top: 4px solid #013B3B;
        }

        .kpi-card:hover {
            transform: translateY(-8px);
            box-shadow: 0 12px 35px rgba(0,0,0,0.15);
        }

        .kpi-card.button-card {
            cursor: pointer;
            background: linear-gradient(135deg, rgba(255,255,255,1) 0%, rgba(245,247,250,1) 100%);
        }

        .kpi-card.button-card:hover {
            transform: translateY(-12px) scale(1.02);
            box-shadow: 0 16px 45px rgba(0,0,0,0.2);
            background: linear-gradient(135deg, rgba(255,255,255,1) 0%, rgba(235,240,250,1) 100%);
        }

        .kpi-card.blue { border-top-color: #3498db; }
        .kpi-card.green { border-top-color: #2ecc71; }
        .kpi-card.orange { border-top-color: #f39c12; }
        .kpi-card.red { border-top-color: #e74c3c; }

        .kpi-header {
            display: flex;
            justify-content: space-between;
            align-items: start;
            margin-bottom: 15px;
        }

        .kpi-label {
            font-size: 12px;
            color: #7f8c8d;
            text-transform: uppercase;
            font-weight: 700;
            letter-spacing: 1px;
        }

        .kpi-icon {
            font-size: 28px;
            opacity: 0.3;
        }

        .kpi-value {
            font-size: 36px;
            font-weight: 900;
            color: #2c3e50;
            margin-bottom: 12px;
            display: flex;
            align-items: baseline;
            gap: 8px;
        }

        .kpi-subtext {
            font-size: 12px;
            color: #95a5a6;
            margin-bottom: 15px;
        }

        .mini-chart {
            width: 100%;
            min-height: 50px;
            max-height: 80px;
        }

        /* Field Filter */
        .filter-section {
            background: white;
            padding: 20px;
            border-radius: 12px;
            margin-bottom: 30px;
            box-shadow: 0 4px 15px rgba(0,0,0,0.05);
        }

        .filter-select {
            width: 100%;
            max-width: 500px;
            padding: 12px 15px;
            border: 2px solid #ddd;
            border-radius: 8px;
            font-size: 14px;
            cursor: pointer;
            transition: 0.3s;
        }

        .filter-select:focus {
            outline: none;
            border-color: #013B3B;
            box-shadow: 0 0 0 3px rgba(1, 59, 59, 0.1);
        }

        /* Dashboard Grid */
        .dashboard-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 25px;
            margin-bottom: 30px;
        }

        .stat-card {
            background: white;
            padding: 25px;
            border-radius: 12px;
            box-shadow: 0 4px 15px rgba(0,0,0,0.05);
            border-left: 5px solid #ddd;
            transition: all 0.3s;
        }
        .stat-card:hover { 
            transform: translateY(-8px);
            box-shadow: 0 8px 25px rgba(0,0,0,0.1);
        }
        .stat-card h3 { margin: 0; font-size: 13px; color: #7f8c8d; text-transform: uppercase; letter-spacing: 1px; }
        .stat-card .value { font-size: 32px; font-weight: 800; color: #2c3e50; margin-top: 10px; }
        .stat-card .icon { float: right; font-size: 30px; opacity: 0.3; }
        .stat-card .subtext { font-size: 12px; color: #95a5a6; margin-top: 8px; }

        /* Controls Section */
        .controls-section {
            background: #f9f9f9;
            padding: 20px;
            border-radius: 8px;
            margin-bottom: 30px;
            border-left: 5px solid #013B3B;
        }

        .control-item label {
            display: block;
            font-weight: 600;
            color: #555;
            margin-bottom: 8px;
            font-size: 13px;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }

        /* Charts Section */
        .charts-container {
            display: grid;
            grid-template-columns: 1fr;
            gap: 30px;
            margin-bottom: 40px;
        }

        .chart-box {
            background: white;
            padding: 30px;
            border-radius: 15px;
            box-shadow: 0 10px 30px rgba(0,0,0,0.1);
            border-top: 4px solid #013B3B;
        }

        .chart-wrapper {
            position: relative;
            width: 100%;
            padding-bottom: 56.25%;
            height: 0;
            overflow: hidden;
        }

        .chart-wrapper canvas {
            position: absolute;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
        }

        .chart-header {
            font-size: 18px;
            font-weight: 700;
            color: #2c3e50;
            margin-bottom: 25px;
            display: flex;
            align-items: center;
            gap: 12px;
        }

        .chart-header i { font-size: 22px; }

        /* Analysis Charts Section */
        .analysis-charts {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(400px, 1fr));
            gap: 25px;
            margin-bottom: 30px;
        }

        .analysis-chart-box {
            background: white;
            padding: 25px;
            border-radius: 15px;
            box-shadow: 0 10px 30px rgba(0,0,0,0.1);
            border-top: 4px solid #013B3B;
        }

        .analysis-chart-box .chart-header {
            font-size: 16px;
            margin-bottom: 20px;
        }

        .analysis-chart-wrapper {
            position: relative;
            width: 100%;
            height: 300px;
        }

        .drilldown-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(420px, 1fr));
            gap: 25px;
            margin-bottom: 30px;
        }

        .drilldown-box {
            background: white;
            padding: 24px;
            border-radius: 15px;
            box-shadow: 0 10px 30px rgba(0,0,0,0.1);
            border-top: 4px solid #0ea5e9;
        }

        .drilldown-box.tool {
            border-top-color: #10b981;
        }

        .drilldown-controls {
            margin-bottom: 15px;
        }

        .drilldown-select {
            width: 100%;
            padding: 10px 12px;
            border: 1px solid #d1d5db;
            border-radius: 8px;
            font-size: 14px;
            color: #1f2937;
            background: #fff;
        }

        .drilldown-select:focus {
            outline: none;
            border-color: #0ea5e9;
            box-shadow: 0 0 0 3px rgba(14, 165, 233, 0.16);
        }

        .drilldown-summary-grid {
            display: grid;
            grid-template-columns: repeat(2, minmax(0, 1fr));
            gap: 12px;
            margin-bottom: 16px;
        }

        .drilldown-summary-card {
            border: 1px solid #e5e7eb;
            border-left: 4px solid #0ea5e9;
            background: #f9fafb;
            border-radius: 10px;
            padding: 10px 12px;
        }

        .drilldown-summary-card .label {
            font-size: 11px;
            color: #6b7280;
            text-transform: uppercase;
            letter-spacing: 0.5px;
            font-weight: 700;
        }

        .drilldown-summary-card .value {
            margin-top: 4px;
            font-size: 22px;
            font-weight: 800;
            color: #111827;
            line-height: 1.15;
        }

        .drilldown-summary-card .sub {
            margin-top: 2px;
            font-size: 11px;
            color: #6b7280;
        }

        .drilldown-chart-grid {
            display: grid;
            grid-template-columns: repeat(2, minmax(0, 1fr));
            gap: 12px;
        }

        .drilldown-chart-card {
            border: 1px solid #e5e7eb;
            border-radius: 10px;
            padding: 10px;
            background: #ffffff;
            min-height: 260px;
        }

        .drilldown-chart-card.full {
            grid-column: 1 / -1;
        }

        .drilldown-chart-card h4 {
            margin: 0 0 8px 0;
            font-size: 12px;
            font-weight: 700;
            color: #374151;
            text-transform: uppercase;
            letter-spacing: 0.4px;
        }

        .drilldown-canvas-wrap {
            position: relative;
            height: 210px;
        }

        .drilldown-empty {
            text-align: center;
            color: #9ca3af;
            font-size: 13px;
            padding: 30px 10px;
        }

        /* Field Cards Section */
        .fields-section {
            background: white;
            padding: 20px;
            border-radius: 15px;
            box-shadow: 0 10px 30px rgba(0,0,0,0.1);
            margin-bottom: 30px;
        }

        .section-title {
            font-size: 16px;
            font-weight: 700;
            color: #2c3e50;
            margin: 0 0 15px 0;
            display: flex;
            align-items: center;
            gap: 10px;
        }

        /* Production Status Table */
        .production-table {
            width: 100%;
            border-collapse: collapse;
            margin-top: 10px;
        }

        .production-table thead {
            background: linear-gradient(135deg, #3498db 0%, #2980b9 100%);
        }

        .production-table th {
            padding: 12px 15px;
            text-align: left;
            font-weight: 700;
            color: white;
            font-size: 12px;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }

        .production-table tbody tr {
            border-bottom: 1px solid #eee;
            transition: 0.2s;
        }

        .production-table tbody tr:hover {
            background: #f8f9fa;
        }

        .production-table td {
            padding: 12px 15px;
            color: #555;
            font-size: 13px;
        }

        .status-badge {
            display: inline-block;
            padding: 4px 10px;
            border-radius: 12px;
            font-size: 11px;
            font-weight: 600;
            text-transform: uppercase;
        }

        .status-badge.active {
            background: #d4edda;
            color: #155724;
        }

        .status-badge.pending {
            background: #fff3cd;
            color: #856404;
        }

        .status-badge.completed {
            background: #cfe2ff;
            color: #084298;
        }

        .fields-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(160px, 1fr));
            gap: 15px;
        }

        .field-card {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 15px;
            border-radius: 12px;
            text-align: center;
            cursor: pointer;
            transition: all 0.3s;
            box-shadow: 0 8px 20px rgba(0,0,0,0.15);
            position: relative;
            overflow: hidden;
        }

        .field-card::before {
            content: '';
            position: absolute;
            top: 0;
            left: -100%;
            width: 100%;
            height: 100%;
            background: rgba(255,255,255,0.1);
            transition: left 0.3s;
        }

        .field-card:hover::before {
            left: 100%;
        }

        .field-card:hover {
            transform: translateY(-8px);
            box-shadow: 0 12px 30px rgba(0,0,0,0.25);
        }

        .field-card .field-name {
            font-size: 12px;
            font-weight: 700;
            margin-bottom: 8px;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }

        .field-card .field-count {
            font-size: 28px;
            font-weight: 900;
            margin-bottom: 5px;
        }

        .field-card .field-type {
            font-size: 10px;
            opacity: 0.9;
        }

        /* Data Table */
        .data-table {
            width: 100%;
            border-collapse: collapse;
            margin-top: 15px;
        }

        .data-table thead {
            background: #f8f9fa;
            border-bottom: 2px solid #ddd;
        }

        .data-table th {
            padding: 15px;
            text-align: left;
            font-weight: 600;
            color: #2c3e50;
            font-size: 13px;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }

        .data-table td {
            padding: 15px;
            border-bottom: 1px solid #eee;
            color: #555;
        }

        .data-table tbody tr:hover {
            background: #f8f9fa;
        }

        .badge {
            display: inline-block;
            padding: 5px 12px;
            background: #e3f2fd;
            color: #1976d2;
            border-radius: 20px;
            font-size: 12px;
            font-weight: 600;
        }

        /* Responsive */
        @media (max-width: 1200px) {
            .kpi-grid {
                grid-template-columns: repeat(2, 1fr);
            }
        }

        @media (max-width: 768px) {
            .main-content {
                padding: 15px;
            }
            .page-header {
                flex-direction: column;
                gap: 15px;
                margin-bottom: 30px;
                align-items: flex-start;
            }
            .page-header h1 {
                font-size: 24px;
            }
            .kpi-grid {
                grid-template-columns: 1fr;
                gap: 15px;
                margin-bottom: 30px;
            }
            .kpi-card {
                padding: 20px;
            }
            .kpi-value {
                font-size: 28px;
            }
            .fields-grid {
                grid-template-columns: 1fr;
            }
            .filter-select {
                max-width: 100%;
            }
            .drilldown-grid {
                grid-template-columns: 1fr;
            }
            .drilldown-summary-grid {
                grid-template-columns: 1fr;
            }
            .drilldown-chart-grid {
                grid-template-columns: 1fr;
            }
        }

        .loading-spinner {
            text-align: center;
            padding: 40px;
            color: #999;
        }

        .spinner {
            border: 4px solid #f3f3f3;
            border-top: 4px solid #013B3B;
            border-radius: 50%;
            width: 40px;
            height: 40px;
            animation: spin 1s linear infinite;
            margin: 0 auto 20px;
        }

        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }

        /* Notification Button Styling */
        .notification-btn {
            position: relative;
            background: none;
            border: none;
            font-size: 20px;
            cursor: pointer;
            color: #333;
            padding: 8px 12px;
            border-radius: 4px;
            transition: all 0.3s;
        }

        .notification-btn:hover {
            background: rgba(0,0,0,0.05);
            color: #e74c3c;
            transform: scale(1.1);
        }

        .notification-badge {
            position: absolute;
            top: 0;
            right: 0;
            background: #e74c3c;
            color: white;
            border-radius: 50%;
            width: 24px;
            height: 24px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 12px;
            font-weight: bold;
            animation: pulse-badge 2s infinite;
        }

        @keyframes pulse-badge {
            0%, 100% { transform: scale(1); }
            50% { transform: scale(1.1); }
        }
    </style>
</head>
<body>

    <div class="main-content">
            <!-- Page Header -->
            <div class="page-header">
                <h1><i class="fas fa-chart-line"></i> Analytics Dashboard</h1>
                <!-- Notification Button (Visible for all logged-in users) -->
                <?php if (isset($_SESSION) && !empty($_SESSION)) { ?>
                <button class="notification-btn" id="notificationBtn" title="View Notifications" onclick="toggleNotificationPopup()">
                    <i class="fas fa-bell"></i>
                </button>
                <?php } ?>
            </div>

            <!-- Filter Section -->

            <!-- KPI Cards with Mini Charts -->
            <div class="kpi-grid">
                <div class="kpi-card blue">
                    <div class="kpi-header">
                        <div class="kpi-label">Total Staff</div>
                        <div class="kpi-icon">👥</div>
                    </div>
                    <div class="kpi-value"><span id="dispStaff">...</span></div>
                    <div class="kpi-subtext">Active Employees</div>
                    <canvas id="miniChart1" class="mini-chart"></canvas>
                </div>

                <div class="kpi-card green">
                    <div class="kpi-header">
                        <div class="kpi-label">Total Production</div>
                        <div class="kpi-icon">⚙️</div>
                    </div>
                    <div class="kpi-value"><span id="dispProd">...</span></div>
                    <div class="kpi-subtext">Units Produced</div>
                    <canvas id="miniChart2" class="mini-chart"></canvas>
                </div>

                <div class="kpi-card orange button-card" onclick="window.location.href='register_emp.php'">
                    <div class="kpi-header">
                        <div class="kpi-label">Employees Registered</div>
                        <div class="kpi-icon">👤</div>
                    </div>
                    <div class="kpi-value"><span id="dispRegister">...</span></div>
                    <div class="kpi-subtext">Click to Register</div>
                    <canvas id="miniChart3" class="mini-chart"></canvas>
                </div>

                <div class="kpi-card red button-card" onclick="window.location.href='entry_log.php'">
                    <div class="kpi-header">
                        <div class="kpi-label">Quick Action</div>
                        <div class="kpi-icon">✏️</div>
                    </div>
                    <div class="kpi-value" style="color: #e74c3c;">+ New</div>
                    <div class="kpi-subtext">Add Entry</div>
                    <canvas id="miniChart4" class="mini-chart"></canvas>
                </div>
            </div>

            <!-- Combined Chart Section -->
            <div class="chart-box">
                <div class="chart-header">
                    <i class="fas fa-chart-area" style="color: #2ecc71;"></i>
                    Combined Analytics: Optimization Trend & Performance
                </div>
                <div class="chart-wrapper">
                    <canvas id="combinedChart"></canvas>
                </div>
            </div>

            <!-- Analysis Section: Field-wise Analysis & Duration -->
            <div class="analysis-charts">
                <div class="analysis-chart-box">
                    <div class="chart-header">
                        <i class="fas fa-layer-group" style="color: #3498db;"></i>
                        Field-wise Analysis Report
                    </div>
                    <div class="report-box" id="fieldAnalysisReport">
                        <div style="text-align: center; padding: 30px; color: #999;">Loading field analysis...</div>
                    </div>
                </div>

                <div class="analysis-chart-box">
                    <div class="chart-header">
                        <i class="fas fa-clock" style="color: #9b59b6;"></i>
                        Employee Working Duration
                    </div>
                    <div class="analysis-chart-wrapper">
                        <canvas id="durationChart"></canvas>
                    </div>
                </div>
            </div>

            <div class="drilldown-grid">
                <div class="drilldown-box machine">
                    <div class="chart-header">
                        <i class="fas fa-cogs" style="color: #0ea5e9;"></i>
                        Machine Details
                    </div>
                    <div class="drilldown-controls">
                        <select id="machineSelect" class="drilldown-select">
                            <option value="">Select Machine</option>
                        </select>
                    </div>
                    <div id="machineSummaryCards" class="drilldown-summary-grid">
                        <div class="drilldown-empty" style="grid-column:1/-1;">Choose a machine to view details</div>
                    </div>
                    <div class="drilldown-chart-grid">
                        <div class="drilldown-chart-card full">
                            <h4>Daily Trend (Production & Hours)</h4>
                            <div class="drilldown-canvas-wrap"><canvas id="machineLineChart"></canvas></div>
                        </div>
                        <div class="drilldown-chart-card">
                            <h4>Top Tools by Production</h4>
                            <div class="drilldown-canvas-wrap"><canvas id="machineBarChart"></canvas></div>
                        </div>
                        <div class="drilldown-chart-card">
                            <h4>Acceptance vs NC</h4>
                            <div class="drilldown-canvas-wrap"><canvas id="machinePieChart"></canvas></div>
                        </div>
                    </div>
                </div>

                <div class="drilldown-box tool">
                    <div class="chart-header">
                        <i class="fas fa-tools" style="color: #10b981;"></i>
                        Tool Details
                    </div>
                    <div class="drilldown-controls">
                        <select id="toolSelect" class="drilldown-select">
                            <option value="">Select Tool</option>
                        </select>
                    </div>
                    <div id="toolSummaryCards" class="drilldown-summary-grid">
                        <div class="drilldown-empty" style="grid-column:1/-1;">Choose a tool to view details</div>
                    </div>
                    <div class="drilldown-chart-grid">
                        <div class="drilldown-chart-card">
                            <h4>Plan / Produced / Accepted / NC</h4>
                            <div class="drilldown-canvas-wrap"><canvas id="toolBarChart"></canvas></div>
                        </div>
                        <div class="drilldown-chart-card">
                            <h4>Acceptance vs NC</h4>
                            <div class="drilldown-canvas-wrap"><canvas id="toolPieChart"></canvas></div>
                        </div>
                        <div class="drilldown-chart-card full">
                            <h4>Worker Output in Selected Tool</h4>
                            <div class="drilldown-canvas-wrap"><canvas id="toolLineChart"></canvas></div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Employee Details Table -->
            <div class="fields-section">
                <div class="section-title">
                    <i class="fas fa-users" style="color: #013B3B;"></i>
                    Employee Details
                </div>
                <table class="production-table">
                    <thead>
                        <tr>
                            <th>Employee ID</th>
                            <th>Employee Name</th>
                            <th>Designation</th>
                            <th>Status</th>
                        </tr>
                    </thead>
                    <tbody id="employeeDetailsBody">
                        <tr>
                            <td colspan="4" style="text-align: center; padding: 30px; color: #999;">Loading employee details...</td>
                        </tr>
                    </tbody>
                </table>
            </div>

    </div>

    <script src="sidebar.js"></script>
    
    <script>
        let miniCharts = {};
        let combinedChart = null;
        let durationChart = null;
        let machineBarChart = null;
        let machineLineChart = null;
        let machinePieChart = null;
        let toolBarChart = null;
        let toolPieChart = null;
        let toolLineChart = null;

        const dashboardState = {
            machineField: '',
            machineList: [],
            toolList: []
        };

        document.addEventListener("DOMContentLoaded", function() {
            loadDashboardData();
            loadFieldAnalysisReport();
            loadEmployeeDetails();
            initMachineToolDrilldown();
        });

        function safeNumber(value) {
            const num = Number(value);
            return Number.isFinite(num) ? num : 0;
        }

        function formatNumber(value, digits = 0) {
            const num = safeNumber(value);
            return num.toLocaleString(undefined, {
                minimumFractionDigits: digits,
                maximumFractionDigits: digits
            });
        }

        function fetchJson(url) {
            return fetch(url).then(res => res.json());
        }

        // ===== LOAD FIELD-WISE ANALYSIS REPORT (REAL TOTALS) =====
        async function loadFieldAnalysisReport() {
            try {
                const [analyticsData, daywiseData] = await Promise.all([
                    fetchJson('api.php?action=get_analytics_dashboard'),
                    fetchJson('api.php?action=get_daywise_analytics')
                ]);

                const dayRows = Array.isArray(daywiseData) ? daywiseData : [];
                const totalProduced = dayRows.length > 0
                    ? dayRows.reduce((sum, row) => sum + safeNumber(row.total_produced_qty), 0)
                    : safeNumber(analyticsData.total_production_qty || analyticsData.total_quantity || analyticsData.total_records);
                const totalAccepted = dayRows.reduce((sum, row) => sum + safeNumber(row.total_accepted_qty), 0);
                const totalNC = dayRows.reduce((sum, row) => sum + safeNumber(row.nc_qty), 0);
                const totalHours = safeNumber(analyticsData.total_hours);
                const acceptanceRate = totalProduced > 0 ? (totalAccepted / totalProduced) * 100 : 0;

                const stats = [
                    {
                        label: 'Total Production',
                        value: formatNumber(totalProduced),
                        desc: 'Units Produced',
                        color: '#3498db',
                        icon: 'fa-industry'
                    },
                    {
                        label: 'Machine Total Hours',
                        value: formatNumber(totalHours, 2),
                        desc: 'Total Running Hours',
                        color: '#8b5cf6',
                        icon: 'fa-clock'
                    },
                    {
                        label: 'Total NC Qty',
                        value: formatNumber(totalNC),
                        desc: 'Rejected Quantity',
                        color: '#ef4444',
                        icon: 'fa-triangle-exclamation'
                    },
                    {
                        label: 'Acceptance Rate',
                        value: `${acceptanceRate.toFixed(1)}%`,
                        desc: `Accepted: ${formatNumber(totalAccepted)} units`,
                        color: '#10b981',
                        icon: 'fa-circle-check'
                    }
                ];

                let html = '<div class="field-analysis-grid" style="display:grid; grid-template-columns:repeat(auto-fit, minmax(220px, 1fr)); gap:20px;">';
                stats.forEach(stat => {
                    html += `
                        <div style="background:#fff; padding:20px; border-radius:12px; border-bottom:4px solid ${stat.color}; box-shadow:0 4px 15px rgba(0,0,0,0.05); display:flex; align-items:center; justify-content:space-between;">
                            <div>
                                <div style="font-size:13px; color:#7f8c8d; font-weight:700; text-transform:uppercase; letter-spacing:0.5px; margin-bottom:5px;">${stat.label}</div>
                                <div style="font-size:28px; font-weight:800; color:${stat.color};">${stat.value}</div>
                                <div style="font-size:12px; color:#aaa; margin-top:2px;">${stat.desc}</div>
                            </div>
                            <div style="background:${stat.color}15; width:50px; height:50px; border-radius:50%; display:flex; align-items:center; justify-content:center;">
                                <i class="fas ${stat.icon}" style="font-size:22px; color:${stat.color};"></i>
                            </div>
                        </div>
                    `;
                });
                html += '</div>';
                document.getElementById('fieldAnalysisReport').innerHTML = html;
            } catch (err) {
                console.error("Analysis Error:", err);
                document.getElementById('fieldAnalysisReport').innerHTML =
                    '<div style="text-align:center; padding:20px; color:#e74c3c;"><i class="fas fa-times-circle"></i> Failed to load analysis data</div>';
            }
        }

        // ===== LOAD EMPLOYEE DETAILS =====
        function loadEmployeeDetails() {
            fetch('api.php?action=get_employees')
            .then(res => res.json())
            .then(data => {
                if (data && Array.isArray(data) && data.length > 0) {
                    let html = '';
                    
                    data.forEach((employee, idx) => {
                        const empId = employee.emp_unique_id || employee.id || 'N/A';
                        const empName = employee.name || 'Unknown';
                        const designation = employee.designation || 'Operator';
                        const isActive = String(employee.status || '').toLowerCase() === 'active' || employee.status === 1;
                        const status = isActive ? 'active' : 'pending';
                        const statusText = isActive ? 'Active' : 'Inactive';
                        
                        html += `
                            <tr>
                                <td><strong>${empId}</strong></td>
                                <td>${empName}</td>
                                <td>${designation}</td>
                                <td><span class="status-badge ${status}">${statusText}</span></td>
                            </tr>
                        `;
                    });

                    document.getElementById('employeeDetailsBody').innerHTML = html;
                } else {
                    document.getElementById('employeeDetailsBody').innerHTML = '<tr><td colspan="4" style="text-align: center; padding: 20px; color: #999;">No employee data available</td></tr>';
                }
            })
            .catch(err => {
                console.error("Employee Details Error:", err);
                document.getElementById('employeeDetailsBody').innerHTML = '<tr><td colspan="4" style="text-align: center; padding: 20px; color: #e74c3c;">Error loading employee details</td></tr>';
            });
        }

        // ===== LOAD MAIN DASHBOARD DATA =====
        async function loadDashboardData() {
            try {
                const [statsData, analyticsData] = await Promise.all([
                    fetchJson('api.php?action=get_dashboard_stats'),
                    fetchJson('api.php?action=get_analytics_dashboard')
                ]);

                const dateStats = Array.isArray(analyticsData.date_stats) ? analyticsData.date_stats : [];
                document.getElementById('dispStaff').textContent = formatNumber(statsData.staff_count || 0);
                document.getElementById('dispProd').textContent = formatNumber(analyticsData.total_production_qty || analyticsData.total_quantity || 0);
                document.getElementById('dispRegister').textContent = formatNumber(statsData.staff_count || 0);

                renderMiniChart(1, dateStats);
                renderMiniChart(2, dateStats);
                renderMiniChart(3, dateStats);
                renderMiniChart(4, dateStats);
                renderCombinedChart(dateStats);
                renderDurationChart(dateStats);
            } catch (err) {
                console.error("Dashboard Error:", err);
                Swal.fire({icon: 'error', title: 'Error', text: 'Failed to load dashboard data', confirmButtonColor: '#dc3545'});
            }
        }

        // ===== RENDER MINI CHART =====
        function renderMiniChart(chartNum, data) {
            const canvasId = `miniChart${chartNum}`;
            const canvas = document.getElementById(canvasId);
            if (!canvas) return;
            
            // Destroy existing chart if any
            if (miniCharts[chartNum]) {
                miniCharts[chartNum].destroy();
            }

            const ctx = canvas.getContext('2d');
            const values = (data || []).map(item => safeNumber(item.total_quantity || item.total_qty || item.count)).slice(-7);
            const finalValues = values.length > 0 ? values : [0];
            
            const colors = ['#3498db', '#2ecc71', '#f39c12', '#e74c3c'];

            miniCharts[chartNum] = new Chart(ctx, {
                type: 'line',
                data: {
                    labels: finalValues.map((_, i) => i + 1),
                    datasets: [{
                        data: finalValues,
                        borderColor: colors[chartNum - 1],
                        backgroundColor: `rgba(${chartNum === 1 ? '52,152,219' : chartNum === 2 ? '46,204,113' : chartNum === 3 ? '243,156,18' : '231,76,60'}, 0.1)`,
                        borderWidth: 2,
                        fill: true,
                        tension: 0.4,
                        pointRadius: 0,
                        pointHoverRadius: 4
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: { legend: { display: false } },
                    scales: {
                        y: { display: false, beginAtZero: true },
                        x: { display: false }
                    }
                }
            });
        }

        // ===== RENDER COMBINED CHART (Bar + Line) =====
        function renderCombinedChart(dateStats) {
            const canvas = document.getElementById('combinedChart');
            if (!canvas) return;
            if (combinedChart) combinedChart.destroy();
            
            const ctx = canvas.getContext('2d');
            const labels = dateStats.map(item => item.log_date || 'N/A');
            const barValues = dateStats.map(item => safeNumber(item.total_quantity || item.total_qty || item.count));
            const lineValues = dateStats.map(item => safeNumber(item.total_hours));

            combinedChart = new Chart(ctx, {
                type: 'bar',
                data: {
                    labels: labels,
                    datasets: [
                        {
                            type: 'bar',
                            label: 'Daily Production',
                            data: barValues,
                            backgroundColor: 'rgba(52, 152, 219, 0.6)',
                            borderColor: '#3498db',
                            borderWidth: 2,
                            borderRadius: 6,
                            order: 2,
                            yAxisID: 'y'
                        },
                        {
                            type: 'line',
                            label: 'Total Hours',
                            data: lineValues,
                            borderColor: '#2ecc71',
                            backgroundColor: 'rgba(46, 204, 113, 0.1)',
                            borderWidth: 3,
                            fill: true,
                            tension: 0.35,
                            pointRadius: 4,
                            pointBackgroundColor: '#2ecc71',
                            pointBorderColor: '#fff',
                            pointBorderWidth: 2,
                            pointHoverRadius: 6,
                            order: 1,
                            yAxisID: 'y1'
                        }
                    ]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    interaction: {
                        mode: 'index',
                        intersect: false
                    },
                    plugins: {
                        legend: {
                            display: true,
                            position: 'top',
                            labels: {
                                padding: 20,
                                font: { size: 13, weight: 'bold' },
                                color: '#2c3e50'
                            }
                        },
                        tooltip: {
                            backgroundColor: 'rgba(0,0,0,0.8)',
                            padding: 15,
                            borderRadius: 8,
                            titleFont: { size: 14, weight: 'bold' },
                            bodyFont: { size: 12 }
                        }
                    },
                    scales: {
                        y: {
                            type: 'linear',
                            display: true,
                            position: 'left',
                            title: {
                                display: true,
                                text: 'Production (Units)',
                                font: { weight: 'bold' }
                            },
                            ticks: { color: '#666' },
                            grid: { color: 'rgba(0,0,0,0.05)' }
                        },
                        y1: {
                            type: 'linear',
                            display: true,
                            position: 'right',
                            title: {
                                display: true,
                                text: 'Hours',
                                font: { weight: 'bold' }
                            },
                            ticks: { color: '#2ecc71' },
                            grid: { display: false }
                        },
                        x: {
                            ticks: { color: '#666' },
                            grid: { color: 'rgba(0,0,0,0.05)' }
                        }
                    }
                }
            });
        }

        // ===== RENDER EMPLOYEE WORKING DURATION CHART =====
        function renderDurationChart(dateStats) {
            const canvas = document.getElementById('durationChart');
            if (!canvas) return;
            if (durationChart) durationChart.destroy();
            
            const ctx = canvas.getContext('2d');
            
            const labels = dateStats.map(item => item.log_date || 'N/A');
            const workingHours = dateStats.map(item => safeNumber(item.total_hours));

            durationChart = new Chart(ctx, {
                type: 'line',
                data: {
                    labels: labels,
                    datasets: [{
                        label: 'Working Hours',
                        data: workingHours,
                        borderColor: '#9b59b6',
                        backgroundColor: 'rgba(155, 89, 182, 0.1)',
                        borderWidth: 3,
                        fill: true,
                        tension: 0.4,
                        pointRadius: 5,
                        pointBackgroundColor: '#9b59b6',
                        pointBorderColor: '#fff',
                        pointBorderWidth: 2,
                        pointHoverRadius: 7
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: { display: false },
                        tooltip: {
                            backgroundColor: 'rgba(0,0,0,0.8)',
                            padding: 12,
                            borderRadius: 6,
                            callbacks: {
                                label: function(context) {
                                    return 'Duration: ' + context.parsed.y + ' hours';
                                }
                            }
                        }
                    },
                    scales: {
                        y: {
                            beginAtZero: true,
                            ticks: { color: '#666' },
                            grid: { color: 'rgba(0,0,0,0.05)' }
                        },
                        x: { ticks: { color: '#666' }, grid: { display: false } }
                    }
                }
            });
        }

        // ===== MACHINE + TOOL DRILLDOWN =====
        async function initMachineToolDrilldown() {
            await loadMachineDrilldown();
            await loadToolDrilldown();

            const machineSelect = document.getElementById('machineSelect');
            const toolSelect = document.getElementById('toolSelect');

            if (machineSelect) {
                machineSelect.addEventListener('change', () => renderMachineDetails(machineSelect.value));
            }
            if (toolSelect) {
                toolSelect.addEventListener('change', () => renderToolDetails(toolSelect.value));
            }
        }

        async function detectMachineField() {
            try {
                const allFieldsRes = await fetchJson('api.php?action=get_all_fields');
                const fields = Array.isArray(allFieldsRes.all_fields) ? allFieldsRes.all_fields : [];
                const machineField = fields.find(f => {
                    const name = String(f.field_name || '').toLowerCase();
                    const label = String(f.field_label || '').toLowerCase();
                    return name.includes('machine') || name.includes('mc') || label.includes('machine') || label.includes('m/c');
                });
                return machineField ? machineField.field_name : '';
            } catch (error) {
                console.error('Machine field detection failed:', error);
                return '';
            }
        }

        async function loadMachineDrilldown() {
            const machineSelect = document.getElementById('machineSelect');
            const summaryContainer = document.getElementById('machineSummaryCards');
            if (!machineSelect || !summaryContainer) return;
            try {
                dashboardState.machineField = await detectMachineField();
                if (!dashboardState.machineField) {
                    machineSelect.innerHTML = '<option value="">Machine field not found</option>';
                    machineSelect.disabled = true;
                    summaryContainer.innerHTML = '<div class="drilldown-empty" style="grid-column:1/-1;">Machine field is not configured in custom fields.</div>';
                    return;
                }

                const machineRes = await fetchJson(`api.php?action=get_machine_summary&machine_field=${encodeURIComponent(dashboardState.machineField)}`);
                const machines = machineRes && machineRes.success && Array.isArray(machineRes.machines) ? machineRes.machines : [];
                dashboardState.machineList = machines;

                if (machines.length === 0) {
                    machineSelect.innerHTML = '<option value="">No machine data</option>';
                    machineSelect.disabled = true;
                    summaryContainer.innerHTML = '<div class="drilldown-empty" style="grid-column:1/-1;">No machine records found.</div>';
                    return;
                }

                machineSelect.disabled = false;
                machineSelect.innerHTML = '<option value="">Select Machine</option>' + machines.map(m =>
                    `<option value="${m.machine_value}">${m.machine_name}</option>`
                ).join('');

                machineSelect.value = machines[0].machine_value;
                await renderMachineDetails(machineSelect.value);
            } catch (error) {
                console.error('Machine drilldown load failed:', error);
                machineSelect.innerHTML = '<option value="">Error loading machines</option>';
                machineSelect.disabled = true;
                summaryContainer.innerHTML = '<div class="drilldown-empty" style="grid-column:1/-1;">Failed to load machine data.</div>';
            }
        }

        async function renderMachineDetails(machineValue) {
            const summaryContainer = document.getElementById('machineSummaryCards');
            if (!summaryContainer || !machineValue || !dashboardState.machineField) {
                summaryContainer.innerHTML = '<div class="drilldown-empty" style="grid-column:1/-1;">Choose a machine to view details</div>';
                return;
            }
            try {
                const detailRes = await fetchJson(`api.php?action=get_machine_detail_analysis&machine_field=${encodeURIComponent(dashboardState.machineField)}&machine_name=${encodeURIComponent(machineValue)}`);
                if (!detailRes || !detailRes.success) {
                    summaryContainer.innerHTML = '<div class="drilldown-empty" style="grid-column:1/-1;">Failed to load machine details</div>';
                    return;
                }

                const summary = detailRes.summary || {};
                const totalProduction = safeNumber(summary.total_production_qty);
                const totalHours = safeNumber(summary.total_hours);
                const totalAccepted = safeNumber(summary.total_accepted_qty);
                const totalNc = safeNumber(summary.total_nc_qty);
                const acceptanceRate = safeNumber(summary.acceptance_rate);

                summaryContainer.innerHTML = `
                    <div class="drilldown-summary-card"><div class="label">Total Production</div><div class="value">${formatNumber(totalProduction)}</div><div class="sub">units</div></div>
                    <div class="drilldown-summary-card"><div class="label">Total Hours</div><div class="value">${formatNumber(totalHours, 2)}</div><div class="sub">hours</div></div>
                    <div class="drilldown-summary-card"><div class="label">Accepted Qty</div><div class="value">${formatNumber(totalAccepted)}</div><div class="sub">accepted units</div></div>
                    <div class="drilldown-summary-card"><div class="label">NC Qty</div><div class="value">${formatNumber(totalNc)}</div><div class="sub">rejected units</div></div>
                    <div class="drilldown-summary-card"><div class="label">Acceptance Rate</div><div class="value">${acceptanceRate.toFixed(2)}%</div><div class="sub">accepted vs produced</div></div>
                    <div class="drilldown-summary-card"><div class="label">Operations</div><div class="value">${formatNumber(summary.total_operations || 0)}</div><div class="sub">entries</div></div>
                `;

                renderMachineCharts(detailRes);
            } catch (error) {
                console.error('Machine detail load failed:', error);
                summaryContainer.innerHTML = '<div class="drilldown-empty" style="grid-column:1/-1;">Failed to load machine details</div>';
            }
        }

        function renderMachineCharts(detailRes) {
            const daily = Array.isArray(detailRes.daily_data) ? detailRes.daily_data : [];
            const tools = Array.isArray(detailRes.tools) ? detailRes.tools.slice(0, 8) : [];
            const summary = detailRes.summary || {};

            if (machineLineChart) machineLineChart.destroy();
            if (machineBarChart) machineBarChart.destroy();
            if (machinePieChart) machinePieChart.destroy();

            const lineCtx = document.getElementById('machineLineChart')?.getContext('2d');
            const barCtx = document.getElementById('machineBarChart')?.getContext('2d');
            const pieCtx = document.getElementById('machinePieChart')?.getContext('2d');

            if (lineCtx) {
                machineLineChart = new Chart(lineCtx, {
                    type: 'line',
                    data: {
                        labels: daily.map(d => d.log_date || ''),
                        datasets: [
                            {
                                label: 'Production',
                                data: daily.map(d => safeNumber(d.total_production_qty)),
                                borderColor: '#0ea5e9',
                                backgroundColor: 'rgba(14, 165, 233, 0.12)',
                                fill: true,
                                tension: 0.3,
                                yAxisID: 'y'
                            },
                            {
                                label: 'Hours',
                                data: daily.map(d => safeNumber(d.total_hours)),
                                borderColor: '#8b5cf6',
                                backgroundColor: 'rgba(139, 92, 246, 0.08)',
                                fill: false,
                                tension: 0.3,
                                yAxisID: 'y1'
                            }
                        ]
                    },
                    options: {
                        responsive: true,
                        maintainAspectRatio: false,
                        interaction: { mode: 'index', intersect: false },
                        scales: {
                            y: { beginAtZero: true, position: 'left' },
                            y1: { beginAtZero: true, position: 'right', grid: { drawOnChartArea: false } }
                        }
                    }
                });
            }

            if (barCtx) {
                machineBarChart = new Chart(barCtx, {
                    type: 'bar',
                    data: {
                        labels: tools.map(t => t.tool_name || 'N/A'),
                        datasets: [{
                            label: 'Production by Tool',
                            data: tools.map(t => safeNumber(t.total_production_qty)),
                            backgroundColor: 'rgba(16, 185, 129, 0.75)',
                            borderColor: '#10b981',
                            borderWidth: 1
                        }]
                    },
                    options: {
                        responsive: true,
                        maintainAspectRatio: false,
                        plugins: { legend: { display: false } },
                        scales: { y: { beginAtZero: true } }
                    }
                });
            }

            if (pieCtx) {
                machinePieChart = new Chart(pieCtx, {
                    type: 'pie',
                    data: {
                        labels: ['Accepted', 'NC'],
                        datasets: [{
                            data: [safeNumber(summary.total_accepted_qty), safeNumber(summary.total_nc_qty)],
                            backgroundColor: ['#10b981', '#ef4444'],
                            borderColor: '#ffffff',
                            borderWidth: 2
                        }]
                    },
                    options: {
                        responsive: true,
                        maintainAspectRatio: false,
                        plugins: { legend: { position: 'bottom' } }
                    }
                });
            }
        }

        async function loadToolDrilldown() {
            const toolSelect = document.getElementById('toolSelect');
            const summaryContainer = document.getElementById('toolSummaryCards');
            if (!toolSelect || !summaryContainer) return;
            try {
                const toolRes = await fetchJson('api.php?action=get_all_tool_production_analytics');
                const tools = Array.isArray(toolRes) ? toolRes : [];
                dashboardState.toolList = tools;

                if (tools.length === 0) {
                    toolSelect.innerHTML = '<option value="">No tool data</option>';
                    toolSelect.disabled = true;
                    summaryContainer.innerHTML = '<div class="drilldown-empty" style="grid-column:1/-1;">No tool records found.</div>';
                    return;
                }

                toolSelect.disabled = false;
                toolSelect.innerHTML = '<option value="">Select Tool</option>' + tools.map(t =>
                    `<option value="${t.tool_no}">${t.tool_no}</option>`
                ).join('');

                toolSelect.value = tools[0].tool_no;
                await renderToolDetails(toolSelect.value);
            } catch (error) {
                console.error('Tool drilldown load failed:', error);
                toolSelect.innerHTML = '<option value="">Error loading tools</option>';
                toolSelect.disabled = true;
                summaryContainer.innerHTML = '<div class="drilldown-empty" style="grid-column:1/-1;">Failed to load tool data.</div>';
            }
        }

        async function renderToolDetails(toolNo) {
            const summaryContainer = document.getElementById('toolSummaryCards');
            if (!summaryContainer || !toolNo) {
                summaryContainer.innerHTML = '<div class="drilldown-empty" style="grid-column:1/-1;">Choose a tool to view details</div>';
                return;
            }
            try {
                const [toolDetailRows, workerRows] = await Promise.all([
                    fetchJson(`api.php?action=get_tool_production_analytics&tool_no=${encodeURIComponent(toolNo)}`),
                    fetchJson(`api.php?action=get_tool_worker_analytics&tool=${encodeURIComponent(toolNo)}`)
                ]);

                const detail = Array.isArray(toolDetailRows) && toolDetailRows.length > 0 ? toolDetailRows[0] : {};
                const workers = Array.isArray(workerRows) ? workerRows : [];

                const totalPlan = safeNumber(detail.total_plan_qty);
                const totalProduced = safeNumber(detail.total_produced_qty);
                const totalAccepted = safeNumber(detail.total_accepted_qty);
                const totalNc = safeNumber(detail.nc_qty);
                const quality = safeNumber(detail.quality_percent);
                const efficiency = safeNumber(detail.efficiency_percent);

                summaryContainer.innerHTML = `
                    <div class="drilldown-summary-card"><div class="label">Total Production</div><div class="value">${formatNumber(totalProduced)}</div><div class="sub">units</div></div>
                    <div class="drilldown-summary-card"><div class="label">Accepted Qty</div><div class="value">${formatNumber(totalAccepted)}</div><div class="sub">accepted units</div></div>
                    <div class="drilldown-summary-card"><div class="label">NC Qty</div><div class="value">${formatNumber(totalNc)}</div><div class="sub">rejected units</div></div>
                    <div class="drilldown-summary-card"><div class="label">Acceptance Rate</div><div class="value">${quality.toFixed(2)}%</div><div class="sub">accepted vs produced</div></div>
                    <div class="drilldown-summary-card"><div class="label">Efficiency</div><div class="value">${efficiency.toFixed(2)}%</div><div class="sub">produced vs plan</div></div>
                    <div class="drilldown-summary-card"><div class="label">Plan Qty</div><div class="value">${formatNumber(totalPlan)}</div><div class="sub">planned units</div></div>
                `;

                renderToolCharts({
                    metrics: { totalPlan, totalProduced, totalAccepted, totalNc },
                    workers
                });
            } catch (error) {
                console.error('Tool detail load failed:', error);
                summaryContainer.innerHTML = '<div class="drilldown-empty" style="grid-column:1/-1;">Failed to load tool details</div>';
            }
        }

        function renderToolCharts(data) {
            const { metrics, workers } = data;
            if (toolBarChart) toolBarChart.destroy();
            if (toolPieChart) toolPieChart.destroy();
            if (toolLineChart) toolLineChart.destroy();

            const barCtx = document.getElementById('toolBarChart')?.getContext('2d');
            const pieCtx = document.getElementById('toolPieChart')?.getContext('2d');
            const lineCtx = document.getElementById('toolLineChart')?.getContext('2d');

            if (barCtx) {
                toolBarChart = new Chart(barCtx, {
                    type: 'bar',
                    data: {
                        labels: ['Plan', 'Produced', 'Accepted', 'NC'],
                        datasets: [{
                            data: [metrics.totalPlan, metrics.totalProduced, metrics.totalAccepted, metrics.totalNc],
                            backgroundColor: ['#94a3b8', '#0ea5e9', '#10b981', '#ef4444'],
                            borderWidth: 1
                        }]
                    },
                    options: {
                        responsive: true,
                        maintainAspectRatio: false,
                        plugins: { legend: { display: false } },
                        scales: { y: { beginAtZero: true } }
                    }
                });
            }

            if (pieCtx) {
                toolPieChart = new Chart(pieCtx, {
                    type: 'pie',
                    data: {
                        labels: ['Accepted', 'NC'],
                        datasets: [{
                            data: [metrics.totalAccepted, metrics.totalNc],
                            backgroundColor: ['#10b981', '#ef4444'],
                            borderColor: '#ffffff',
                            borderWidth: 2
                        }]
                    },
                    options: {
                        responsive: true,
                        maintainAspectRatio: false,
                        plugins: { legend: { position: 'bottom' } }
                    }
                });
            }

            if (lineCtx) {
                toolLineChart = new Chart(lineCtx, {
                    type: 'line',
                    data: {
                        labels: workers.map(w => w.worker_name || 'N/A'),
                        datasets: [{
                            label: 'Produced Qty by Worker',
                            data: workers.map(w => safeNumber(w.produced_qty)),
                            borderColor: '#0ea5e9',
                            backgroundColor: 'rgba(14, 165, 233, 0.12)',
                            fill: true,
                            tension: 0.3,
                            pointRadius: 4
                        }]
                    },
                    options: {
                        responsive: true,
                        maintainAspectRatio: false,
                        plugins: { legend: { display: false } },
                        scales: { y: { beginAtZero: true } }
                    }
                });
            }
        }
    </script>

    <?php if (!empty($pageFlashNotice)) { ?>
    <script>
        Swal.fire({
            icon: 'warning',
            title: 'Access Restricted',
            text: <?php echo json_encode($pageFlashNotice); ?>,
            confirmButtonColor: '#f39c12'
        });
    </script>
    <?php } ?>
    
    <!-- Include Notification UI Component (For all logged-in users) -->
    <?php if (isset($_SESSION) && !empty($_SESSION)) { ?>
    <?php include 'notification_ui.html'; ?>
    <?php } ?>
</body>
</html>
