<?php
include 'session_check.php';
include 'db.php'; 

$view = isset($_GET['view']) ? $_GET['view'] : 'all';
$pageTitle = "Maintenance Dashboard";
if ($view === 'machine') $pageTitle = "Machine Fleet Status";
if ($view === 'tool') $pageTitle = "Tool Stroke Management";
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title><?php echo $pageTitle; ?></title>
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link rel="stylesheet" href="style.css">
    <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
    <style>
        body { background:#f4f6f9; font-family:'Segoe UI',sans-serif; margin: 0; }
        .main-content { margin-left:260px; padding:30px; }
        @media (max-width: 768px) { .main-content { margin-left: 0; } }
        .dashboard-header { display: flex; justify-content: space-between; align-items: center; margin-bottom: 30px; }
        .card { background:white; border-radius:8px; box-shadow:0 2px 8px rgba(0,0,0,0.05); margin-bottom:30px; padding:0; overflow:hidden; }
        .card-header { padding: 15px 20px; border-bottom: 1px solid #eee; background: #fff; font-weight: bold; font-size: 18px; color: #333; display: flex; justify-content: space-between; align-items: center; }
        .table-responsive { overflow-x:auto; }
        table { width:100%; border-collapse:collapse; }
        th, td { padding:12px 15px; border-bottom:1px solid #eee; text-align:left; font-size: 13px; }
        th { background: #f8f9fa; color: #555; font-weight: 600; text-transform: uppercase; font-size: 11px; }
        .badge { padding:4px 8px; border-radius:3px; font-size:10px; font-weight:bold; display: inline-block; }
        .bg-red { background:#f8d7da; color:#721c24; }
        .bg-green { background:#d4edda; color:#155724; }
        .bg-yellow { background:#fff3cd; color:#856404; }
        .btn-action { padding:6px 12px; border:none; border-radius:4px; cursor:pointer; color:white; background:#3c8dbc; display: inline-flex; align-items: center; gap: 5px; text-decoration:none; font-size:12px; margin-right: 5px; }
        .btn-action:hover { opacity: 0.9; }
        .btn-hist { background: #605ca8; }
        .btn-refresh { background: #27ae60; }
        .back-link { display: inline-flex; align-items: center; gap: 5px; margin-bottom: 15px; color: #666; text-decoration: none; font-weight: 600; }
        .back-link:hover { color: #333; }
        
        @keyframes pulse {
            0% { transform: scale(1); opacity: 1; }
            50% { transform: scale(1.05); opacity: 0.8; }
            100% { transform: scale(1); opacity: 1; }
        }
    </style>
</head>
<body>
<script src="sidebar.js"></script>

<div class="main-content">
    
    <?php if ($view !== 'all'): ?>
    <a href="maintenance.php" class="back-link"><i class="fas fa-arrow-left"></i> Back to Overview</a>
    <?php endif; ?>

    <div class="dashboard-header">
        <h2 style="margin:0;"><i class="fas fa-wrench"></i> <?php echo $pageTitle; ?></h2>
    </div>

    <?php if ($view == 'all' || $view == 'machine'): ?>
    <div class="card">
        <div class="card-header"><i class="fas fa-cogs"></i> Machine Fleet Status</div>
        <div class="table-responsive">
            <table>
                <thead><tr><th>Machine Name</th><th>Interval</th><th>Last Service</th><th>Next Due</th><th>Status</th><th>Actions</th></tr></thead>
                <tbody>
                <?php
                $stmt = $pdo->prepare("SELECT id FROM custom_fields WHERE field_label LIKE '%M/C%' OR field_label LIKE '%Machine%' LIMIT 1");
                $stmt->execute();
                $field = $stmt->fetch();

                if ($field) {
                    $fid = $field['id'];
                    $opts = $pdo->prepare("SELECT * FROM field_options WHERE field_id = ? ORDER BY label ASC");
                    $opts->execute([$fid]);

                    if ($opts->rowCount() > 0) {
                        while ($row = $opts->fetch()) {
                            $interval = $row['life_limit'] > 0 ? $row['life_limit'] : 3; 
                            $last = $row['last_service'] ? $row['last_service'] : date('Y-m-d');
                            $next = date('Y-m-d', strtotime("$last +$interval months"));
                            
                            $daysLeft = (strtotime($next) - time()) / (60*60*24);
                            
                            $status = "<span class='badge bg-green'>OK</span>";
                            if ($daysLeft < 0) $status = "<span class='badge bg-red'>OVERDUE</span>";
                            elseif ($daysLeft < 15) $status = "<span class='badge bg-yellow'>DUE SOON</span>";

                            echo "<tr>
                                <td><strong>{$row['label']}</strong></td>
                                <td>{$interval} Months</td>
                                <td>{$last}</td>
                                <td>{$next}</td>
                                <td>{$status}</td>
                                <td>
                                    <button class='btn-action btn-hist' onclick='viewHistory({$row['id']}, \"{$row['label']}\")'><i class='fas fa-history'></i> History</button>
                                    <button class='btn-action' onclick='logService({$row['id']}, \"{$row['label']}\")'><i class='fas fa-check'></i> Log Service</button>
                                </td>
                            </tr>";
                        }
                    } else {
                        echo "<tr><td colspan='6' style='text-align:center; padding:20px; color:#999;'>No machines added yet.</td></tr>";
                    }
                }
                ?>
                </tbody>
            </table>
        </div>
    </div>
    <?php endif; ?>

    <?php if ($view == 'all' || $view == 'tool'): ?>
    <!-- MERGED TOOL MANAGEMENT TABLE -->
    <div class="card">
        <div class="card-header">
            <div style="display: flex; justify-content: space-between; align-items: center; width: 100%;">
                <span><i class="fas fa-hammer"></i> Tool Management & Production Summary (AUTO-UPDATING)</span>
                <button onclick="manualRefreshToolTable()" class="btn-action btn-refresh" style="margin: 0;">
                    <i class="fas fa-sync-alt"></i> Refresh Now
                </button>
            </div>
        </div>
        <div class="table-responsive">
            <table id="toolTable">
                <thead>
                    <tr>
                        <th>Tool Name</th>
                        <th>Total Produced (pcs)</th>
                        <th>Strokes Used</th>
                        <th>Max Life</th>
                        <th>Remaining</th>
                        <th>Usage %</th>
                        <th>Status</th>
                        <th>Action</th>
                    </tr>
                </thead>
                <tbody id="toolTableBody">
                <?php
                $stmt = $pdo->prepare("SELECT id FROM custom_fields WHERE field_label LIKE '%Tool%' LIMIT 1");
                $stmt->execute();
                $toolField = $stmt->fetch();

                if ($toolField) {
                    $fieldId = $toolField['id'];
                    
                    // Fetch all tools from field_options
                    $toolQuery = $pdo->prepare("
                        SELECT 
                            id,
                            label,
                            life_limit,
                            current_usage
                        FROM field_options 
                        WHERE field_id = ? 
                        ORDER BY label ASC
                    ");
                    $toolQuery->execute([$fieldId]);
                    $tools = $toolQuery->fetchAll(PDO::FETCH_ASSOC);

                    if (count($tools) > 0) {
                        foreach ($tools as $tool) {
                            $maxLife = $tool['life_limit'] > 0 ? $tool['life_limit'] : 100000;
                            $usedStrokes = $tool['current_usage'] ?: 0;
                            $remainingStrokes = max(0, $maxLife - $usedStrokes);
                            $usagePercent = ($usedStrokes / $maxLife) * 100;
                            $totalProduced = $usedStrokes;  // current_usage = total qty produced
                            
                            // Determine status
                            $statusBadge = '<span class="badge bg-green">GOOD</span>';
                            $badgeClass = 'bg-green';
                            if ($remainingStrokes <= 0) {
                                $statusBadge = '<span class="badge bg-red">EXPIRED</span>';
                                $badgeClass = 'bg-red';
                            } elseif ($remainingStrokes < 1000) {
                                $statusBadge = '<span class="badge bg-yellow">CRITICAL</span>';
                                $badgeClass = 'bg-yellow';
                            }

                            echo "<tr class='tool-row' data-tool-id='{$tool['id']}' data-max-life='{$maxLife}'>
                                <td class='tool-name'><strong>{$tool['label']}</strong></td>
                                <td style='text-align:center;' class='used-strokes'>" . number_format($totalProduced) . "</td>
                                <td style='text-align:center; font-weight: bold; color: #e74c3c;' class='used-strokes'>" . number_format($usedStrokes) . "</td>
                                <td style='text-align:center;' class='max-strokes'>" . number_format($maxLife) . "</td>
                                <td style='text-align:center; font-weight: bold; color: #27ae60;' class='remaining-strokes'>" . number_format($remainingStrokes) . "</td>
                                <td style='text-align:center;'><span style='background: #3498db; color: white; padding: 3px 8px; border-radius: 3px; font-weight: bold;'>" . round($usagePercent, 1) . "%</span></td>
                                <td class='tool-status'>{$statusBadge}</td>
                                <td><button class='btn-action' style='background:#dd4b39' onclick=\"resetToolStroke({$tool['id']}, '{$tool['label']}')\"><i class='fas fa-sync'></i> Reset</button></td>
                            </tr>";
                        }
                    } else {
                        echo "<tr><td colspan='8' style='text-align:center; padding:20px; color:#999;'>No tools added yet.</td></tr>";
                    }
                } else {
                    echo "<tr><td colspan='8' style='text-align:center; padding:20px; color:#999;'>Tool field not configured.</td></tr>";
                }
                ?>
                </tbody>
            </table>
        </div>
    </div>
    <?php endif; ?>
</div>

<script>
// --- RESET TOOL STROKE (Direct Database Update) ---
function resetToolStroke(toolId, toolName) {
    Swal.fire({
        title: 'Refurbish Tool?',
        text: "Reset stroke count for " + toolName + " to 0? Only do this after physical replacement.",
        icon: 'warning',
        showCancelButton: true,
        confirmButtonText: 'Yes, Reset',
        confirmButtonColor: '#dd4b39'
    }).then((result) => {
        if (result.isConfirmed) {
            let fd = new FormData();
            fd.append('action', 'reset_life_cycle');
            fd.append('id', toolId);
            fd.append('type', 'tool');

            fetch('api.php', { method: 'POST', body: fd })
            .then(res => res.json())
            .then(data => {
                if(data.success) {
                    Swal.fire('Reset!', 'Tool strokes reset to 0.', 'success').then(() => location.reload());
                } else {
                    Swal.fire('Error', data.error || 'Failed to reset tool.', 'error');
                }
            })
            .catch(error => {
                Swal.fire('Error', 'Network error: ' + error.message, 'error');
            });
        }
    });
}

// --- 1. VIEW MACHINE HISTORY ---
function viewHistory(id, name) {
    let fd = new FormData();
    fd.append('action', 'get_machine_history');
    fd.append('id', id);

    fetch('api.php', { method: 'POST', body: fd })
    .then(res => res.json())
    .then(data => {
        let html = `<table style="width:100%; text-align:left; font-size:13px; border-collapse:collapse;">
                    <thead style="background:#f0f0f0;"><tr><th style="padding:8px;">Date</th><th style="padding:8px;">Tech</th><th style="padding:8px;">Notes</th></tr></thead><tbody>`;
        
        if(Array.isArray(data) && data.length > 0) {
            data.forEach(row => {
                html += `<tr>
                            <td style="padding:8px; border-bottom:1px solid #eee;">${row.service_date || 'N/A'}</td>
                            <td style="padding:8px; border-bottom:1px solid #eee;">${row.technician_name || 'N/A'}</td>
                            <td style="padding:8px; border-bottom:1px solid #eee;">${row.service_notes || 'N/A'}</td>
                         </tr>`;
            });
        } else {
            html += `<tr><td colspan="3" style="padding:15px; text-align:center; color:#999;">No history found.</td></tr>`;
        }
        html += `</tbody></table>`;

        Swal.fire({
            title: 'Maintenance History: ' + name,
            html: html,
            width: '600px',
            showConfirmButton: false,
            showCloseButton: true
        });
    })
    .catch(error => {
        Swal.fire('Error', 'Failed to load history: ' + error.message, 'error');
    });
}

// --- 2. LOG MACHINE SERVICE ---
async function logService(id, name) {
    const { value: formValues } = await Swal.fire({
        title: 'Log Service',
        html: `
            <div style="text-align:left; font-size:13px; color:#555; margin-bottom:10px;">Logging maintenance for <strong>${name}</strong> will reset the due date based on its interval.</div>
            <input id="swal-tech" class="swal2-input" placeholder="Technician Name" required>
            <textarea id="swal-notes" class="swal2-textarea" placeholder="Service Notes (e.g. Oil changed, Belt Replaced)"></textarea>
        `,
        focusConfirm: false,
        showCancelButton: true,
        confirmButtonText: 'Save & Reset Cycle',
        preConfirm: () => {
            return {
                tech: document.getElementById('swal-tech').value,
                notes: document.getElementById('swal-notes').value
            }
        }
    });

    if (formValues) {
        if(!formValues.tech || formValues.tech.trim() === '') { 
            Swal.fire('Error', 'Technician name is required', 'error'); 
            return; 
        }

        let fd = new FormData();
        fd.append('action', 'reset_life_cycle');
        fd.append('id', id);
        fd.append('type', 'machine');
        fd.append('technician', formValues.tech.trim());
        fd.append('notes', formValues.notes.trim());

        try {
            const response = await fetch('api.php', { method: 'POST', body: fd });
            const data = await response.json();
            
            if(data.success) {
                Swal.fire('Success!', 'Maintenance recorded and cycle reset.', 'success').then(() => location.reload());
            } else {
                Swal.fire('Error', data.error || 'Failed to update the service log.', 'error');
            }
        } catch (error) {
            Swal.fire('Error', 'Network error: ' + error.message, 'error');
        }
    }
}

// --- DYNAMIC TABLE REFRESH WITHOUT FULL RELOAD ---
async function refreshToolTable() {
    try {
        const response = await fetch('api.php?action=get_tool_data');
        const result = await response.json();
        
        if (result.success && Array.isArray(result.tools)) {
            let hasChanges = false;
            
            result.tools.forEach(tool => {
                const row = document.querySelector(`tr[data-tool-id="${tool.id}"]`);
                if (row) {
                    // Update production (first used-strokes cell)
                    const usedCells = row.querySelectorAll('.used-strokes');
                    if (usedCells.length >= 1) {
                        const displayValue = number_format(tool.current_usage);
                        if (usedCells[0].textContent !== displayValue) {
                            highlightCell(usedCells[0], displayValue);
                            hasChanges = true;
                        }
                    }
                    
                    // Update strokes used (second cell)
                    if (usedCells.length >= 2) {
                        const displayValue = number_format(tool.current_usage);
                        if (usedCells[1].textContent !== displayValue) {
                            highlightCell(usedCells[1], displayValue);
                            hasChanges = true;
                        }
                    }
                    
                    // Update remaining strokes
                    const remainingCell = row.querySelector('.remaining-strokes');
                    if (remainingCell) {
                        const displayValue = number_format(tool.remaining);
                        if (remainingCell.textContent !== displayValue) {
                            highlightCell(remainingCell, displayValue);
                            hasChanges = true;
                        }
                    }
                    
                    // Update usage percent
                    const usageCell = row.querySelector('[style*="background: #3498db"]');
                    if (usageCell) {
                        const displayValue = tool.usage_percent.toFixed(1) + '%';
                        if (!usageCell.textContent.includes(displayValue)) {
                            usageCell.textContent = displayValue;
                            usageCell.style.animation = 'none';
                            setTimeout(() => {
                                usageCell.style.animation = 'pulse 0.6s ease-in-out';
                            }, 10);
                            hasChanges = true;
                        }
                    }
                    
                    // Update status badge
                    const statusCell = row.querySelector('.tool-status');
                    if (statusCell) {
                        const newHTML = getStatusBadge(tool.status);
                        if (statusCell.innerHTML !== newHTML) {
                            statusCell.innerHTML = newHTML;
                            statusCell.style.animation = 'none';
                            setTimeout(() => {
                                statusCell.style.animation = 'pulse 0.6s ease-in-out';
                            }, 10);
                            hasChanges = true;
                        }
                    }
                }
            });
            
            // Log updates for debugging
            if (hasChanges) {
                console.log('✅ Tool data updated at', new Date().toLocaleTimeString());
            }
        }
    } catch (error) {
        console.error('Tool table refresh error:', error);
    }
}

function number_format(num) {
    return num.toString().replace(/\B(?=(\d{3})+(?!\d))/g, ",");
}

function manualRefreshToolTable() {
    console.log('Manual refresh triggered');
    refreshToolTable();
}

function highlightCell(cell, newValue) {
    cell.textContent = newValue;
    cell.style.backgroundColor = '#fff3cd';
    cell.style.transition = 'background-color 0.3s ease';
    setTimeout(() => {
        cell.style.backgroundColor = '';
    }, 1000);
}

function getStatusBadge(status) {
    switch(status) {
        case 'GOOD': return '<span class="badge bg-green">GOOD</span>';
        case 'CRITICAL': return '<span class="badge bg-yellow">CRITICAL</span>';
        case 'EXPIRED': return '<span class="badge bg-red">EXPIRED</span>';
        default: return '<span class="badge bg-green">GOOD</span>';
    }
}

// Global refresh counter and state
let lastRefreshTime = null;
let refreshCounter = 0;

// --- AUTO-REFRESH ON PRODUCTION ENTRY SAVE ---
window.addEventListener('storage', function(e) {
    if (e.key === 'maintenance_refresh_trigger') {
        console.log('🔔 Production entry detected, refreshing tool data immediately...');
        refreshToolTable();
    }
});

// --- RAPID AUTO-REFRESH EVERY 3 SECONDS (PRODUCTION READY) ---
let refreshInterval;

function startAutoRefresh() {
    if (!refreshInterval) {
        refreshToolTable(); // Initial refresh
        
        // Refresh every 3 seconds for real-time updates
        refreshInterval = setInterval(() => {
            refreshCounter++;
            lastRefreshTime = new Date().toLocaleTimeString();
            refreshToolTable();
        }, 3000);
        
        console.log('✅ Auto-refresh started (3-second interval)');
    }
}

function stopAutoRefresh() {
    if (refreshInterval) {
        clearInterval(refreshInterval);
        refreshInterval = null;
        console.log('⏸ Auto-refresh stopped');
    }
}

// Start auto-refresh when page loads
document.addEventListener('DOMContentLoaded', function() {
    startAutoRefresh();
    
    // Add refresh status display
    const header = document.querySelector('.dashboard-header');
    if (header) {
        const statusDiv = document.createElement('div');
        statusDiv.id = 'refreshStatus';
        statusDiv.style.cssText = 'font-size: 12px; color: #666; margin-left: 20px;';
        header.appendChild(statusDiv);
    }
});

// Stop when page is hidden, resume when visible (saves resources)
document.addEventListener('visibilitychange', function() {
    if (document.hidden) {
        stopAutoRefresh();
        console.log('📱 Page hidden - auto-refresh paused');
    } else {
        startAutoRefresh();
        console.log('📱 Page visible - auto-refresh resumed');
    }
});
</script>
</body>
</html>