<?php
/**
 * Notification System
 * ==================
 * Handles:
 * 1. Machine maintenance alerts (due date < 15 days)
 * 2. Tool stroke alerts (remaining < 1000)
 * 3. Admin-only visibility
 * 4. Configurable thresholds
 */

session_start();
header('Content-Type: application/json');
ini_set('display_errors', 0);
error_reporting(E_ALL);

// Log all requests for debugging
$action = trim($_GET['action'] ?? $_POST['action'] ?? '');
$username = $_SESSION['username'] ?? 'unknown';
error_log("[Notification API] Action: $action, Method: {$_SERVER['REQUEST_METHOD']}, User: $username");

try {
  $host = 'localhost'; 
    $db = 'zemaraim_mektron_db'; 
    $user = 'zemaraim'; 
    $pass = '8wwxk]-X3Kx]CJ]5';
    
    $pdo = new PDO("mysql:host=$host;dbname=$db;charset=utf8mb4", $user, $pass, [
        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC
    ]);

    // Initialize database if needed
    require_once 'init_notifications_db.php';

} catch (\PDOException $e) {
    error_log("[Notification API] DB Connection Failed: " . $e->getMessage());
    http_response_code(500);
    echo json_encode(['success' => false, 'error' => 'DB Connection Failed: ' . $e->getMessage()]);
    exit;
} catch (\Exception $e) {
    error_log("[Notification API] DB Initialization Failed: " . $e->getMessage());
    http_response_code(500);
    echo json_encode(['success' => false, 'error' => 'DB Initialization Failed: ' . $e->getMessage()]);
    exit;
}

// Check if user is admin
function isAdmin() {
    $role = strtolower(trim($_SESSION['role'] ?? ''));
    return in_array($role, ['admin', 'administrator'], true);
}

// Return empty notifications if not admin
function requireAdminOrEmpty() {
    if (!isAdmin()) {
        http_response_code(200);
        echo json_encode(['success' => true, 'notifications' => []]);
        exit;
    }
}

$action = trim($_GET['action'] ?? $_POST['action'] ?? '');

// ========== NOTIFICATIONS ==========

if ($action === 'get_notifications') {
    // Show notifications for ALL users
    
    try {
        $notifications = [];
        
        // Get machine field ID
        $machineFieldStmt = $pdo->prepare("SELECT id FROM custom_fields WHERE field_label LIKE '%M/C%' OR field_label LIKE '%Machine%' LIMIT 1");
        $machineFieldStmt->execute();
        $machineField = $machineFieldStmt->fetch();
        
        // Get tool field ID
        $toolFieldStmt = $pdo->prepare("SELECT id FROM custom_fields WHERE field_label LIKE '%Tool%' LIMIT 1");
        $toolFieldStmt->execute();
        $toolField = $toolFieldStmt->fetch();
        
        // === MACHINES DUE WITHIN 30 DAYS ===
        if ($machineField) {
            $machineStmt = $pdo->prepare("
                SELECT 
                    id, label as machine_name, last_service, life_limit
                FROM field_options
                WHERE field_id = ?
                AND last_service IS NOT NULL
                AND last_service != ''
            ");
            $machineStmt->execute([$machineField['id']]);
            
            while ($machine = $machineStmt->fetch()) {
                // Calculate next due date (life_limit is in months for machines)
                $intervalMonths = intval($machine['life_limit']) > 0 ? intval($machine['life_limit']) : 3;
                $nextDueDate = date('Y-m-d', strtotime($machine['last_service'] . " +" . $intervalMonths . " months"));
                
                $daysLeft = (strtotime($nextDueDate) - time()) / (60*60*24);
                $daysLeftInt = ceil($daysLeft);
                
                // Show if due within 30 days and not expired yet
                if ($daysLeftInt > -7 && $daysLeftInt <= 30) {
                    $notifications[] = [
                        'id' => 'machine_' . $machine['id'],
                        'type' => 'machine',
                        'title' => 'Machine Due For Maintenance',
                        'message' => $machine['machine_name'] . ' is due for maintenance in ' . $daysLeftInt . ' day' . ($daysLeftInt !== 1 ? 's' : ''),
                        'severity' => $daysLeftInt <= 7 ? 'critical' : ($daysLeftInt <= 14 ? 'warning' : 'info'),
                        'item_id' => $machine['id'],
                        'item_name' => $machine['machine_name'],
                        'last_service' => $machine['last_service'],
                        'due_date' => $nextDueDate,
                        'life_limit_days' => $intervalMonths * 30, // Convert months to approximate days for display
                        'days_left' => $daysLeftInt,
                        'timestamp' => date('Y-m-d H:i:s')
                    ];
                }
            }
            
            // === EXPIRED/OVERDUE MACHINES ===
            $expiredMachineStmt = $pdo->prepare("
                SELECT 
                    id, label as machine_name, last_service, life_limit
                FROM field_options
                WHERE field_id = ?
                AND last_service IS NOT NULL
                AND last_service != ''
            ");
            $expiredMachineStmt->execute([$machineField['id']]);
            
            while ($machine = $expiredMachineStmt->fetch()) {
                $intervalMonths = intval($machine['life_limit']) > 0 ? intval($machine['life_limit']) : 3;
                $nextDueDate = date('Y-m-d', strtotime($machine['last_service'] . " +" . $intervalMonths . " months"));
                
                $daysOverdue = (time() - strtotime($nextDueDate)) / (60*60*24);
                $daysOverdueInt = ceil($daysOverdue);
                
                // Show only overdue machines
                if ($daysOverdueInt >= 7) {
                    $notifications[] = [
                        'id' => 'machine_expired_' . $machine['id'],
                        'type' => 'machine',
                        'title' => '⚠️ Machine EXPIRED - Immediate Maintenance Required',
                        'message' => $machine['machine_name'] . ' maintenance is ' . $daysOverdueInt . ' day' . ($daysOverdueInt !== 1 ? 's' : '') . ' overdue',
                        'severity' => 'critical',
                        'item_id' => $machine['id'],
                        'item_name' => $machine['machine_name'],
                        'last_service' => $machine['last_service'],
                        'due_date' => $nextDueDate,
                        'life_limit_days' => $intervalMonths * 30,
                        'days_overdue' => $daysOverdueInt,
                        'timestamp' => date('Y-m-d H:i:s')
                    ];
                }
            }
        }
        
        // === EXPIRED TOOLS (reached max strokes) ===
        if ($toolField) {
            $expiredToolStmt = $pdo->prepare("
                SELECT 
                    id, label as tool_name, current_usage, life_limit
                FROM field_options
                WHERE field_id = ?
                AND current_usage >= life_limit
                AND life_limit > 0
                ORDER BY current_usage DESC
            ");
            $expiredToolStmt->execute([$toolField['id']]);
            
            while ($tool = $expiredToolStmt->fetch()) {
                $notifications[] = [
                    'id' => 'tool_expired_' . $tool['id'],
                    'type' => 'tool',
                    'title' => '🔧 Tool EXPIRED - Replace Immediately',
                    'message' => $tool['tool_name'] . ' has reached its stroke limit (' . number_format(intval($tool['life_limit'])) . ' strokes) and must be replaced',
                    'severity' => 'critical',
                    'item_id' => $tool['id'],
                    'item_name' => $tool['tool_name'],
                    'current_usage' => intval($tool['current_usage'] ?? 0),
                    'max_strokes' => intval($tool['life_limit'] ?? 0),
                    'remaining_strokes' => 0,
                    'usage_percent' => 100,
                    'timestamp' => date('Y-m-d H:i:s')
                ];
            }
        }
        
        // === TOOLS LOW ON STROKES (< 20% remaining) ===
        if ($toolField) {
            $lowToolStmt = $pdo->prepare("
                SELECT 
                    id, label as tool_name, current_usage, life_limit
                FROM field_options
                WHERE field_id = ?
                AND life_limit > 0
                AND current_usage < life_limit
                AND (life_limit - current_usage) <= (life_limit * 0.2)
                ORDER BY current_usage DESC
            ");
            $lowToolStmt->execute([$toolField['id']]);
            
            while ($tool = $lowToolStmt->fetch()) {
                $remaining = intval($tool['life_limit']) - intval($tool['current_usage']);
                $usagePercent = round(((floatval($tool['current_usage']) / floatval($tool['life_limit'])) * 100), 1);
                
                $notifications[] = [
                    'id' => 'tool_low_' . $tool['id'],
                    'type' => 'tool',
                    'title' => '⚠️ Tool Maintenance Needed',
                    'message' => $tool['tool_name'] . ' strokes remaining: ' . number_format($remaining) . ' / ' . number_format(intval($tool['life_limit'])) . ' (' . $usagePercent . '%)',
                    'severity' => $remaining <= (intval($tool['life_limit']) * 0.05) ? 'critical' : 'warning',
                    'item_id' => $tool['id'],
                    'item_name' => $tool['tool_name'],
                    'current_usage' => intval($tool['current_usage'] ?? 0),
                    'max_strokes' => intval($tool['life_limit'] ?? 0),
                    'remaining_strokes' => $remaining,
                    'usage_percent' => $usagePercent,
                    'timestamp' => date('Y-m-d H:i:s')
                ];
            }
        }
        
        // Sort by severity (critical first) and then by date
        usort($notifications, function($a, $b) {
            $severityOrder = ['critical' => 0, 'warning' => 1, 'info' => 2];
            $severityCompare = $severityOrder[$a['severity']] - $severityOrder[$b['severity']];
            if ($severityCompare !== 0) return $severityCompare;
            
            // For same severity, sort by days_left (ascending) or days_overdue (descending)
            if (isset($a['days_left']) && isset($b['days_left'])) {
                return $a['days_left'] - $b['days_left'];
            }
            if (isset($a['days_overdue']) && isset($b['days_overdue'])) {
                return $b['days_overdue'] - $a['days_overdue'];
            }
            return 0;
        });
        
        http_response_code(200);
        echo json_encode([
            'success' => true,
            'notifications' => $notifications,
            'total_count' => count($notifications),
            'critical_count' => count(array_filter($notifications, fn($n) => $n['severity'] === 'critical')),
            'warning_count' => count(array_filter($notifications, fn($n) => $n['severity'] === 'warning')),
            'info_count' => count(array_filter($notifications, fn($n) => $n['severity'] === 'info'))
        ]);
    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode(['success' => false, 'error' => $e->getMessage()]);
    }
    exit;
}

if ($action === 'get_notification_settings') {
    requireAdminOrEmpty();
    
    try {
        $settingsStmt = $pdo->query("SELECT setting_key, setting_value FROM notification_settings");
        $settingsResult = $settingsStmt->fetchAll();
        
        if (empty($settingsResult)) {
            error_log("[Notification API] No settings found in database, using defaults");
        }
        
        $settings = [];
        foreach ($settingsResult as $row) {
            $settings[$row['setting_key']] = intval($row['setting_value']);
        }
        
        // Provide defaults if not found
        $settings['machine_days_threshold'] = $settings['machine_days_threshold'] ?? 15;
        $settings['tool_strokes_threshold'] = $settings['tool_strokes_threshold'] ?? 1000;
        
        error_log("[Notification API] Settings loaded: Machine={$settings['machine_days_threshold']}, Tool={$settings['tool_strokes_threshold']}");
        
        http_response_code(200);
        echo json_encode([
            'success' => true,
            'settings' => $settings
        ]);
    } catch (Exception $e) {
        error_log("[Notification API] Error loading settings: " . $e->getMessage());
        http_response_code(500);
        echo json_encode(['success' => false, 'error' => $e->getMessage()]);
    }
    exit;
}

if ($action === 'update_notification_settings') {
    // Settings update is admin only
    if (!isAdmin()) {
        http_response_code(403);
        echo json_encode(['success' => false, 'error' => 'Admin access required']);
        error_log("[Notification API] Access denied - not admin");
        exit;
    }
    
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        http_response_code(405);
        echo json_encode(['success' => false, 'error' => 'POST required']);
        exit;
    }
    
    try {
        $machineThreshold = intval($_POST['machine_days_threshold'] ?? 15);
        $toolThreshold = intval($_POST['tool_strokes_threshold'] ?? 1000);
        
        error_log("[Notification API] Updating settings - Machine: $machineThreshold, Tool: $toolThreshold");
        
        // Validate ranges
        if ($machineThreshold < 1 || $machineThreshold > 365) {
            throw new Exception('Machine threshold must be between 1 and 365 days');
        }
        if ($toolThreshold < 1 || $toolThreshold > 100000) {
            throw new Exception('Tool threshold must be between 1 and 100,000 strokes');
        }
        
        // Update or insert settings - using better approach
        // First, try to update existing
        $updateStmt = $pdo->prepare("
            UPDATE notification_settings 
            SET setting_value = ?, updated_at = CURRENT_TIMESTAMP
            WHERE setting_key = ?
        ");
        
        $updateStmt->execute([$machineThreshold, 'machine_days_threshold']);
        $machineUpdated = $updateStmt->rowCount();
        
        $updateStmt->execute([$toolThreshold, 'tool_strokes_threshold']);
        $toolUpdated = $updateStmt->rowCount();
        
        // If not updated (0 rows), insert instead
        if ($machineUpdated == 0) {
            $insertStmt = $pdo->prepare("
                INSERT INTO notification_settings (setting_key, setting_value, description) 
                VALUES (?, ?, ?)
            ");
            $insertStmt->execute(['machine_days_threshold', $machineThreshold, 'Days before machine maintenance alert']);
            error_log("[Notification API] Inserted machine_days_threshold");
        }
        
        if ($toolUpdated == 0) {
            $insertStmt = $pdo->prepare("
                INSERT INTO notification_settings (setting_key, setting_value, description) 
                VALUES (?, ?, ?)
            ");
            $insertStmt->execute(['tool_strokes_threshold', $toolThreshold, 'Remaining strokes before tool alert']);
            error_log("[Notification API] Inserted tool_strokes_threshold");
        }
        
        error_log("[Notification API] Settings saved successfully");
        
        http_response_code(200);
        echo json_encode([
            'success' => true,
            'message' => 'Settings updated successfully',
            'settings' => [
                'machine_days_threshold' => $machineThreshold,
                'tool_strokes_threshold' => $toolThreshold
            ]
        ]);
    } catch (Exception $e) {
        error_log("[Notification API] Error updating settings: " . $e->getMessage());
        http_response_code(500);
        echo json_encode(['success' => false, 'error' => $e->getMessage()]);
    }
    exit;
}

// Default response
http_response_code(400);
echo json_encode([
    'success' => false,
    'error' => 'No action specified',
    'available_actions' => [
        'get_notifications' => 'GET - Get all active notifications',
        'get_notification_settings' => 'GET - Get notification thresholds',
        'update_notification_settings' => 'POST - Update notification thresholds'
    ]
]);
?>
