<?php
/**
 * Production Stroke Handler
 * ========================
 * This file handles real-time updates to tool stroke counters when production is logged.
 * It's called by the main API and handles all tool lifecycle management.
 * 
 * Usage Examples:
 * GET /production_stroke_handler.php?action=get_tool_status&tool_id=123
 * POST /production_stroke_handler.php with action=update_tool_stroke
 */

session_start();
header('Content-Type: application/json');
ini_set('display_errors', 0);
error_reporting(E_ALL);

// Database connection
try {
    $host = 'localhost'; 
    $db = 'zemaraim_mektron_db'; 
    $user = 'root'; 
    $pass = '';
    
    $pdo = new PDO("mysql:host=$host;dbname=$db;charset=utf8mb4", $user, $pass, [
        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC
    ]);
} catch (\PDOException $e) { 
    http_response_code(500);
    echo json_encode(['success' => false, 'error' => 'Database Connection Failed: ' . $e->getMessage()]);
    exit;
}

// Helper function to update tool strokes
function updateToolStrokes($toolId, $producedQty, $logDate = null) {
    global $pdo;
    
    if (empty($toolId) || empty($producedQty) || $producedQty <= 0) {
        error_log("⚠️ Invalid tool update request: toolId=$toolId, qty=$producedQty");
        return false;
    }
    
    try {
        $toolId = intval($toolId);
        $producedQty = intval($producedQty);
        $logDate = $logDate ?? date('Y-m-d');
        
        error_log("🔧 updateToolStrokes() called: toolId=$toolId, qty=$producedQty, date=$logDate");
        
        // Get tool info
        $toolStmt = $pdo->prepare(
            "SELECT id, label, current_usage, life_limit FROM field_options WHERE id = ?"
        );
        $toolStmt->execute([$toolId]);
        $tool = $toolStmt->fetch();
        
        if (!$tool) {
            error_log("❌ Tool not found: $toolId");
            return false;
        }
        
        $currentUsage = intval($tool['current_usage'] ?: 0);
        $maxLife = intval($tool['life_limit'] ?: 100000);
        $newUsage = $currentUsage + $producedQty;
        $remaining = max(0, $maxLife - $newUsage);
        
        // Update tool strokes
        $updateStmt = $pdo->prepare(
            "UPDATE field_options SET current_usage = current_usage + ? WHERE id = ?"
        );
        $result = $updateStmt->execute([$producedQty, $toolId]);
        
        if ($result) {
            error_log("✅ Tool strokes updated successfully");
            error_log("   Tool: {$tool['label']}");
            error_log("   Old usage: $currentUsage");
            error_log("   New usage: $newUsage");
            error_log("   Remaining: $remaining");
            error_log("   Max life: $maxLife");
            
            return [
                'success' => true,
                'tool_id' => $toolId,
                'tool_label' => $tool['label'],
                'previous_usage' => $currentUsage,
                'current_usage' => $newUsage,
                'remaining' => $remaining,
                'max_life' => $maxLife,
                'usage_percent' => round(($newUsage / $maxLife) * 100, 1)
            ];
        } else {
            error_log("❌ Failed to update tool strokes for tool $toolId");
            return false;
        }
    } catch (Exception $e) {
        error_log("❌ Error in updateToolStrokes: " . $e->getMessage());
        return false;
    }
}

// Direct API call handler
$action = trim($_GET['action'] ?? $_POST['action'] ?? '');

if (empty($action)) {
    // If no action, show helpful HTML page
    $isAjax = !empty($_GET['json']) || !empty($_POST['json']) || 
              (isset($_SERVER['HTTP_X_REQUESTED_WITH']) && $_SERVER['HTTP_X_REQUESTED_WITH'] === 'XMLHttpRequest');
    
    if ($isAjax) {
        // Return JSON for AJAX requests
        http_response_code(400);
        echo json_encode([
            'success' => false, 
            'error' => 'No action specified',
            'available_actions' => [
                'update_tool_stroke' => 'POST - Update tool strokes (tool_id, quantity, log_date)',
                'get_tool_status' => 'GET - Get tool current status (tool_id)',
                'get_all_tools' => 'GET - Get all tools status'
            ]
        ]);
    } else {
        // Show HTML page for direct browser access
        header('Content-Type: text/html; charset=utf-8');
        ?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Production Stroke Handler - API Documentation</title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 20px;
        }
        .container {
            background: white;
            border-radius: 12px;
            box-shadow: 0 20px 60px rgba(0,0,0,0.3);
            max-width: 900px;
            width: 100%;
            padding: 40px;
        }
        h1 {
            color: #333;
            margin-bottom: 10px;
            display: flex;
            align-items: center;
            gap: 15px;
        }
        h1 i {
            font-size: 32px;
            color: #667eea;
        }
        .subtitle {
            color: #666;
            margin-bottom: 30px;
            font-size: 16px;
        }
        .alert {
            background: #fff3cd;
            border: 1px solid #ffc107;
            color: #856404;
            padding: 15px 20px;
            border-radius: 6px;
            margin-bottom: 30px;
            display: flex;
            gap: 10px;
        }
        .alert i {
            color: #ffc107;
            flex-shrink: 0;
        }
        .section {
            margin-bottom: 30px;
        }
        .section h2 {
            color: #333;
            font-size: 20px;
            margin-bottom: 15px;
            padding-bottom: 10px;
            border-bottom: 2px solid #667eea;
        }
        .endpoint {
            background: #f8f9fa;
            border-left: 4px solid #667eea;
            padding: 20px;
            margin-bottom: 15px;
            border-radius: 6px;
        }
        .endpoint h3 {
            color: #667eea;
            margin-bottom: 8px;
            display: flex;
            align-items: center;
            gap: 10px;
        }
        .badge {
            display: inline-block;
            padding: 4px 12px;
            border-radius: 20px;
            font-size: 12px;
            font-weight: bold;
            color: white;
        }
        .badge-get {
            background: #28a745;
        }
        .badge-post {
            background: #007bff;
        }
        .description {
            color: #666;
            font-size: 14px;
            margin-bottom: 10px;
        }
        .params {
            background: white;
            padding: 12px;
            border-radius: 4px;
            margin-top: 10px;
            font-family: 'Courier New', monospace;
            font-size: 13px;
            color: #333;
        }
        .param-item {
            margin-bottom: 8px;
            line-height: 1.6;
        }
        .param-name {
            color: #e83e8c;
            font-weight: bold;
        }
        .example {
            background: #2d2d2d;
            color: #f8f8f2;
            padding: 15px;
            border-radius: 4px;
            font-family: 'Courier New', monospace;
            font-size: 13px;
            margin-top: 10px;
            overflow-x: auto;
            line-height: 1.5;
        }
        .footer {
            text-align: center;
            color: #999;
            font-size: 13px;
            margin-top: 40px;
            padding-top: 20px;
            border-top: 1px solid #eee;
        }
        .link-pill {
            display: inline-block;
            background: #667eea;
            color: white;
            padding: 10px 20px;
            border-radius: 20px;
            text-decoration: none;
            margin-top: 10px;
            transition: background 0.3s;
        }
        .link-pill:hover {
            background: #764ba2;
        }
        @media (max-width: 768px) {
            .container { padding: 20px; }
            h1 { font-size: 24px; }
            .section h2 { font-size: 18px; }
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>
            <span>🔧</span> Production Stroke Handler API
        </h1>
        <p class="subtitle">Tool Stroke Management & Tracking System</p>
        
        <div class="alert">
            <span>⚠️</span>
            <div>
                <strong>No Action Specified</strong><br>
                This endpoint requires an action parameter. Please use one of the available endpoints below.
            </div>
        </div>

        <div class="section">
            <h2>📋 Available Endpoints</h2>
            
            <div class="endpoint">
                <h3>
                    <span class="badge badge-post">POST</span>
                    Update Tool Strokes
                </h3>
                <p class="description">Increase tool stroke counter when production is completed</p>
                <div class="params">
                    <div class="param-item"><span class="param-name">action</span> = "update_tool_stroke"</div>
                    <div class="param-item"><span class="param-name">tool_id</span> = tool ID (required)</div>
                    <div class="param-item"><span class="param-name">quantity</span> = strokes to add (required)</div>
                    <div class="param-item"><span class="param-name">log_date</span> = date (optional)</div>
                </div>
                <div class="example">POST /production_stroke_handler.php<br>
Body: tool_id=123&quantity=50&log_date=2026-02-12</div>
            </div>

            <div class="endpoint">
                <h3>
                    <span class="badge badge-get">GET</span>
                    Get Tool Status
                </h3>
                <p class="description">Retrieve current stroke status for a specific tool</p>
                <div class="params">
                    <div class="param-item"><span class="param-name">action</span> = "get_tool_status"</div>
                    <div class="param-item"><span class="param-name">tool_id</span> = tool ID (required)</div>
                </div>
                <div class="example">GET /production_stroke_handler.php?action=get_tool_status&tool_id=123</div>
                <a href="production_stroke_handler.php?action=get_tool_status&tool_id=1" class="link-pill">Try Example</a>
            </div>

            <div class="endpoint">
                <h3>
                    <span class="badge badge-get">GET</span>
                    Get All Tools
                </h3>
                <p class="description">Retrieve stroke status for all tools in the system</p>
                <div class="params">
                    <div class="param-item"><span class="param-name">action</span> = "get_all_tools"</div>
                </div>
                <div class="example">GET /production_stroke_handler.php?action=get_all_tools</div>
                <a href="production_stroke_handler.php?action=get_all_tools" class="link-pill">Try Example</a>
            </div>
        </div>

        <div class="section">
            <h2>✨ Response Examples</h2>
            
            <h3 style="color: #333; margin-top: 20px; margin-bottom: 10px;">Success Response (200)</h3>
            <div class="example">{
  "success": true,
  "tool": {
    "id": 123,
    "label": "g3",
    "current_usage": 1500,
    "max_life": 50000,
    "remaining": 48500,
    "usage_percent": 3.0,
    "status": "GOOD"
  }
}</div>

            <h3 style="color: #333; margin-top: 20px; margin-bottom: 10px;">Error Response (400/404/500)</h3>
            <div class="example">{
  "success": false,
  "error": "Tool not found"
}</div>
        </div>

        <div class="section">
            <h2>🔗 Quick Links</h2>
            <p style="margin-bottom: 15px; color: #666;">Navigate to other parts of the system:</p>
            <div>
                <a href="maintenance.php" style="color: #667eea; text-decoration: none; margin-right: 20px;">📊 Maintenance Dashboard</a><br>
                <a href="entry_log.php" style="color: #667eea; text-decoration: none; margin-right: 20px;">📝 Production Entry</a><br>
                <a href="api.php" style="color: #667eea; text-decoration: none;">🔌 Main API</a>
            </div>
        </div>

        <div class="footer">
            <p>Production Stroke Handler v1.0 | ERP Manufacturing System</p>
            <p>Last Updated: <?php echo date('F d, Y'); ?></p>
        </div>
    </div>

    <!-- Add Font Awesome for icons -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
</body>
</html>
        <?php
    }
    exit;
}

if ($action === 'update_tool_stroke') {
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        http_response_code(405);
        echo json_encode(['success' => false, 'error' => 'POST request required']);
        exit;
    }
    
    $toolId = $_POST['tool_id'] ?? null;
    $qty = $_POST['quantity'] ?? $_POST['produced_qty'] ?? null;
    $logDate = $_POST['log_date'] ?? null;
    
    $result = updateToolStrokes($toolId, $qty, $logDate);
    http_response_code($result ? 200 : 400);
    echo json_encode($result ?: ['success' => false, 'error' => 'Update failed']);
    exit;
}

if ($action === 'get_tool_status') {
    $toolId = intval($_GET['tool_id'] ?? 0);
    
    if ($toolId <= 0) {
        http_response_code(400);
        echo json_encode(['success' => false, 'error' => 'Invalid tool ID']);
        exit;
    }
    
    try {
        $stmt = $pdo->prepare(
            "SELECT id, label, current_usage, life_limit FROM field_options WHERE id = ?"
        );
        $stmt->execute([$toolId]);
        $tool = $stmt->fetch();
        
        if (!$tool) {
            http_response_code(404);
            echo json_encode(['success' => false, 'error' => 'Tool not found']);
            exit;
        }
        
        $current = intval($tool['current_usage'] ?: 0);
        $max = intval($tool['life_limit'] ?: 100000);
        $remaining = max(0, $max - $current);
        $usagePercent = round(($current / $max) * 100, 1);
        
        $status = 'GOOD';
        if ($remaining <= 0) $status = 'EXPIRED';
        elseif ($remaining < 1000) $status = 'CRITICAL';
        
        http_response_code(200);
        echo json_encode([
            'success' => true,
            'tool' => [
                'id' => $tool['id'],
                'label' => $tool['label'],
                'current_usage' => $current,
                'max_life' => $max,
                'remaining' => $remaining,
                'usage_percent' => $usagePercent,
                'status' => $status
            ]
        ]);
    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode(['success' => false, 'error' => $e->getMessage()]);
    }
    exit;
}

if ($action === 'get_all_tools') {
    try {
        // Get the tool field ID from custom_fields
        $fieldStmt = $pdo->prepare("SELECT id FROM custom_fields WHERE field_label LIKE '%Tool%' LIMIT 1");
        $fieldStmt->execute();
        $field = $fieldStmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$field) {
            http_response_code(404);
            echo json_encode(['success' => false, 'error' => 'Tool field not found']);
            exit;
        }
        
        $toolFieldId = $field['id'];
        
        // Fetch all tools with their current usage
        $stmt = $pdo->prepare("
            SELECT 
                id, 
                label, 
                current_usage, 
                life_limit 
            FROM field_options 
            WHERE field_id = ? 
            ORDER BY label ASC
        ");
        $stmt->execute([$toolFieldId]);
        $tools = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Process each tool with calculations
        $toolsData = [];
        foreach ($tools as $tool) {
            $maxLife = ($tool['life_limit'] ?: 100000);
            $usedStrokes = ($tool['current_usage'] ?: 0);
            $remaining = max(0, $maxLife - $usedStrokes);
            $usagePercent = ($usedStrokes / $maxLife) * 100;
            
            $status = 'GOOD';
            if ($remaining <= 0) {
                $status = 'EXPIRED';
            } elseif ($remaining < 1000) {
                $status = 'CRITICAL';
            }
            
            $toolsData[] = [
                'id' => $tool['id'],
                'label' => $tool['label'],
                'current_usage' => $usedStrokes,
                'max_life' => $maxLife,
                'remaining' => max(0, $remaining),
                'usage_percent' => round($usagePercent, 1),
                'status' => $status
            ];
        }
        
        http_response_code(200);
        echo json_encode(['success' => true, 'tools' => $toolsData, 'total_tools' => count($toolsData)]);
    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode(['success' => false, 'error' => $e->getMessage()]);
    }
    exit;
}

// Unknown action
http_response_code(400);
echo json_encode([
    'success' => false, 
    'error' => "Unknown action: '$action'",
    'available_actions' => [
        'update_tool_stroke' => 'POST',
        'get_tool_status' => 'GET',
        'get_all_tools' => 'GET'
    ]
]);
?>
