# 🔗 Advanced Hierarchical Analytics System - Complete Guide

## 📋 Overview

The **Advanced Hierarchical Analytics System** provides personalized, machine-focused data analysis where:

- ✅ **Select a Machine** → System shows only related dependent fields
- ✅ **Each dependent field** has separate chart and statistics
- ✅ **Data is filtered** to that specific machine from production_logs
- ✅ **Complete employee tracking** for that machine
- ✅ **Modern, personalized interface** with 3-step hierarchy

---

## 🎯 Key Features

### 1. **Hierarchical Field Selection**
- **Step 1**: Select Machine Field (e.g., "Machine No")
- **Step 2**: Select Specific Machine (e.g., "CNC-01")
- **Step 3**: View machine statistics (operations, employees, days)

### 2. **Dependent Fields**
When you select a machine, the system shows only fields that depend on that machine:
- Fields with `parent_field_id` pointing to your machine field
- These appear as selectable cards
- Check/uncheck to toggle analysis

### 3. **Personalized Data Representation**
For each selected dependent field:
- **Bar Chart**: Visual distribution of field values
- **Data Table**: Exact counts and percentages
- **Progress Bars**: Visual representation of percentages
- **Statistics**: All filtered to the selected machine

### 4. **Machine-Specific Statistics**
Displays 4 key metrics:
- 📦 **Total Operations**: How many times machine was used
- 👥 **Employees**: How many people operated it
- 📅 **Working Days**: How many days it was in use
- ⚡ **Avg/Day**: Average operations per day

### 5. **Employee Assignment Tracking**
Shows all employees who worked on the selected machine:
- Employee ID and Name
- Number of operations they performed
- Percentage of total machine operations

---

## 🚀 How to Use

### Step 1: Access Advanced Analytics
**Option A - Via Sidebar:**
```
1. Click Sidebar Menu
2. Go to ANALYTICS section
3. Click "Advanced Analytics"
```

**Option B - Via Report Page:**
```
1. Go to View Reports
2. Click "Advanced" button (dark button)
3. Opens in new page
```

**Direct URL:**
```
http://localhost/erp.manufacture/analytics_advanced.php
```

### Step 2: Select Machine Field
1. Click dropdown "Select Machine Field"
2. Choose your machine field (e.g., "Machine No", "Equipment")
3. System loads dependent fields

### Step 3: Choose Specific Machine
1. Click dropdown "Select Specific Machine"
2. Choose machine (e.g., "CNC-01")
3. View quick statistics for that machine

### Step 4: Select Fields to Analyze
1. Dependent field cards appear below
2. Click cards to check/uncheck fields
3. Charts appear for selected fields

### Step 5: View Results
- Machine statistics cards (top)
- Field analysis charts and tables
- Employee assignment information

---

## 📊 Data Flow

```
┌─────────────────────────────────┐
│  User Opens analytics_advanced  │
└──────────────┬──────────────────┘
               ▼
    ┌──────────────────────────┐
    │ Load all active fields   │
    │ API: get_all_fields      │
    └──────────────┬───────────┘
                   ▼
    ┌──────────────────────────┐
    │ Populate machine field   │
    │ dropdown with parents    │
    └──────────────┬───────────┘
                   ▼
        ┌─ User Selects Machine Field
        │
        ▼
    ┌──────────────────────────┐
    │ Get dependent fields     │
    │ API: get_dependent_fields│
    └──────────────┬───────────┘
                   ▼
    ┌──────────────────────────┐
    │ Load all machines        │
    │ API: get_machine_summary │
    └──────────────┬───────────┘
                   ▼
    ┌──────────────────────────┐
    │ Show machine dropdown    │
    │ + dependent field cards  │
    └──────────────┬───────────┘
                   ▼
        ┌─ User Selects Machine
        │
        ▼
    ┌──────────────────────────────┐
    │ Load machine statistics      │
    │ API: get_machine_summary     │
    │ (Gets ops, employees, days)  │
    └──────────────┬───────────────┘
                   ▼
    ┌──────────────────────────────┐
    │ Display stats cards + load   │
    │ employees for machine        │
    │ API: get_machine_specific... │
    └──────────────┬───────────────┘
                   ▼
        ┌─ User Selects Dependent Fields
        │
        ▼
    ┌──────────────────────────────┐
    │ For each selected field:     │
    │ Get field statistics         │
    │ API: get_machine_specific... │
    └──────────────┬───────────────┘
                   ▼
    ┌──────────────────────────────┐
    │ Build chart + table for each │
    │ field filtered by machine    │
    └──────────────────────────────┘
```

---

## 🔧 API Endpoints (New)

### 1. `get_all_fields`
**Purpose**: Get parent and child field relationships
**Parameters**: None
**Returns**:
- `parent_fields`: All fields without parent (e.g., Machine No)
- `child_fields`: Fields grouped by parent ID
- `all_fields`: All active fields

**Example**:
```javascript
fetch('api.php?action=get_all_fields')
  .then(r => r.json())
  .then(data => {
    console.log(data.parent_fields);      // [Machine No, Equipment...]
    console.log(data.child_fields);       // {1: [Type, Speed...]}
  });
```

### 2. `get_dependent_fields`
**Purpose**: Get child fields for a specific parent field
**Parameters**: `parent_field_id` (required)
**Returns**: Array of dependent field objects

**Example**:
```javascript
fetch('api.php?action=get_dependent_fields&parent_field_id=5')
  .then(r => r.json())
  .then(data => {
    console.log(data.dependent_fields); // Fields that depend on field 5
  });
```

### 3. `get_machine_specific_stats`
**Purpose**: Get statistics for a field filtered by specific machine value
**Parameters**:
- `machine_field`: Field name containing machine (e.g., f_machine_no)
- `machine_value`: Specific machine value (e.g., "CNC-01")
- `dependent_field`: Field to analyze (e.g., f_production_type)

**Returns**:
- Field label and type
- Data distribution (value, count)
- Total records for this machine
- Employees working on machine

**Example**:
```javascript
fetch('api.php?action=get_machine_specific_stats&machine_field=f_machine_no&machine_value=CNC-01&dependent_field=f_production_type')
  .then(r => r.json())
  .then(data => {
    console.log(data.data);              // [{value: "Type1", count: 45}...]
    console.log(data.total_records);     // 245
    console.log(data.employees_working); // [{emp_id, operator_name, count}...]
  });
```

### 4. `get_machine_summary`
**Purpose**: Get summary statistics for all machines in a field
**Parameters**: `machine_field` (e.g., f_machine_no)
**Returns**: Array of machines with:
- `machine_name`: Machine name
- `total_operations`: Total times used
- `employee_count`: How many employees used it
- `working_days`: How many days it operated
- `first_used`: First usage date
- `last_used`: Last usage date

**Example**:
```javascript
fetch('api.php?action=get_machine_summary&machine_field=f_machine_no')
  .then(r => r.json())
  .then(data => {
    data.machines.forEach(m => {
      console.log(`${m.machine_name}: ${m.total_operations} ops, ${m.employee_count} employees`);
    });
  });
```

---

## 📈 Chart & Table Visualizations

### For Each Dependent Field:

#### 1. **Bar Chart**
```
Field Distribution (Top 10 values)
┌─────────────────────────────────────┐
│ Type1    ███████████ 45 records     │
│ Type2    ████████ 32 records        │
│ Type3    ██████ 28 records          │
│ Type4    ███ 15 records             │
└─────────────────────────────────────┘
```

#### 2. **Data Table**
```
┌────────┬────────┬────────────┐
│ Value  │ Count  │ Percentage │
├────────┼────────┼────────────┤
│ Type1  │  45    │ 18.4%  ██  │
│ Type2  │  32    │ 13.1%  █   │
│ Type3  │  28    │ 11.4%  █   │
└────────┴────────┴────────────┘
```

---

## 🎨 UI Components

### Machine Statistics Cards (After Selection)
```
┌──────────────┐ ┌──────────────┐ ┌──────────────┐ ┌──────────────┐
│ 📦           │ │ 👥           │ │ 📅           │ │ ⚡           │
│ Operations   │ │ Employees    │ │ Working Days │ │ Avg/Day      │
│ 1,245        │ │ 12           │ │ 42           │ │ 29.6         │
└──────────────┘ └──────────────┘ └──────────────┘ └──────────────┘
```

### Dependent Field Selection Cards
```
┌──────────────────────┐  ┌──────────────────────┐
│ ☐ Production Type    │  │ ☑ Quality Status     │
│   Dropdown          │  │   Selected ✓        │
└──────────────────────┘  └──────────────────────┘
```

---

## 💡 Sample Workflows

### Workflow 1: Analyze CNC-01 Machine
1. Open Advanced Analytics
2. Select "Machine No" from field dropdown
3. Select "CNC-01" from machine dropdown
4. See: 1,245 operations, 12 employees, 42 days, 29.6 avg/day
5. Check "Production Type" field
6. View how many of each type were produced on CNC-01
7. Check "Quality Status"
8. View quality distribution for CNC-01 only

### Workflow 2: Compare Field Values Across Machine
1. Select machine field
2. Select specific machine (e.g., Lathe-02)
3. Select 3 dependent fields
4. Get 3 separate charts showing data specific to Lathe-02
5. Identify patterns in that machine's usage

### Workflow 3: Employee Performance on Specific Machine
1. Select machine
2. View "Employees Working on This Machine" table
3. See who operates it and how many times
4. Identify high performers on that specific machine

---

## 🎯 Features in Detail

### Hierarchical Selection (3 Steps)

**Step 1: Select Machine Field**
- Shows all parent fields (fields without parent_field_id)
- Choose the field containing machines/tools
- Example: "Machine No", "Equipment Type"

**Step 2: Select Specific Machine**
- Shows all unique values in that field
- Sorted by operation count
- Display format: "CNC-01 (245 ops)"

**Step 3: View Quick Stats**
- Shows 4 key metrics instantly
- Total operations, employee count, working days, average

### Dynamic Dependent Fields

When machine is selected:
- Query fetches all fields with `parent_field_id = machine_field_id`
- Display as selectable cards with checkboxes
- Click card to toggle analysis
- Multiple fields can be selected simultaneously

### Personalized Statistics

For each selected field on chosen machine:
- Query filters by: `WHERE machine_field = 'CNC-01'`
- Shows only data for that specific machine
- Multiple independent charts/tables
- Each with own data and summary

### Employee Tracking

Automatically loaded for selected machine:
- Lists all employees who worked on that machine
- Count of operations per employee
- Percentage of total machine usage
- Sorted by operation count

---

## 🔍 Example Data Scenarios

### Scenario 1: CNC-01 Production Analysis

**Selection**: Machine Field = "Machine No", Machine = "CNC-01"

**Dependent Fields**: Production Type, Quality Status

**Results**:
```
CNC-01 Statistics:
├─ Total Operations: 1,245
├─ Employees: 12
├─ Working Days: 42
└─ Avg/Day: 29.6

Production Type on CNC-01:
├─ Standard: 845 (67.9%)
├─ Premium: 285 (22.9%)
└─ Custom: 115 (9.2%)

Quality Status on CNC-01:
├─ Accepted: 1,180 (94.8%)
├─ Rejected: 45 (3.6%)
└─ Rework: 20 (1.6%)

Employees on CNC-01:
├─ John Doe: 245 ops (19.7%)
├─ Sarah Smith: 198 ops (15.9%)
└─ ... (10 more)
```

---

## 📊 Database Queries Used

### Get Dependent Fields
```sql
SELECT * FROM custom_fields 
WHERE parent_field_id = ? AND is_active = 1
ORDER BY display_order ASC
```

### Get Machine-Specific Stats
```sql
SELECT `field_name` as value, COUNT(*) as count
FROM production_logs
WHERE `machine_field` = ?
AND `dependent_field` IS NOT NULL AND `dependent_field` != ''
GROUP BY `dependent_field`
ORDER BY count DESC
```

### Get Machine Summary
```sql
SELECT 
  `machine_field` as machine_name,
  COUNT(*) as total_operations,
  COUNT(DISTINCT emp_id) as employee_count,
  COUNT(DISTINCT log_date) as working_days,
  MIN(log_date) as first_used,
  MAX(log_date) as last_used
FROM production_logs
WHERE `machine_field` IS NOT NULL AND `machine_field` != ''
GROUP BY `machine_field`
ORDER BY total_operations DESC
```

---

## ⚙️ Configuration

### Field Hierarchy Setup

In your Admin → Manage Fields:

1. **Create Parent Field** (Machine)
   - Field: "Machine No"
   - Type: "Select"
   - Parent: (empty)

2. **Create Dependent Fields**
   - Field: "Production Type"
   - Parent: "Machine No" (select from dropdown)
   
   - Field: "Quality Status"
   - Parent: "Machine No"
   
   - Field: "Operator Shift"
   - Parent: "Machine No"

3. **Enable All Fields**
   - Toggle to Active

Now these dependent fields appear in Advanced Analytics!

---

## 🔐 Security

- ✅ Session validation required
- ✅ SQL parameterized queries
- ✅ No direct SQL injection possible
- ✅ Read-only operations only
- ✅ Field names escaped with backticks
- ✅ User must be logged in

---

## 🐛 Troubleshooting

### Dependent Fields Not Showing
- Check that fields have correct `parent_field_id`
- Verify parent field is selected
- Ensure dependent fields are marked as Active
- Check browser console for errors

### Charts Not Rendering
- Verify machine value has data in production_logs
- Check dependent field contains non-null values
- Refresh page and try again
- Check Chart.js is loaded

### No Data in Tables
- Confirm production logs exist for that machine
- Verify field contains data (not empty/null)
- Try different machine/field combination
- Check database connectivity

### Machine List Empty
- Ensure machine field contains data
- Verify it's marked as active field
- Check production_logs has entries
- Try refreshing page

---

## 📚 Files Created/Modified

### Created:
- **analytics_advanced.php** - Advanced hierarchical analytics page

### Modified:
- **api.php** - Added 4 new endpoints
- **view_log.php** - Added "Advanced" button
- **sidebar.js** - Added Advanced Analytics menu item

### Unchanged:
- All existing functionality preserved
- Database schema compatible
- No breaking changes

---

## ✨ Key Advantages

| Feature | Benefit |
|---------|---------|
| Hierarchical Selection | Intuitive 3-step process |
| Dependent Fields | Only relevant data shown |
| Personalized Stats | Data filtered by machine |
| Multiple Analyses | Select multiple fields at once |
| Employee Tracking | See who worked on what |
| Rich Visualizations | Charts + Tables + Progress bars |
| Real-time Data | Fresh queries each time |
| Mobile Responsive | Works on all devices |

---

## 🎓 For Developers

### Adding New Dependent Fields

1. In Admin → Manage Fields, set `parent_field_id`
2. Fields automatically appear in Advanced Analytics
3. No code changes needed!

### Creating Custom Reports

1. Use API endpoints directly
2. Build custom JavaScript
3. No database schema changes required

### Extending Functionality

```php
// Example: Custom machine metric
if ($action === 'get_machine_efficiency') {
    $machineField = $_GET['machine_field'];
    $machineValue = $_GET['machine_value'];
    
    $query = "SELECT 
        COUNT(*) as total,
        SUM(CASE WHEN status='accepted' THEN 1 ELSE 0 END) as accepted,
        (SUM(CASE WHEN status='accepted' THEN 1 ELSE 0 END) * 100.0 / COUNT(*)) as efficiency
    FROM production_logs
    WHERE `$machineField` = ?";
    
    // Execute and return JSON
}
```

---

## 📞 Quick Links

- **Main Dashboard**: `analytics.php`
- **Advanced Analytics**: `analytics_advanced.php`
- **Quick Analytics**: View Reports → Click Analytics button
- **API Docs**: Check api.php around line 720+
- **Field Setup**: Admin → Manage Fields

---

**System Version**: 2.0 (Advanced)  
**Last Updated**: January 2026  
**Status**: ✅ Complete and Production Ready

Enjoy personalized, hierarchical analytics! 🚀
