# Advanced Analytics System - Complete Implementation Guide

## Overview

A comprehensive analytics dashboard has been implemented with day-wise, field-wise, tool-wise, and employee-wise production analysis with interactive visualizations.

---

## New API Endpoints

### 1. **Day-Wise Production Analytics**
**Endpoint:** `api.php?action=get_daywise_analytics`

**Response:** Returns last 30 days of production data grouped by date
```json
[
  {
    "date": "2024-01-20",
    "total_entries": 15,
    "total_plan_qty": "500.00",
    "total_produced_qty": "450.00",
    "total_accepted_qty": "420.00",
    "nc_qty": "30.00"
  }
]
```

### 2. **Field/Tool-Wise Analytics**
**Endpoint:** `api.php?action=get_fieldwise_analytics`

**Response:** Production data grouped by tool/field
```json
[
  {
    "field_name": "TOOL-001",
    "total_entries": 25,
    "worker_count": 3,
    "total_plan_qty": "1200.00",
    "total_produced_qty": "1100.00",
    "total_accepted_qty": "1050.00",
    "nc_qty": "50.00"
  }
]
```

### 3. **Employee-Wise Analytics**
**Endpoint:** `api.php?action=get_employeewise_analytics`

**Response:** Individual employee performance metrics
```json
[
  {
    "employee_name": "John Doe",
    "total_entries": 45,
    "working_days": 10,
    "tools_handled": 3,
    "total_plan_qty": "2000.00",
    "total_produced_qty": "1900.00",
    "total_accepted_qty": "1800.00",
    "nc_qty": "100.00",
    "quality_rate": "94.74"
  }
]
```

### 4. **Tool-Wise Worker Analysis**
**Endpoint:** `api.php?action=get_tool_worker_analytics&tool=TOOL-001`

**Response:** Workers assigned to a specific tool with their production metrics
```json
[
  {
    "worker_name": "John Doe",
    "entries": 15,
    "days": 5,
    "plan_qty": "600.00",
    "produced_qty": "550.00",
    "accepted_qty": "525.00",
    "nc_qty": "25.00"
  }
]
```

### 5. **Employee Detailed Analytics**
**Endpoint:** `api.php?action=get_employee_full_analytics&employee=John%20Doe`

**Response:** Detailed daily breakdown for an employee
```json
[
  {
    "date": "2024-01-20",
    "tool_no": "TOOL-001",
    "operation": "Cutting",
    "part_name": "Panel A",
    "plan_qty": "100.00",
    "produced_qty": "95.00",
    "accepted_qty": "90.00",
    "nc_qty": "5.00",
    "quality_rate": "94.74"
  }
]
```

---

## JavaScript Analytics Functions

### **Data Fetching Functions**

```javascript
// Fetch day-wise analytics and execute callback
fetchDayWiseAnalytics(callback)

// Fetch field-wise analytics and execute callback
fetchFieldWiseAnalytics(callback)

// Fetch employee-wise analytics and execute callback
fetchEmployeeWiseAnalytics(callback)

// Fetch tool-worker analytics for a specific tool
fetchToolWorkerAnalytics(toolName, callback)

// Fetch detailed analytics for an employee
fetchEmployeeDetailedAnalytics(employeeName, callback)
```

### **Rendering Functions**

```javascript
// Render day-wise chart with plan, produced, accepted, NC quantities
renderDayWiseChart(containerId, data)

// Render field-wise horizontal bar chart
renderFieldWiseChart(containerId, data)

// Render employee-wise analytics table (top 15)
renderEmployeeWiseTable(containerId, data)

// Render detailed employee analytics with summary cards
renderEmployeeDetailedAnalyticsChart(containerId, data)

// Render tool-worker breakdown table
renderToolWorkerBreakdown(containerId, data)
```

---

## Pages with Analytics

### 1. **Analytics Dashboard** (`analytics.php`)

**Features:**
- **Day-Wise Chart:** Bar chart showing Plan Qty, Produced Qty, Accepted Qty, and NC Qty for last 30 days
- **Field-Wise Chart:** Horizontal bar chart showing production by tool/field
- **Employee-Wise Table:** Summary table with top 15 employees showing:
  - Working days
  - Tools handled
  - Quantities (Plan, Produced, Accepted, NC)
  - Quality rate percentage

**Access:** `http://localhost/erp.manufacture/analytics.php`

### 2. **Employee Search Page** (`search_emp.php`)

**Features:**
- **Employee Search:** Find employee by ID or name with auto-fill
- **Work History:** Displays all work logs with dynamic columns
- **Production Analytics:** Shows employee-specific metrics:
  - Summary cards: Total Plan, Produced, Accepted, NC Qty + Quality % + Efficiency %
  - Detailed table with date-wise breakdown by tool and operation
  - Shows Plan, Produced, Accepted, NC, and Quality % for each entry

**Access:** `http://localhost/erp.manufacture/search_emp.php`

**How to Use:**
1. Enter Employee ID or Name in search fields
2. Click "Search" or press Enter
3. View employee profile and work history
4. Scroll down to see production analytics summary and detailed breakdown

---

## Data Calculations

### **NC Quantity (Non-Conforming)**
```
NC Qty = Produced Qty - Accepted Qty
```

### **Quality Rate**
```
Quality Rate % = (Accepted Qty / Produced Qty) × 100
```

### **Efficiency**
```
Efficiency % = (Produced Qty / Plan Qty) × 100
```

---

## Features & Capabilities

### ✅ **Complete Analytics Coverage**

| Metric | Day-Wise | Field-Wise | Employee-Wise | Tool-Worker | Search Page |
|--------|----------|-----------|--------------|------------|------------|
| Plan Quantity | ✓ | ✓ | ✓ | ✓ | ✓ |
| Produced Quantity | ✓ | ✓ | ✓ | ✓ | ✓ |
| Accepted Quantity | ✓ | ✓ | ✓ | ✓ | ✓ |
| NC Quantity | ✓ | ✓ | ✓ | ✓ | ✓ |
| Quality Rate | - | - | ✓ | - | ✓ |
| Efficiency % | - | - | - | - | ✓ |
| Worker Count | - | ✓ | - | - | - |
| Tools Handled | - | - | ✓ | - | - |
| Working Days | - | - | ✓ | ✓ | - |

### 🎨 **Visual Representations**

- **Bar Charts:** Day-wise and Field-wise production metrics with 4 data series
- **Summary Cards:** Color-coded cards showing totals and percentages
- **Data Tables:** Interactive tables with hover effects
- **Color Coding:** 
  - Blue: Plan Qty
  - Green: Produced Qty
  - Orange: Accepted Qty
  - Red: NC Qty

---

## Database Implementation

All analytics functions use **direct database aggregation** with SQL GROUP BY queries:

```php
// Example: Day-wise aggregation
SELECT log_date as date,
       SUM(CAST(plan_qty AS DECIMAL(10,2))) as total_plan_qty,
       SUM(CAST(produced_qty AS DECIMAL(10,2))) as total_produced_qty,
       SUM(CAST(accepted_qty AS DECIMAL(10,2))) as total_accepted_qty,
       SUM(produced_qty - accepted_qty) as nc_qty
FROM production_logs 
GROUP BY log_date 
ORDER BY log_date DESC LIMIT 30
```

---

## Performance Notes

- **Data Limit:** Day-wise shows last 30 days; Field-wise shows top 10 fields
- **Employee Analytics:** Employee-wise table shows top 15 employees; Detailed table shows top 20 entries
- **Auto-Load:** All analytics automatically load when dashboard initializes
- **Responsive:** Charts adjust to container size and are mobile-friendly

---

## Advanced Usage

### **Custom Analytics Query**

To fetch tool-specific worker analytics:
```javascript
fetchToolWorkerAnalytics('TOOL-001', function(data) {
    renderToolWorkerBreakdown('myContainer', data);
});
```

### **Real-Time Updates**

To refresh analytics periodically:
```javascript
setInterval(() => {
    fetchDayWiseAnalytics(function(data) {
        renderDayWiseChart('dayWiseChartContainer', data);
    });
}, 300000); // Refresh every 5 minutes
```

### **Filtered Analytics**

To extend analytics with date range filtering, modify the SQL in `api.php`:
```php
WHERE log_date BETWEEN ? AND ?
// Add date parameters to prepared statement
```

---

## Troubleshooting

### **Charts Not Displaying**
1. Check browser console for errors
2. Verify Chart.js is loaded: `Chart.version` in console
3. Ensure container elements exist in DOM
4. Check API responses: `api.php?action=get_daywise_analytics`

### **Empty Analytics**
1. Verify production_logs table has data
2. Check that custom columns exist (plan_qty, produced_qty, accepted_qty)
3. Review API response for NULL values

### **Employee Analytics Not Loading**
1. Ensure employee name in search matches database exactly
2. Check employee name format in work logs (operator_name field)
3. Verify employee has production log entries

---

## Future Enhancements

- [ ] Date range filtering for all analytics
- [ ] Export to CSV/Excel functionality
- [ ] Drill-down capability from summary to detailed data
- [ ] Real-time analytics updates
- [ ] Comparative analysis (month-over-month, year-over-year)
- [ ] Goal vs Actual tracking
- [ ] Predictive analytics and forecasting
- [ ] Custom report builder

---

## Summary

The advanced analytics system provides comprehensive production insights with:
✅ 5 new API endpoints for data aggregation
✅ 10 visualization and rendering functions
✅ 2 pages with integrated analytics (Analytics Dashboard + Search Page)
✅ 4 quantity metrics tracked (Plan, Produced, Accepted, NC)
✅ Quality and Efficiency calculations
✅ Interactive charts and tables
✅ Employee-specific performance analytics
✅ Tool-wise worker analysis

All data is derived from the production_logs table with direct SQL aggregation for optimal performance.
