# Notification System - Setup & Verification Guide

## 📋 Pre-Setup Checklist

Before integrating the notification system, verify you have:

- [ ] PHP 7.0+ (for MySQL prepared statements)
- [ ] MySQL/MariaDB database access
- [ ] Admin rights to create tables
- [ ] Access to modify PHP files
- [ ] Browser supporting ES6 JavaScript
- [ ] FontAwesome icons available

---

## 🔧 Setup Steps

### Step 1: Create Database Table

Run the SQL setup script:

```bash
mysql -u your_username -p your_database < NOTIFICATION_SYSTEM_SETUP.sql
```

Or manually execute in phpMyAdmin:

```sql
CREATE TABLE IF NOT EXISTS notification_settings (
    id INT PRIMARY KEY AUTO_INCREMENT,
    setting_key VARCHAR(50) UNIQUE NOT NULL,
    setting_value INT NOT NULL DEFAULT 0,
    description TEXT,
    last_updated TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
);

INSERT INTO notification_settings (setting_key, setting_value, description) VALUES
('machine_days_threshold', 15, 'Days before machine maintenance alert (default: 15 days)'),
('tool_strokes_threshold', 1000, 'Remaining strokes before tool alert (default: 1000 strokes)');
```

**Verify**: Check in phpMyAdmin that the table exists and has 2 rows

### Step 2: Copy Notification Files

Copy these files to your project root:

```
notification_ui.html ✓
notifications.php    ✓
```

Verify files exist:
```bash
ls -la notification*
```

### Step 3: Update Admin Dashboard

Update your `index.php`:

1. **Add notification button to page-header div**:
```html
<?php if (isset($_SESSION['admin']) && $_SESSION['admin'] == true) { ?>
<button class="notification-btn" id="notificationBtn" title="View Notifications" onclick="toggleNotificationSidebar()">
    <i class="fas fa-bell"></i>
</button>
<?php } ?>
```

2. **Add notification CSS to `<style>` section**:
```css
/* Notification Button Styling */
.notification-btn {
    position: relative;
    background: none;
    border: none;
    font-size: 20px;
    cursor: pointer;
    color: #333;
    padding: 8px 12px;
    border-radius: 4px;
    transition: all 0.3s;
}

.notification-btn:hover {
    background: rgba(0,0,0,0.05);
    color: #e74c3c;
    transform: scale(1.1);
}

.notification-badge {
    position: absolute;
    top: 0;
    right: 0;
    background: #e74c3c;
    color: white;
    border-radius: 50%;
    width: 24px;
    height: 24px;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 12px;
    font-weight: bold;
    animation: pulse-badge 2s infinite;
}

@keyframes pulse-badge {
    0%, 100% { transform: scale(1); }
    50% { transform: scale(1.1); }
}
```

3. **Include notification UI before closing body tag**:
```html
<!-- Include Notification UI Component (Admin Only) -->
<?php if (isset($_SESSION['admin']) && $_SESSION['admin'] == true) { ?>
<?php include 'notification_ui.html'; ?>
<?php } ?>
</body>
</html>
```

### Step 4: Verify FontAwesome

Ensure FontAwesome is loaded in your `<head>`:

```html
<link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
```

Or check your existing stylesheet imports.

### Step 5: Test the Setup

1. **Access admin dashboard**:
   ```
   http://localhost/erp.manufacture/index.php
   ```

2. **Check for bell icon** in top-right corner

3. **Verify database connection**:
   - Open browser DevTools (F12)
   - Go to Network tab
   - Click bell icon
   - Check that notifications.php returns JSON

4. **Test notification loading**:
   - Should show "No notifications" or list alerts
   - Badge should update with count

---

## ✅ Verification Tests

### Test 1: Admin-Only Access

**Objective**: Verify bell icon only shows for admins

```bash
# Test as regular user
Login as non-admin user
→ Navigate to dashboard
→ Bell icon should NOT appear

# Test as admin
Login as admin user
→ Navigate to dashboard
→ Bell icon SHOULD appear
```

**Expected Result**: ✅ Icon visible only for admins

### Test 2: Notification Loading

**Objective**: Verify notifications load from database

```bash
# With browser DevTools open (F12)
→ Click bell icon
→ Check Network tab
→ notifications.php?action=get_notifications request
→ Should return JSON with notifications array
→ Status code should be 200
```

**Expected Result**: ✅ JSON response received successfully

### Test 3: Sidebar Functionality

**Objective**: Verify sidebar opens/closes smoothly

```bash
→ Click bell icon
→ Sidebar should slide in from right
→ Overlay should appear
→ Click close (X) or overlay
→ Sidebar should slide out
→ Overlay should disappear
```

**Expected Result**: ✅ Smooth animations, no layout shift

### Test 4: Notification Rendering

**Objective**: Verify notifications display correctly

If notifications exist in database:
```bash
→ Open sidebar
→ Check notification items display:
  - Icon (machine/tool)
  - Title
  - Severity badge
  - Message
  - Item name
  - Timestamp
  - "Go to Maintenance" button
```

**Expected Result**: ✅ All elements visible and formatted

### Test 5: Settings Update

**Objective**: Verify settings can be changed

```bash
→ Open sidebar
→ Click "Settings" button
→ Settings modal appears (centered)
→ Input fields show current values
→ Change machine threshold to 20
→ Change tool threshold to 500
→ Click "Save Settings"
→ Confirmation message appears
→ Check browser DevTools Network tab
→ POST to notifications.php should succeed
```

**Expected Result**: ✅ Settings updated in database

### Test 6: Auto-Refresh

**Objective**: Verify notifications update every 30 seconds

```bash
→ Keep sidebar open
→ Watch Network tab in DevTools
→ Wait 30+ seconds
→ New GET request to notifications.php should appear
→ Repeat 2-3 times to confirm interval
```

**Expected Result**: ✅ Requests appear approximately every 30 seconds

### Test 7: Go to Maintenance Button

**Objective**: Verify action button navigates correctly

```bash
→ Click "Go to Maintenance" on any notification
→ Should redirect to maintenance.php
→ Page should load successfully
→ Tool/machine details should be visible
```

**Expected Result**: ✅ Navigation works and page loads

### Test 8: Clear All Function

**Objective**: Verify clear button clears notifications

```bash
→ Open sidebar with notifications present
→ Click "Clear All" button
→ Confirmation dialog appears
→ Click "OK"
→ Notifications should disappear from sidebar
→ Display message: "No notifications at the moment"
→ Badge on bell should disappear
```

**Expected Result**: ✅ All notifications cleared from view

---

## 🐛 Debugging Checklist

If tests fail, troubleshoot:

### Bell Icon Doesn't Appear

```php
// Check session_check.php
<?php
    if (isset($_SESSION['admin']) && $_SESSION['admin'] == true) {
        // This should be true for admins
        echo "Admin status confirmed";
    }
?>
```

**Solutions**:
- [ ] Verify admin is logged in
- [ ] Check session_check.php sets $_SESSION['admin']
- [ ] Clear browser cookies and re-login
- [ ] Check PHP error logs

### Sidebar Doesn't Open

```javascript
// Check browser DevTools Console (F12 → Console)
// Look for JavaScript errors
```

**Solutions**:
- [ ] Verify notification_ui.html is included
- [ ] Check for JavaScript syntax errors in console
- [ ] Verify CSS is loading (check for .notification-sidebar class)
- [ ] Clear browser cache (Ctrl+Shift+Del)

### Notifications Don't Load

```php
// Test API directly
$_SESSION['admin'] = true;
// Access notifications.php?action=get_notifications
// Should return JSON
```

**Solutions**:
- [ ] Run NOTIFICATION_SYSTEM_SETUP.sql
- [ ] Verify database connection in notifications.php
- [ ] Check notification_settings table exists
- [ ] Verify custom_fields and field_options tables exist
- [ ] Check MySQL error logs

### Settings Won't Save

```php
// Test POST request in Network tab
// Check response for error messages
```

**Solutions**:
- [ ] Verify form data is sent correctly
- [ ] Check number inputs contain valid values
- [ ] Verify database has write permissions
- [ ] Check error logs for SQL errors
- [ ] Verify admin authentication passes

### Database Connection Error

**Solutions**:
- [ ] Test database connection:
  ```php
  // In test.php
  try {
      $pdo = new PDO('mysql:host=localhost;dbname=db_name', 'user', 'pass');
      echo "Connected successfully";
  } catch(PDOException $e) {
      echo "Error: " . $e->getMessage();
  }
  ```

- [ ] Verify credentials in notifications.php
- [ ] Check MySQL is running
- [ ] Verify user has database access

---

## 📊 Database Verification

### Check Table Creation

```sql
DESCRIBE notification_settings;
```

**Expected Output**:
```
id | int | PRIMARY KEY
setting_key | varchar(50) | UNIQUE
setting_value | int | 
description | text | 
last_updated | timestamp | 
```

### Check Data

```sql
SELECT * FROM notification_settings;
```

**Expected Output**:
```
1 | machine_days_threshold | 15 | Days before machine maintenance alert
2 | tool_strokes_threshold | 1000 | Remaining strokes before tool alert
```

### Check Related Tables

```sql
-- Verify machines table exists
SELECT COUNT(*) FROM machines;

-- Verify custom_fields exists (for tools)
SELECT COUNT(*) FROM custom_fields WHERE field_type = 'NUMBER';

-- Verify field_options exists (for tool data)
SELECT COUNT(*) FROM field_options;
```

---

## 🚀 Post-Setup Verification

After setup, verify with this checklist:

- [ ] Database tables created
- [ ] notification_ui.html file present
- [ ] notifications.php file present
- [ ] index.php updated with bell button
- [ ] index.php includes notification UI
- [ ] FontAwesome CSS loaded
- [ ] Admin can see bell icon
- [ ] Sidebar opens/closes smoothly
- [ ] Settings update successfully
- [ ] Notifications auto-refresh
- [ ] Navigation works
- [ ] No JavaScript errors in console

---

## 📝 Common Setup Issues

### Issue: "notification_settings table doesn't exist"

```
Error: SQLSTATE[42S02]: Table 'db_name.notification_settings' doesn't exist
```

**Solution**:
```bash
# Run setup SQL
mysql -u user -p database < NOTIFICATION_SYSTEM_SETUP.sql

# Or manually create:
# Copy SQL from NOTIFICATION_SYSTEM_SETUP.sql
# Execute in phpMyAdmin
```

### Issue: Bell icon shows but sidebar doesn't open

**Solution**:
```
1. Check notification_ui.html is included
2. Look for JavaScript errors (F12 → Console)
3. Verify CSS is loaded properly
4. Check z-index of modal elements
5. Clear cache and refresh
```

### Issue: Settings modal appears but won't load current values

**Solution**:
```
1. Check that notifications.php?action=get_notification_settings returns data
2. Verify JSON response is valid
3. Check browser console for JavaScript errors
4. Verify admin user has database read permissions
```

### Issue: Notifications appear but always show "No notifications"

**Solution**:
1. Check if machines/tools meet alert criteria
2. Verify threshold values in notification_settings
3. Check if next_due_date is calculated correctly for machines
4. Verify current_usage is updated for tools
5. Run query directly in phpMyAdmin:
   ```sql
   SELECT * FROM machines 
   WHERE DATE_SUB(
       DATE_ADD(last_service, INTERVAL life_limit DAY), 
       INTERVAL (SELECT setting_value FROM notification_settings WHERE setting_key='machine_days_threshold') DAY
   ) <= CURDATE();
   ```

---

## 🔍 Performance Monitoring

Monitor these metrics:

```javascript
// In browser DevTools → Performance tab
// Monitor:
// - Time to open sidebar: < 300ms
// - Time to render notifications: < 500ms
// - API response time: < 1000ms
// - Memory usage: < 10MB
```

---

## 📞 Support Resources

**If issues persist**:
1. Check browser DevTools Console (F12) for error messages
2. Check PHP error logs: `/var/log/php-errors.log` or `php_error.log`
3. Check MySQL error logs: `/var/log/mysql/error.log`
4. Review NOTIFICATION_ADMIN_GUIDE.md for troubleshooting
5. Review notifications.php code for logic issues

**Files Updated**:
- `index.php` - Added bell button and notification UI include
- `notification_ui.html` - Notification component (new)
- `notifications.php` - Backend API (new)

**Database**:
- `notification_settings` table created with defaults

---

## ✨ You're Ready!

Once all tests pass, the notification system is ready for production use:

1. ✅ Database configured
2. ✅ Files in place
3. ✅ Admin dashboard updated
4. ✅ All tests passing
5. ✅ No errors in logs

**Start using notifications**:
- Admin users see bell icon in dashboard
- Click to view machine and tool alerts
- Adjust thresholds as needed
- System auto-refreshes every 30 seconds

