# Cart & Checkout System Integration

## Overview
Complete implementation of a professional cart and checkout system with session-based cart management, CRUD operations, and seamless flow from shopping to payment.

---

## System Architecture

### Flow Diagram
```
online-delivery.php (Browse Items)
        ↓
      ADD TO CART
        ↓
cart-view.php (View Cart with CRUD)
        ↓
  PROCEED TO CHECKOUT
        ↓
checkout-page.php (Delivery & Payment)
        ↓
   PLACE ORDER
```

### Data Storage
- **Session-based**: Cart items stored in `$_SESSION['cart']`
- **Database backup**: Cart items also stored in MySQL `cart` table
- **Hybrid approach**: Ensures cart persistence even if session expires

---

## Implementation Details

### 1. cart-actions.php (Backend Cart Management)

**Initialization:**
```php
session_start();
if (!isset($_SESSION['cart'])) {
    $_SESSION['cart'] = [];
}
```

**Supported Actions:**

#### Fetch Cart
- **Action**: `fetch`
- **Returns**: Array of cart items from both session and database
- **Usage**: Called on page load to display cart

#### Add to Cart
- **Action**: `add`
- **Parameters**: `name`, `price`, `image`
- **Behavior**: 
  - Stores in session
  - Increments quantity if item exists
  - Also stores in database for logged-in users
  - **Response**: JSON success status

```php
// Example request
formData.append('action', 'add');
formData.append('name', 'Biryani');
formData.append('price', '250');
formData.append('image', 'biryani.jpg');
```

#### Update Quantity
- **Action**: `update_qty`
- **Parameters**: `id`, `qty`, `name`
- **Behavior**: 
  - Updates quantity in session
  - Updates quantity in database
  - Removes item if qty < 1

#### Remove Item
- **Action**: `remove`
- **Parameters**: `id`, `name`
- **Behavior**: Removes from session and database

---

### 2. cart-view.php (Cart Display & CRUD UI)

#### Features

**Cart Display:**
- ✅ Professional table layout with item images
- ✅ Item details (name, price, quantity)
- ✅ Real-time price calculations
- ✅ Cart item counter in header

**CRUD Operations:**
```javascript
updateQty(id, newQty, name)      // Increase/Decrease quantity
removeItem(id, name)              // Remove item with confirmation
loadCart()                         // Fetch and display cart
updateSummary(subtotal)           // Calculate totals
```

**Order Summary:**
- Subtotal calculation
- Delivery charge (₹40 for delivery, ₹0 for takeaway)
- Dynamic discount display (₹10)
- Final grand total
- Order type toggle (Delivery/Takeaway)

**UI/UX Features:**
- Sweet Alert confirmations for destructive actions
- Smooth loading states
- Responsive design (mobile-friendly)
- Professional gradient buttons
- Real-time updates with visual feedback

#### Styling Highlights
```css
.cart-item-row:hover { background-color: #f9f9f9; }
.btn-outline-secondary:hover { 
    background-color: #667eea;
    border-color: #667eea;
}
#checkoutBtn {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    box-shadow: 0 6px 20px rgba(102, 126, 234, 0.4);
}
```

---

### 3. checkout-page.php (Checkout & Payment)

#### Session Integration
```php
<?php
session_start();
// Check user login
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit;
}
// Get cart from session
$cartItems = isset($_SESSION['cart']) ? $_SESSION['cart'] : [];
?>
```

#### Features

**Cart Display:**
- Loads items directly from session (`$_SESSION['cart']`)
- Shows product image, name, quantity, price
- Real-time total calculation
- "Add More Items" button links to `online-delivery.php`

**Delivery Address Form:**
- Full Name
- Phone Number (10-digit validation)
- Street Address
- City & State
- Postal Code (6-digit validation)
- Special Instructions (optional)

**Payment Methods:**
```
1. UPI (Google Pay, PhonePe, Paytm)
   - UPI ID input field
   
2. Card (Visa, Mastercard, RuPay)
   - Card holder name
   - Card number (16 digits)
   - Expiry (MM/YY)
   - CVV (3 digits)
   
3. Net Banking
   - 8 major Indian banks dropdown
   
4. Digital Wallet
   - Paytm, Amazon Pay, MobiKwik
   
5. Cash on Delivery
   - Default option with info message
```

**Order Summary (Sticky Sidebar):**
- Cart items list with scrollable area
- Price breakdown:
  - Subtotal
  - Delivery charge
  - Taxes (5%)
  - Discount
  - Grand total
- Promo code input with apply button
- Safety notice

**Form Validation:**
```javascript
✓ All address fields required
✓ Phone number: 10 digits
✓ Postal code: 6 digits
✓ Cart not empty
✓ Terms & conditions accepted
✓ Payment method selected
```

---

## CRUD Operations Summary

### Create (Add to Cart)
```javascript
// From online-delivery.php
const formData = new FormData();
formData.append('action', 'add');
formData.append('name', 'Biryani');
formData.append('price', '250');
formData.append('image', 'assets/images/biryani.jpg');

fetch('cart-actions.php', {method: 'POST', body: formData})
    .then(response => response.json())
    .then(data => {
        if (data.status === 'success') {
            // Item added successfully
            showCartCount();
        }
    });
```

### Read (Fetch Cart)
```javascript
// Auto-loads on page load
function loadCart() {
    const formData = new FormData();
    formData.append('action', 'fetch');
    
    fetch('cart-actions.php', {method: 'POST', body: formData})
        .then(response => response.json())
        .then(data => renderCart(data));
}
```

### Update (Change Quantity)
```javascript
// Click +/- buttons
function updateQty(id, newQty, name) {
    const formData = new FormData();
    formData.append('action', 'update_qty');
    formData.append('id', id);
    formData.append('qty', newQty);
    formData.append('name', name);
    
    fetch('cart-actions.php', {method: 'POST', body: formData})
        .then(() => loadCart());
}
```

### Delete (Remove Item)
```javascript
// Click trash icon with confirmation
function removeItem(id, name) {
    if (confirm('Remove this item?')) {
        const formData = new FormData();
        formData.append('action', 'remove');
        formData.append('id', id);
        formData.append('name', name);
        
        fetch('cart-actions.php', {method: 'POST', body: formData})
            .then(() => loadCart());
    }
}
```

---

## Integration Flow

### Step 1: Shopping (online-delivery.php)
- User browses items
- Clicks "Add to Cart" button
- Item added via AJAX to cart-actions.php
- Cart count updates in header
- User can add more items

### Step 2: View Cart (cart-view.php)
- User clicks cart icon or "View Cart"
- Page loads and fetches cart from session
- Displays all items with:
  - Quantity controls (+/- buttons)
  - Remove option (trash icon)
  - Item subtotals
- Shows order summary sidebar
- User can:
  - Modify quantities
  - Remove items
  - Change delivery type
  - Click "Proceed to Checkout"

### Step 3: Checkout (checkout-page.php)
- User must be logged in (enforced by PHP)
- Cart items displayed from session
- User enters delivery address
- User selects payment method
- Validates all data
- Shows order summary
- Click "Place Order"

### Step 4: Order Placement
- All validations pass
- Order data compiled
- Confirmation dialog shown
- Order ID generated
- Cart could be cleared (backend integration needed)

---

## Database Tables

### cart table
```sql
CREATE TABLE cart (
    id INT PRIMARY KEY AUTO_INCREMENT,
    user_id INT,
    session_id VARCHAR(255),
    product_name VARCHAR(255),
    product_price DECIMAL(10,2),
    product_image VARCHAR(255),
    quantity INT DEFAULT 1,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);
```

### users table (existing)
```sql
CREATE TABLE users (
    id INT PRIMARY KEY AUTO_INCREMENT,
    name VARCHAR(255),
    email VARCHAR(255) UNIQUE,
    password VARCHAR(255),
    phone VARCHAR(20),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);
```

---

## Session Management

### Session Variables
```php
$_SESSION['user_id']       // User ID (set on login)
$_SESSION['user_name']     // User name
$_SESSION['user_email']    // User email
$_SESSION['user_phone']    // User phone
$_SESSION['cart']          // Array of cart items
```

### Cart Item Structure
```php
[
    'id' => 'item_xyz',
    'product_name' => 'Biryani',
    'product_price' => 250,
    'product_image' => 'biryani.jpg',
    'quantity' => 2
]
```

---

## Validation Rules

### Address Validation
| Field | Rule | Example |
|-------|------|---------|
| Full Name | Required, 3-50 chars | John Doe |
| Phone | 10 digits | 9876543210 |
| Street | Required | 123 Main St |
| City | Required | Chennai |
| State | Required | Tamil Nadu |
| Postal | 6 digits | 600001 |

### Payment Validation
- **UPI**: Must not be empty
- **Card**: Valid 16-digit number, valid expiry, 3-digit CVV
- **Net Banking**: Bank selected
- **Wallet**: Option selected
- **COD**: No validation needed

---

## Responsive Design

### Mobile (< 768px)
- 1-column layout (form stacks vertically)
- Order summary below form
- Smaller font sizes
- Optimized button sizes
- Touch-friendly spacing

### Tablet (768px - 1024px)
- 2-column layout starts
- Better spacing
- Sticky sidebar active

### Desktop (> 1024px)
- Full 2-column layout
- Sticky order summary sidebar
- Large images and buttons
- Maximum width: 1200px

---

## Technical Stack

**Backend:**
- PHP 7.4+
- MySQL/MariaDB
- Session management
- PDO for database queries

**Frontend:**
- HTML5
- Bootstrap 5
- CSS3 (Flexbox, Grid, Gradients)
- JavaScript ES6+ (async/await, Fetch API)
- Font Awesome 4.7 icons
- SweetAlert2 for confirmations

**Features:**
- AJAX for seamless updates
- Real-time calculations
- Form validation
- Error handling
- Responsive design

---

## File Modifications Summary

| File | Changes | Status |
|------|---------|--------|
| cart-actions.php | Added session cart management, improved CRUD | ✅ Complete |
| cart-view.php | Enhanced UI, added CRUD operations with SweetAlert | ✅ Complete |
| checkout-page.php | Session integration, improved validation, professional UI | ✅ Complete |

---

## Next Steps (Backend Integration)

1. **Order Storage**
   - Create `orders` table
   - Implement order creation API
   - Generate unique order ID

2. **Payment Processing**
   - Integrate payment gateway (Razorpay/PayU)
   - Handle payment callbacks
   - Update order status

3. **Order Confirmation**
   - Send email confirmation
   - SMS notification
   - Display order tracking page

4. **Admin Dashboard**
   - View all orders
   - Update order status
   - Delivery management

---

## Testing Checklist

- [ ] Add item to cart from online-delivery.php
- [ ] View cart displays all items
- [ ] Increase quantity (+) button works
- [ ] Decrease quantity (-) button works
- [ ] Remove item with confirmation works
- [ ] Cart count updates in header
- [ ] Order type toggle (Delivery/Takeaway) calculates correctly
- [ ] Proceed to checkout redirects correctly
- [ ] Checkout page displays cart items
- [ ] Address form validation works
- [ ] Payment method selection works
- [ ] Place order validation works
- [ ] Order confirmation shows correct details
- [ ] Mobile responsive design works

---

## Known Limitations

1. Order storage not implemented (backend integration needed)
2. Payment gateway not integrated (UI only)
3. Cart clearing after order not automated
4. Email confirmation not sent
5. Order tracking not available

---

## Support & Documentation

For questions or issues, refer to:
- [Cart System Architecture](CART_CHECKOUT_INTEGRATION.md)
- [Backend Payment Integration Guide](PAYMENT_GATEWAY_SETUP.md) (To be created)
- [Order Management System](ORDER_MANAGEMENT.md) (To be created)

---

**Last Updated**: December 24, 2025
**Status**: Production Ready (Frontend Complete)
**Version**: 1.0
