# Order Management System - Complete Documentation

## Overview
The order management system is a complete end-to-end solution that handles:
- Customer orders from checkout page
- Order storage in database
- Admin panel order management
- Order status tracking
- Order history for customers

---

## System Architecture

### Data Flow
```
online-delivery.php (Browse Items)
        ↓
    Add to Cart → cart-actions.php (Store in DB + Session)
        ↓
cart-view.php (View & Manage Cart)
        ↓
    localStorage (Bridge to Checkout)
        ↓
checkout-page.php (Final Checkout)
        ↓
    Place Order → order-handler.php (Save to DB)
        ↓
admin/orders.php (Admin Management)
```

---

## Database Schema

### orders Table
```sql
CREATE TABLE orders (
  id INT PRIMARY KEY AUTO_INCREMENT,
  order_id VARCHAR(50) UNIQUE NOT NULL,          -- ORDYYYYMMDDhhmmssRandom
  user_id INT NOT NULL,                           -- Reference to users table
  user_name VARCHAR(255) NOT NULL,
  user_phone VARCHAR(20),
  user_email VARCHAR(255),
  delivery_address TEXT,                          -- Full address
  delivery_street VARCHAR(255),
  delivery_city VARCHAR(100),
  delivery_state VARCHAR(100),
  delivery_postal VARCHAR(10),
  special_instructions TEXT,                      -- Special delivery notes
  payment_method VARCHAR(50),                     -- UPI, Card, Net Banking, etc.
  subtotal DECIMAL(10, 2) DEFAULT 0,
  delivery_charge DECIMAL(10, 2) DEFAULT 0,
  taxes DECIMAL(10, 2) DEFAULT 0,
  discount DECIMAL(10, 2) DEFAULT 0,
  total_amount DECIMAL(10, 2) DEFAULT 0,
  order_status ENUM('pending', 'confirmed', 'preparing', 
                    'out_for_delivery', 'delivered', 'cancelled'),
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
);
```

### order_items Table
```sql
CREATE TABLE order_items (
  id INT PRIMARY KEY AUTO_INCREMENT,
  order_id INT NOT NULL,                          -- Reference to orders.id
  product_name VARCHAR(255) NOT NULL,
  product_price DECIMAL(10, 2) NOT NULL,
  quantity INT NOT NULL DEFAULT 1,
  item_total DECIMAL(10, 2) DEFAULT 0,            -- product_price × quantity
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  FOREIGN KEY (order_id) REFERENCES orders(id) ON DELETE CASCADE
);
```

---

## Backend Files

### order-handler.php
Main backend handler for all order operations.

**Location**: `c:\xampp\htdocs\hotel\order-handler.php`

**Actions**:

#### 1. create_order
Creates a new order in the database.

**Request**:
```json
{
  "action": "create_order",
  "order_data": {
    "user_name": "John Doe",
    "user_phone": "9876543210",
    "user_email": "john@example.com",
    "delivery_address": "123 Main St, City, State 12345",
    "delivery_street": "123 Main St",
    "delivery_city": "City",
    "delivery_state": "State",
    "delivery_postal": "12345",
    "special_instructions": "Ring bell twice",
    "payment_method": "UPI",
    "order_items": [
      {
        "product_name": "Butter Chicken",
        "product_price": 250.00,
        "quantity": 2
      }
    ],
    "subtotal": 500.00,
    "delivery_charge": 40.00,
    "taxes": 25.00,
    "discount": 10.00,
    "total_amount": 555.00
  }
}
```

**Response - Success**:
```json
{
  "status": "success",
  "message": "Order created successfully",
  "order_id": "ORD20241224234512345"
}
```

**Response - Error**:
```json
{
  "status": "error",
  "message": "User not logged in"
}
```

#### 2. fetch_user_orders
Retrieves all orders for the logged-in user.

**Request**:
```json
{
  "action": "fetch_user_orders"
}
```

**Response**:
```json
{
  "status": "success",
  "orders": [
    {
      "id": 1,
      "order_id": "ORD20241224234512345",
      "user_name": "John Doe",
      "total_amount": "555.00",
      "order_status": "pending",
      "created_at": "2024-12-24 23:45:12"
    }
  ]
}
```

#### 3. fetch_order_details
Retrieves complete details for a specific order including items.

**Request**:
```json
{
  "action": "fetch_order_details",
  "order_id": 1
}
```

**Response**:
```json
{
  "status": "success",
  "order": {
    "id": 1,
    "order_id": "ORD20241224234512345",
    "user_name": "John Doe",
    "user_phone": "9876543210",
    "user_email": "john@example.com",
    "delivery_address": "123 Main St, City, State 12345",
    "total_amount": "555.00",
    "order_status": "pending",
    "created_at": "2024-12-24 23:45:12"
  },
  "items": [
    {
      "product_name": "Butter Chicken",
      "product_price": "250.00",
      "quantity": "2",
      "item_total": "500.00"
    }
  ]
}
```

#### 4. update_order_status (Admin Only)
Updates the status of an order. Admin authentication required.

**Request**:
```json
{
  "action": "update_order_status",
  "order_id": 1,
  "status": "confirmed"
}
```

**Valid Status Values**:
- `pending` - Order received, awaiting confirmation
- `confirmed` - Order confirmed by restaurant
- `preparing` - Order being prepared
- `out_for_delivery` - Order out for delivery
- `delivered` - Order delivered successfully
- `cancelled` - Order cancelled

**Response**:
```json
{
  "status": "success",
  "message": "Order status updated"
}
```

---

## Frontend Integration

### checkout-page.php - Order Placement
The checkout page sends order data to the backend when user clicks "Place Order".

**Key Function**: `placeOrder()`

```javascript
// Preparation
const orderData = {
    user_name: document.getElementById('fullName').value,
    user_phone: document.getElementById('phone').value,
    user_email: userEmail,
    delivery_address: fullAddress,
    delivery_street: street,
    delivery_city: city,
    delivery_state: state,
    delivery_postal: postal,
    special_instructions: document.getElementById('instructions').value,
    payment_method: selectedPayment,
    order_items: cart,
    subtotal, delivery_charge, taxes, discount, total_amount
};

// Send to backend
fetch('order-handler.php', {
    method: 'POST',
    headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
    body: 'action=create_order&order_data=' + JSON.stringify(orderData)
})
.then(response => response.json())
.then(data => {
    if (data.status === 'success') {
        // Clear localStorage
        localStorage.removeItem('cart');
        // Redirect to home
        window.location.href = 'index.php';
    }
});
```

---

### admin/orders.php - Order Management
Professional admin interface for viewing and managing orders.

**Features**:
- View all orders with pagination
- Search orders by ID, customer name, or phone
- Filter orders by status
- View detailed order information
- Update order status
- Real-time status badges with color coding

**Status Color Scheme**:
- **Pending**: Yellow/Amber (awaiting confirmation)
- **Confirmed**: Blue (confirmed by restaurant)
- **Preparing**: Purple (being prepared)
- **Out for Delivery**: Green (on the way)
- **Delivered**: Bright Green (completed)
- **Cancelled**: Red (cancelled order)

---

## Installation & Setup

### Step 1: Create Database Tables
```bash
php setup_database.php
```

This will create:
- `orders` table
- `order_items` table

### Step 2: Verify Database Connection
Ensure `db_connect.php` is properly configured with your database credentials.

### Step 3: Access Admin Panel
Navigate to: `http://localhost/hotel/admin/orders.php`

---

## Order Lifecycle

### 1. Customer Places Order
1. Customer browses menu in `online-delivery.php`
2. Adds items to cart → stored in database + session
3. Clicks cart icon → navigates to `cart-view.php`
4. Manages cart (update quantity, remove items)
5. Clicks "Proceed to Checkout"
6. Cart saved to localStorage
7. Navigates to `checkout-page.php`
8. Fills delivery address form
9. Selects payment method
10. Clicks "Place Order"

### 2. Order Created
1. `checkout-page.php` sends order data to `order-handler.php`
2. Backend validates data
3. Inserts order header in `orders` table
4. Inserts each item in `order_items` table
5. Clears cart from database and session
6. Returns `order_id` to frontend
7. Frontend clears localStorage
8. Redirects to home page

### 3. Order Visible in Admin
1. Admin logs in
2. Navigates to `admin/orders.php`
3. Views all orders with status `pending`
4. Can search or filter orders
5. Clicks "View" to see order details
6. Clicks "Update" to change status

### 4. Status Tracking
Admin updates status through lifecycle:
- `pending` → `confirmed` → `preparing` → `out_for_delivery` → `delivered`

---

## Data Validation

### Order Validation
- **User**: Must be logged in
- **Cart**: Must not be empty
- **Phone**: Must be 10 digits (regex: `/^\d{10}$/)
- **Postal Code**: Must be 6 digits
- **Delivery Address**: Required
- **Payment Method**: Must be selected
- **All prices**: Must be valid decimals

### Order Item Validation
- **Product Name**: Required, not empty
- **Product Price**: Must be positive decimal
- **Quantity**: Must be positive integer

---

## Error Handling

### Common Errors & Solutions

**"User not logged in"**
- Solution: Ensure user is authenticated before placing order
- Code checks `$_SESSION['user_id']` in order-handler.php

**"Invalid order data"**
- Solution: Verify all required fields in order data
- Check JSON formatting of order_data

**"Order not found"**
- Solution: Verify correct order ID is being requested
- Ensure order belongs to current user

**Database Connection Error**
- Solution: Check `db_connect.php` credentials
- Verify database server is running

---

## Security Considerations

### Current Implementation
✅ Session-based authentication (prevents unauthorized access)
✅ Prepared statements (prevents SQL injection)
✅ User-owned data isolation (users can only see their orders)
✅ Admin-only status updates (only admins can change order status)
✅ Input validation (phone, postal code formats)

### Future Enhancements
- [ ] Add CSRF tokens
- [ ] Add rate limiting
- [ ] Encrypt sensitive data
- [ ] Add audit logging
- [ ] Add payment gateway integration

---

## API Reference

### POST /order-handler.php

**Headers Required**:
```
Content-Type: application/x-www-form-urlencoded
```

**Authentication**:
- User endpoints: Requires `$_SESSION['user_id']`
- Admin endpoints: Requires `$_SESSION['admin_id']`

**Response Format**: JSON
```json
{
  "status": "success|error",
  "message": "Description",
  "data": {}  // Optional, depends on action
}
```

---

## Testing Checklist

- [ ] Create a test order from checkout page
- [ ] Verify order appears in admin panel
- [ ] Verify order items display correctly
- [ ] Test order status update from pending to confirmed
- [ ] Test status filter in admin panel
- [ ] Test search by order ID
- [ ] Test search by customer name
- [ ] Test empty cart validation
- [ ] Test phone number validation
- [ ] Test postal code validation
- [ ] Test unauthenticated access (should redirect to login)
- [ ] Test admin access control

---

## File Locations

| File | Location | Purpose |
|------|----------|---------|
| order-handler.php | `/hotel/` | Backend order API |
| orders.php | `/hotel/admin/` | Admin panel interface |
| checkout-page.php | `/hotel/` | Customer checkout form |
| db_connect.php | `/hotel/` | Database connection |
| database_schema.sql | `/hotel/` | Table definitions |
| setup_database.php | `/hotel/` | Setup script |

---

## Troubleshooting

### Orders Not Appearing in Admin Panel
1. Check if orders table exists: `SELECT * FROM orders;`
2. Verify user is logged in before placing order
3. Check browser console for JavaScript errors
4. Check PHP error logs

### Order Status Not Updating
1. Verify admin is logged in
2. Check admin session: `$_SESSION['admin_id']` must be set
3. Verify order_id is correct
4. Check database connection

### Cart Data Not Transferring to Checkout
1. Verify localStorage is enabled in browser
2. Check browser developer tools > Application > Local Storage
3. Verify data is being saved: `localStorage.getItem('cart')`
4. Check for JavaScript errors in checkout page

---

## Performance Optimization

### Current State
- ✅ Database indexes on `user_id`, `order_id`, `order_status`
- ✅ Prepared statements for query efficiency
- ✅ Limit 10 orders per fetch for user orders

### Future Optimizations
- [ ] Add pagination to admin orders list
- [ ] Add caching for frequently accessed data
- [ ] Optimize image loading for order receipts
- [ ] Add query result caching
- [ ] Consider denormalizing for reporting

---

## Maintenance Tasks

### Daily
- [ ] Monitor order volume
- [ ] Check admin panel responsiveness
- [ ] Verify email notifications (when implemented)

### Weekly
- [ ] Review cancelled orders
- [ ] Check database backup
- [ ] Analyze order trends

### Monthly
- [ ] Archive old completed orders
- [ ] Review and optimize queries
- [ ] Update documentation as needed

---

## Support & Documentation

For additional help:
1. Check error messages in response JSON
2. Review PHP error logs
3. Test API endpoints with Postman
4. Check database integrity with `CHECK TABLE orders;`
5. Review session data: `echo json_encode($_SESSION);`

---

**Last Updated**: December 24, 2025  
**Version**: 2.0  
**Status**: Production Ready ✅
