# Order System - Quick Reference Guide

## 🚀 Quick Start

### Setup (One-time)
```bash
cd c:\xampp\htdocs\hotel
php setup_database.php
```

### Access Points
- **Customer**: Browse → Add to Cart → Checkout → Place Order
- **Admin**: `http://localhost/hotel/admin/orders.php`

---

## 📋 Order Data Flow

### 1. Add to Cart (online-delivery.php)
```javascript
// User clicks "Add to Cart"
→ Sends to: cart-actions.php
→ Stores in: Database (if logged in) + Session
→ Updates: Cart icon count
```

### 2. Manage Cart (cart-view.php)
```javascript
// User views cart
→ Fetches from: cart-actions.php
→ Can: Update qty, Remove items
→ Then: Click "Proceed to Checkout"
→ Saves to: localStorage
```

### 3. Checkout (checkout-page.php)
```javascript
// User fills form & clicks "Place Order"
→ Loads cart from: localStorage
→ Sends order to: order-handler.php
→ Order stored in: orders table + order_items table
→ Clears: localStorage
→ Redirects to: index.php
```

### 4. Admin Management (admin/orders.php)
```
Admin views: All orders
Can: Search, Filter, View details, Update status
Updates: order_status field
```

---

## 📊 Database Tables

### orders (Order Headers)
| Column | Type | Purpose |
|--------|------|---------|
| id | INT | Primary key (auto-increment) |
| order_id | VARCHAR(50) | Display ID (ORD20241224...) |
| user_id | INT | Customer reference |
| user_name | VARCHAR(255) | Customer name |
| user_phone | VARCHAR(20) | Contact number |
| user_email | VARCHAR(255) | Email address |
| delivery_address | TEXT | Full delivery address |
| payment_method | VARCHAR(50) | UPI, Card, etc. |
| total_amount | DECIMAL(10,2) | Final amount |
| order_status | ENUM | pending, confirmed, preparing, etc. |
| created_at | TIMESTAMP | Order placement time |

### order_items (Order Line Items)
| Column | Type | Purpose |
|--------|------|---------|
| id | INT | Primary key |
| order_id | INT | Reference to orders.id |
| product_name | VARCHAR(255) | Item name |
| product_price | DECIMAL(10,2) | Unit price |
| quantity | INT | Qty ordered |
| item_total | DECIMAL(10,2) | price × qty |

---

## 🔧 Order Status Values

```
pending       → Order placed, awaiting confirmation
confirmed     → Restaurant confirmed the order
preparing     → Kitchen is preparing
out_for_delivery → Delivery in progress
delivered     → Successfully delivered
cancelled     → Order was cancelled
```

---

## 📡 API Endpoints

### Create Order
```
POST: order-handler.php
Body: action=create_order&order_data={JSON}
Returns: {status, order_id}
Auth: User must be logged in
```

### Fetch User Orders
```
POST: order-handler.php
Body: action=fetch_user_orders
Returns: {status, orders array}
Auth: User must be logged in
```

### Fetch Order Details
```
POST: order-handler.php
Body: action=fetch_order_details&order_id={id}
Returns: {status, order, items}
Auth: User must be logged in
```

### Update Order Status
```
POST: order-handler.php
Body: action=update_order_status&order_id={id}&status={status}
Returns: {status, message}
Auth: Admin must be logged in
```

---

## ✅ Validation Rules

### Order Creation
- User must be logged in
- Cart must not be empty
- Phone: 10 digits only
- Postal: 6 digits only
- Delivery address: Required
- Payment method: Required
- All prices: Must be positive decimals

### Order Items
- Product name: Required, non-empty
- Product price: Must be > 0
- Quantity: Must be > 0

---

## 🐛 Common Issues & Fixes

### Cart shows "undefined" or "₹NaN"
**Fix**: Ensure `parseFloat()` is used for prices
```javascript
// ✅ Correct
let price = parseFloat(item.product_price);

// ❌ Wrong
let price = item.product_price;
```

### Order not appearing in admin panel
**Check**:
1. User logged in when placing order
2. Order data sent to order-handler.php
3. No JavaScript errors in console
4. Database table exists: `SHOW TABLES;`

### localStorage data not persisting
**Check**:
1. Browser allows localStorage
2. Data saved before navigation: `localStorage.setItem('cart', JSON.stringify(cart))`
3. Data loaded correctly: `JSON.parse(localStorage.getItem('cart'))`

### Admin can't update order status
**Check**:
1. Admin is logged in: `$_SESSION['admin_id']` exists
2. Valid order ID passed
3. Valid status value (use enum values)

---

## 🔐 Security Checklist

✅ Prepared statements (prevents SQL injection)
✅ Session authentication (user isolation)
✅ Admin-only status updates
✅ Input validation (phone, postal formats)
✅ User-owned data isolation

⚠️ To-Do:
- [ ] Add CSRF tokens
- [ ] Add rate limiting
- [ ] Encrypt sensitive data
- [ ] Add audit logging

---

## 📝 Testing Commands

### Check if tables exist
```sql
SHOW TABLES;
DESCRIBE orders;
DESCRIBE order_items;
```

### View sample order
```sql
SELECT * FROM orders LIMIT 1;
SELECT * FROM order_items WHERE order_id = 1;
```

### View orders by status
```sql
SELECT * FROM orders WHERE order_status = 'pending';
```

### Clear test data
```sql
DELETE FROM order_items;
DELETE FROM orders;
```

---

## 📁 Files Overview

| File | Location | Usage |
|------|----------|-------|
| `order-handler.php` | `/hotel/` | Backend API |
| `orders.php` | `/hotel/admin/` | Admin interface |
| `checkout-page.php` | `/hotel/` | Customer checkout |
| `cart-view.php` | `/hotel/` | Cart management |
| `online-delivery.php` | `/hotel/` | Browse & add items |
| `database_schema.sql` | `/hotel/` | Table definitions |
| `setup_database.php` | `/hotel/` | Setup script |

---

## 🎯 Workflow Example

### Scenario: Customer places order

1. **Browse** → Customer opens online-delivery.php
2. **Add items** → Clicks "Add to Cart" button
   - item stored in: database + session
   - cart icon updated: +1 count
3. **View cart** → Clicks cart icon
   - navigates to: cart-view.php
   - items fetched from: database
   - displays with prices, quantities, total
4. **Modify cart** → Can update quantity or remove items
   - sends: cart-actions.php (update_qty or remove)
   - updates: database + session
   - display: refreshes automatically
5. **Checkout** → Clicks "Proceed to Checkout"
   - cart saved to: localStorage
   - navigates to: checkout-page.php
6. **Fill form** → Enters delivery address, payment method
7. **Place order** → Clicks "Place Order"
   - loads cart from: localStorage
   - validates: all fields
   - sends to: order-handler.php
   - creates: order + order_items in database
   - clears: localStorage
   - redirects to: index.php
8. **Order confirmed** → Customer can view orders in profile page

### Scenario: Admin manages order

1. **Login** → Admin logs into admin panel
2. **Navigate** → Goes to admin/orders.php
3. **View orders** → Sees all orders with status badges
4. **Search** → Can search by order ID, customer name, phone
5. **Filter** → Can filter by status (pending, confirmed, etc.)
6. **View details** → Clicks "View" to see:
   - Customer details
   - Delivery address
   - Order items with prices
   - Payment method
   - Total amount
7. **Update status** → Clicks "Update" to change status
   - Select new status from dropdown
   - Click "Update Status"
   - Status changes in database
   - Page refreshes

---

## 🚨 Error Messages

| Error | Cause | Solution |
|-------|-------|----------|
| User not logged in | Session expired | Redirect to login.php |
| Invalid order data | Malformed JSON | Check data format |
| Order not found | Wrong ID or not owner | Verify order ID, user ID |
| Database error | Connection issue | Check db_connect.php |
| Invalid status | Unknown enum value | Use valid status values |
| Admin access required | Not admin user | Check admin session |

---

## 📞 Support

For issues:
1. Check browser console (F12)
2. Check PHP error logs
3. Verify database tables exist
4. Test API with Postman
5. Review error messages in JSON responses

---

**Last Updated**: December 24, 2025  
**Version**: 2.0  
**Status**: Complete ✅
