<?php
// =====================================================
// ADMIN GALLERY MANAGEMENT PAGE
// =====================================================
session_start();
require_once 'check_admin.php'; // Ensure admin is logged in

// =====================================================
// DATABASE TABLE CREATION (if not exists)
// =====================================================
$create_table_sql = "CREATE TABLE IF NOT EXISTS `gallery` (
  `id` INT AUTO_INCREMENT PRIMARY KEY,
  `title` VARCHAR(255) NOT NULL,
  `category` VARCHAR(50) NOT NULL,
  `image_path` VARCHAR(255) NOT NULL,
  `created_date` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  `updated_date` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4";

if (!$conn->query($create_table_sql)) {
    error_log("Table creation error: " . $conn->error);
}

// =====================================================
// PROCESS FORM SUBMISSIONS
// =====================================================

// ADD IMAGE
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'add') {
    $title = trim($_POST['title'] ?? '');
    $category = trim($_POST['category'] ?? '');
    
    if (empty($title) || empty($category)) {
        $error_message = "Title and Category are required!";
    } elseif (!isset($_FILES['image']) || $_FILES['image']['error'] !== UPLOAD_ERR_OK) {
        $error_message = "Please select a valid image file!";
    } else {
        $valid_categories = ['food', 'events', 'staff'];
        if (!in_array($category, $valid_categories)) {
            $error_message = "Invalid category selected!";
        } else {
            $file = $_FILES['image'];
            $allowed_types = ['image/jpeg', 'image/png', 'image/gif', 'image/webp'];
            $max_size = 5 * 1024 * 1024; // 5MB
            
            if (!in_array($file['type'], $allowed_types)) {
                $error_message = "Only JPG, PNG, GIF, and WebP images are allowed!";
            } elseif ($file['size'] > $max_size) {
                $error_message = "Image size must be less than 5MB!";
            } else {
                // Upload logic
                $upload_dir = '../assets/images/gallery/';
                if (!is_dir($upload_dir)) mkdir($upload_dir, 0755, true);
                
                $file_ext = pathinfo($file['name'], PATHINFO_EXTENSION);
                $filename = time() . '_' . bin2hex(random_bytes(5)) . '.' . $file_ext;
                $filepath = $upload_dir . $filename;
                $image_path = 'assets/images/gallery/' . $filename; // Path for DB
                
                if (move_uploaded_file($file['tmp_name'], $filepath)) {
                    $stmt = $conn->prepare("INSERT INTO gallery (title, category, image_path) VALUES (?, ?, ?)");
                    if ($stmt) {
                        $stmt->bind_param("sss", $title, $category, $image_path);
                        if ($stmt->execute()) {
                            $success_message = "Image added successfully!";
                        } else {
                            $error_message = "Database error: " . $stmt->error;
                            unlink($filepath);
                        }
                        $stmt->close();
                    }
                } else {
                    $error_message = "Failed to upload image.";
                }
            }
        }
    }
}

// UPDATE IMAGE
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'update') {
    $id = intval($_POST['id'] ?? 0);
    $title = trim($_POST['title'] ?? '');
    $category = trim($_POST['category'] ?? '');
    
    if ($id > 0 && !empty($title) && !empty($category)) {
        $stmt = $conn->prepare("UPDATE gallery SET title = ?, category = ? WHERE id = ?");
        $stmt->bind_param("ssi", $title, $category, $id);
        if ($stmt->execute()) {
            $success_message = "Image updated successfully!";
        } else {
            $error_message = "Update failed: " . $stmt->error;
        }
        $stmt->close();
    }
}

// DELETE IMAGE
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'delete') {
    $id = intval($_POST['id'] ?? 0);
    if ($id > 0) {
        $stmt = $conn->prepare("SELECT image_path FROM gallery WHERE id = ?");
        $stmt->bind_param("i", $id);
        $stmt->execute();
        $res = $stmt->get_result();
        if ($row = $res->fetch_assoc()) {
            $file_path = '../' . $row['image_path'];
            
            $del = $conn->prepare("DELETE FROM gallery WHERE id = ?");
            $del->bind_param("i", $id);
            if ($del->execute()) {
                if (file_exists($file_path)) unlink($file_path);
                $success_message = "Image deleted successfully!";
            }
        }
    }
}

// FETCH IMAGES
$gallery_images = [];
$result = $conn->query("SELECT * FROM gallery ORDER BY created_date DESC");
if ($result) {
    while ($row = $result->fetch_assoc()) {
        $gallery_images[] = $row;
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>Gallery Management - Admin</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/bootstrap/5.3.0/css/bootstrap.min.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <link rel="stylesheet" href="assets/css/admin-style.css">
    <style>
        /* Lightbox & Gallery Specific Styles */
        .gallery-thumb {
            width: 80px;
            height: 60px;
            object-fit: cover;
            border-radius: 6px;
            cursor: pointer;
            transition: transform 0.2s;
            border: 1px solid #ddd;
        }
        .gallery-thumb:hover { transform: scale(1.1); }
        
        .lightbox-modal {
            display: none;
            position: fixed;
            z-index: 2000;
            left: 0; top: 0;
            width: 100%; height: 100%;
            background-color: rgba(0,0,0,0.9);
        }
        .lightbox-content {
            margin: auto;
            display: block;
            max-width: 90%;
            max-height: 90vh;
            position: absolute;
            top: 50%; left: 50%;
            transform: translate(-50%, -50%);
            border: 5px solid white;
            border-radius: 5px;
        }
        .lightbox-close {
            position: absolute;
            top: 20px; right: 35px;
            color: #fff;
            font-size: 40px;
            font-weight: bold;
            cursor: pointer;
        }
        
        /* Drag Drop Zone */
        .file-upload-wrapper {
            border: 2px dashed #ccc;
            padding: 30px;
            text-align: center;
            background: #f9f9f9;
            border-radius: 8px;
            cursor: pointer;
            transition: all 0.3s;
        }
        .file-upload-wrapper:hover {
            border-color: #931A2A;
            background: #fff5f5;
        }
        .file-upload-wrapper input[type="file"] {
            display: none;
        }
    </style>
</head>
<body>

<div class="wrapper">
    
    <?php include 'sidebar.php'; ?>

    <div class="main-content">
        <nav class="top-nav">
            <div class="nav-left">
                <h2><i class="fas fa-images"></i> Gallery Management</h2>
            </div>
            <div class="nav-right">
                <span class="admin-name">Welcome, Admin</span>
            </div>
        </nav>

        <div class="content">
            
            <?php if (isset($success_message)): ?>
                <div class="alert alert-success alert-dismissible fade show">
                    <i class="fas fa-check-circle"></i> <?= htmlspecialchars($success_message) ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                </div>
            <?php endif; ?>

            <?php if (isset($error_message)): ?>
                <div class="alert alert-danger alert-dismissible fade show">
                    <i class="fas fa-exclamation-circle"></i> <?= htmlspecialchars($error_message) ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                </div>
            <?php endif; ?>

            <div class="admin-form-section mb-4">
                <h5 class="form-title mb-3"><i class="fas fa-cloud-upload-alt"></i> Add New Image</h5>
                <form method="POST" enctype="multipart/form-data">
                    <input type="hidden" name="action" value="add">
                    <div class="row g-3">
                        <div class="col-md-6">
                            <label class="form-label">Image Title</label>
                            <input type="text" class="form-control" name="title" placeholder="e.g., Spicy Chicken Wings" required>
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">Category</label>
                            <select class="form-select" name="category" required>
                                <option value="">Select Category</option>
                                <option value="food">Food</option>
                                <option value="events">Ambience & Events</option>
                                <option value="staff">Staff</option>
                            </select>
                        </div>
                        <div class="col-12">
                            <label class="form-label">Upload Image</label>
                            <label class="file-upload-wrapper w-100">
                                <input type="file" name="image" id="imageInput" accept="image/*" required>
                                <i class="fas fa-cloud-upload-alt fa-2x text-muted mb-2"></i>
                                <p class="mb-0 text-muted" id="fileName">Click or Drag image here (JPG, PNG, WEBP)</p>
                            </label>
                        </div>
                        <div class="col-12">
                            <button type="submit" class="btn btn-primary-custom">
                                <i class="fas fa-save"></i> Save to Gallery
                            </button>
                        </div>
                    </div>
                </form>
            </div>

            <div class="table-responsive">
                <table class="table table-modern align-middle table-hover">
                    <thead>
                        <tr>
                            <th width="100">Preview</th>
                            <th>Title</th>
                            <th>Category</th>
                            <th>Date Added</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (empty($gallery_images)): ?>
                            <tr>
                                <td colspan="5" class="text-center py-4 text-muted">No images found in gallery.</td>
                            </tr>
                        <?php else: ?>
                            <?php foreach ($gallery_images as $img): ?>
                            <tr>
                                <td>
                                    <img src="../<?= htmlspecialchars($img['image_path']) ?>" 
                                         class="gallery-thumb" 
                                         onclick="openLightbox(this.src)"
                                         alt="thumb">
                                </td>
                                <td class="fw-bold"><?= htmlspecialchars($img['title']) ?></td>
                                <td>
                                    <?php 
                                        $badgeClass = 'bg-secondary';
                                        if($img['category'] == 'food') $badgeClass = 'bg-warning text-dark';
                                        if($img['category'] == 'events') $badgeClass = 'bg-info text-dark';
                                        if($img['category'] == 'staff') $badgeClass = 'bg-success';
                                    ?>
                                    <span class="badge <?= $badgeClass ?> text-uppercase"><?= $img['category'] ?></span>
                                </td>
                                <td><?= date('M d, Y', strtotime($img['created_date'])) ?></td>
                                <td>
                                    <div class="action-buttons">
                                        <button class="btn btn-sm btn-info text-white" 
                                                onclick="openLightbox('../<?= htmlspecialchars($img['image_path']) ?>')">
                                            <i class="fas fa-eye"></i>
                                        </button>
                                        
                                        <button class="btn btn-sm btn-warning text-white" 
                                                onclick="editImage(<?= $img['id'] ?>, '<?= htmlspecialchars($img['title']) ?>', '<?= $img['category'] ?>')">
                                            <i class="fas fa-edit"></i>
                                        </button>

                                        <form method="POST" style="display:inline;" onsubmit="return confirm('Delete this image?');">
                                            <input type="hidden" name="action" value="delete">
                                            <input type="hidden" name="id" value="<?= $img['id'] ?>">
                                            <button class="btn btn-sm btn-danger"><i class="fas fa-trash"></i></button>
                                        </form>
                                    </div>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>

        </div> </div> </div> <div class="modal fade" id="editModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header" style="background: #931A2A; color: white;">
                <h5 class="modal-title">Edit Image Details</h5>
                <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
            </div>
            <form method="POST">
                <div class="modal-body">
                    <input type="hidden" name="action" value="update">
                    <input type="hidden" name="id" id="edit-id">
                    
                    <div class="mb-3">
                        <label class="form-label">Title</label>
                        <input type="text" class="form-control" name="title" id="edit-title" required>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Category</label>
                        <select class="form-select" name="category" id="edit-category" required>
                            <option value="food">Food</option>
                            <option value="events">Ambience & Events</option>
                            <option value="staff">Staff</option>
                        </select>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-primary-custom">Update</button>
                </div>
            </form>
        </div>
    </div>
</div>

<div id="lightbox" class="lightbox-modal" onclick="closeLightbox()">
    <span class="lightbox-close">&times;</span>
    <img id="lightbox-img" class="lightbox-content">
</div>

<script src="https://cdnjs.cloudflare.com/ajax/libs/bootstrap/5.3.0/js/bootstrap.bundle.min.js"></script>
<script>
    // File Input styling
    document.getElementById('imageInput').addEventListener('change', function() {
        let fileName = this.files[0].name;
        document.getElementById('fileName').innerText = "Selected: " + fileName;
        document.querySelector('.file-upload-wrapper').style.borderColor = '#28a745';
        document.querySelector('.file-upload-wrapper').style.backgroundColor = '#e8f5e9';
    });

    // Edit Logic
    function editImage(id, title, category) {
        document.getElementById('edit-id').value = id;
        document.getElementById('edit-title').value = title;
        document.getElementById('edit-category').value = category;
        new bootstrap.Modal(document.getElementById('editModal')).show();
    }

    // Lightbox Logic
    function openLightbox(src) {
        document.getElementById('lightbox-img').src = src;
        document.getElementById('lightbox').style.display = 'block';
    }
    function closeLightbox() {
        document.getElementById('lightbox').style.display = 'none';
    }
</script>
</body>
</html>