<?php
require_once 'check_admin.php';

$message = "";
$error = "";

// Handle status update
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['action'])) {
    $order_id = intval($_POST['order_id']);
    $action = $_POST['action'];

    $status = null; // Initialize status variable

    if ($action == 'pending') {
        $status = 'pending';
    } elseif ($action == 'processing') {
        $status = 'processing';
    } elseif ($action == 'completed') {
        $status = 'completed';
    } elseif ($action == 'cancelled') {
        $status = 'cancelled';
    } elseif ($action == 'delete') {
        $stmt = $conn->prepare("DELETE FROM orders WHERE id = ?");
        $stmt->bind_param("i", $order_id);
        if ($stmt->execute()) {
            $message = "Order deleted successfully!";
        } else {
            $error = "Failed to delete order.";
        }
        $status = null; // Ensure no update runs after delete
    }

    if ($status !== null) {
        $stmt = $conn->prepare("UPDATE orders SET status = ? WHERE id = ?");
        $stmt->bind_param("si", $status, $order_id);
        if ($stmt->execute()) {
            $message = "Order status updated to " . $status . "!";
        } else {
            $error = "Failed to update order.";
        }
    }
}

// Get filter option
$filter = isset($_GET['filter']) ? $_GET['filter'] : 'all';
$query = "SELECT * FROM orders WHERE 1=1";

if ($filter == 'pending') {
    $query .= " AND status='pending'";
} elseif ($filter == 'processing') {
    $query .= " AND status='processing'";
} elseif ($filter == 'completed') {
    $query .= " AND status='completed'";
} elseif ($filter == 'cancelled') {
    $query .= " AND status='cancelled'";
}

$query .= " ORDER BY id DESC";
$result = $conn->query($query);
$orders = [];
while ($row = $result->fetch_assoc()) {
    $orders[] = $row;
}

// Get statistics
$stats = [
    'total' => $conn->query("SELECT COUNT(*) as count FROM orders")->fetch_assoc()['count'],
    'pending' => $conn->query("SELECT COUNT(*) as count FROM orders WHERE status='pending'")->fetch_assoc()['count'],
    'processing' => $conn->query("SELECT COUNT(*) as count FROM orders WHERE status='processing'")->fetch_assoc()['count'],
    'completed' => $conn->query("SELECT COUNT(*) as count FROM orders WHERE status='completed'")->fetch_assoc()['count'],
    'total_revenue' => $conn->query("SELECT SUM(total_amount) as sum FROM orders WHERE status='completed'")->fetch_assoc()['sum'] ?? 0,
];
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Order Management - Admin</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/bootstrap/5.3.0/css/bootstrap.min.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <link rel="stylesheet" href="assets/css/admin-style.css">
</head>
<body>
    <div class="wrapper">
        
        <?php include 'sidebar.php'; ?>

        <div class="main-content">
            <nav class="top-nav">
                <div class="nav-left">
                    <h2><i class="fas fa-shopping-cart"></i> Online Orders & Delivery</h2>
                </div>
                <div class="nav-right">
                    <span class="admin-name">Welcome, Admin</span>
                </div>
            </nav>

            <div class="content">
                <?php if (!empty($message)): ?>
                    <div class="alert alert-success alert-dismissible fade show" role="alert">
                        <?php echo htmlspecialchars($message); ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                <?php endif; ?>

                <?php if (!empty($error)): ?>
                    <div class="alert alert-danger alert-dismissible fade show" role="alert">
                        <?php echo htmlspecialchars($error); ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                <?php endif; ?>

                <div class="row mb-4">
                    <div class="col-md-3">
                        <div class="stat-card">
                            <h6>Total Orders</h6>
                            <p class="count"><?php echo $stats['total']; ?></p>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="stat-card pending">
                            <h6>Pending</h6>
                            <p class="count"><?php echo $stats['pending']; ?></p>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="stat-card approved"> <h6>Processing</h6>
                            <p class="count"><?php echo $stats['processing']; ?></p>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="stat-card total"> <h6>Revenue</h6>
                            <p class="count">₹<?php echo number_format($stats['total_revenue'], 2); ?></p>
                        </div>
                    </div>
                </div>

                <div class="mb-3">
                    <a href="?filter=all" class="btn btn-outline-secondary <?php echo $filter == 'all' ? 'active' : ''; ?>">All</a>
                    <a href="?filter=pending" class="btn btn-outline-warning <?php echo $filter == 'pending' ? 'active' : ''; ?>">Pending</a>
                    <a href="?filter=processing" class="btn btn-outline-info <?php echo $filter == 'processing' ? 'active' : ''; ?>">Processing</a>
                    <a href="?filter=completed" class="btn btn-outline-success <?php echo $filter == 'completed' ? 'active' : ''; ?>">Completed</a>
                    <a href="?filter=cancelled" class="btn btn-outline-danger <?php echo $filter == 'cancelled' ? 'active' : ''; ?>">Cancelled</a>
                </div>

                <div class="table-responsive">
                    <table class="table table-striped table-hover table-modern">
                        <thead class="table-dark">
                            <tr>
                                <th>Order ID</th>
                                <th>Customer</th>
                                <th>Phone</th>
                                <th>Address</th>
                                <th>Total Amount</th>
                                <th>Date</th>
                                <th>Status</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($orders as $order): ?>
                            <tr>
                                <td>#<?php echo $order['id']; ?></td>
                                <td><?php echo htmlspecialchars($order['customer_name']); ?></td>
                                <td><?php echo htmlspecialchars($order['customer_phone']); ?></td>
                                <td><?php echo substr(htmlspecialchars($order['delivery_address']), 0, 40) . '...'; ?></td>
                                <td><strong>₹<?php echo number_format($order['total_amount'], 2); ?></strong></td>
                                <td><?php echo date('M d, Y H:i', strtotime($order['created_at'])); ?></td>
                                <td>
                                    <?php
                                    $status_colors = [
                                        'pending' => 'warning',
                                        'processing' => 'info',
                                        'completed' => 'success',
                                        'cancelled' => 'danger'
                                    ];
                                    $color = $status_colors[$order['status']] ?? 'secondary';
                                    ?>
                                    <span class="badge bg-<?php echo $color; ?>"><?php echo ucfirst($order['status']); ?></span>
                                </td>
                                <td>
                                    <div class="btn-group btn-group-sm">
                                        <button class="btn btn-info text-white" data-bs-toggle="modal" data-bs-target="#orderModal" onclick="showOrder(<?php echo htmlspecialchars(json_encode($order)); ?>)" title="View Details">
                                            <i class="fas fa-eye"></i>
                                        </button>
                                        <div class="btn-group btn-group-sm">
                                            <button type="button" class="btn btn-secondary dropdown-toggle dropdown-toggle-split" data-bs-toggle="dropdown">
                                                <span class="visually-hidden">Change Status</span>
                                            </button>
                                            <ul class="dropdown-menu">
                                                <li>
                                                    <form method="POST" style="display:inline;">
                                                        <input type="hidden" name="action" value="pending">
                                                        <input type="hidden" name="order_id" value="<?php echo $order['id']; ?>">
                                                        <button type="submit" class="dropdown-item">Pending</button>
                                                    </form>
                                                </li>
                                                <li>
                                                    <form method="POST" style="display:inline;">
                                                        <input type="hidden" name="action" value="processing">
                                                        <input type="hidden" name="order_id" value="<?php echo $order['id']; ?>">
                                                        <button type="submit" class="dropdown-item">Processing</button>
                                                    </form>
                                                </li>
                                                <li>
                                                    <form method="POST" style="display:inline;">
                                                        <input type="hidden" name="action" value="completed">
                                                        <input type="hidden" name="order_id" value="<?php echo $order['id']; ?>">
                                                        <button type="submit" class="dropdown-item">Completed</button>
                                                    </form>
                                                </li>
                                                <li><hr class="dropdown-divider"></li>
                                                <li>
                                                    <form method="POST" style="display:inline;">
                                                        <input type="hidden" name="action" value="cancelled">
                                                        <input type="hidden" name="order_id" value="<?php echo $order['id']; ?>">
                                                        <button type="submit" class="dropdown-item text-danger">Cancel</button>
                                                    </form>
                                                </li>
                                            </ul>
                                        </div>
                                        <form method="POST" style="display:inline;">
                                            <input type="hidden" name="action" value="delete">
                                            <input type="hidden" name="order_id" value="<?php echo $order['id']; ?>">
                                            <button type="submit" class="btn btn-danger" onclick="return confirm('Are you sure?')" title="Delete">
                                                <i class="fas fa-trash"></i>
                                            </button>
                                        </form>
                                    </div>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>

    <div class="modal fade" id="orderModal" tabindex="-1">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header" style="background: #931A2A; color: white;">
                    <h5 class="modal-title">Order Details</h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body" id="orderContent">
                    </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdnjs.cloudflare.com/ajax/libs/bootstrap/5.3.0/js/bootstrap.bundle.min.js"></script>
    <script>
        function showOrder(order) {
            const items = order.items ? JSON.parse(order.items) : [];
            let itemsHtml = '<ul class="list-group mb-3">';
            if (Array.isArray(items)) {
                items.forEach(item => {
                    itemsHtml += `<li class="list-group-item d-flex justify-content-between align-items-center">
                        <div>${item.name} <small class="text-muted">x${item.quantity}</small></div>
                        <span>₹${item.price}</span>
                    </li>`;
                });
            }
            itemsHtml += '</ul>';

            const html = `
                <div class="order-detail">
                    <div class="row mb-3">
                        <div class="col-md-6">
                            <h6 class="text-uppercase text-muted border-bottom pb-2">Customer Info</h6>
                            <p><strong>Name:</strong> ${order.customer_name}</p>
                            <p><strong>Email:</strong> ${order.customer_email || 'N/A'}</p>
                            <p><strong>Phone:</strong> ${order.customer_phone}</p>
                        </div>
                        <div class="col-md-6">
                            <h6 class="text-uppercase text-muted border-bottom pb-2">Delivery Info</h6>
                            <p><strong>Address:</strong> ${order.delivery_address}</p>
                            <p><strong>City:</strong> ${order.delivery_city || 'N/A'}</p>
                            <p><strong>Zip Code:</strong> ${order.delivery_zip || 'N/A'}</p>
                        </div>
                    </div>
                    
                    <h6 class="text-uppercase text-muted border-bottom pb-2">Order Items</h6>
                    ${itemsHtml}
                    
                    <div class="bg-light p-3 rounded">
                        <div class="d-flex justify-content-between"><span>Subtotal:</span> <strong>₹${order.subtotal || 0}</strong></div>
                        <div class="d-flex justify-content-between"><span>Delivery Fee:</span> <strong>₹${order.delivery_fee || 0}</strong></div>
                        <hr>
                        <div class="d-flex justify-content-between fs-5"><span>Total Amount:</span> <strong class="text-primary">₹${order.total_amount}</strong></div>
                    </div>
                    
                    <div class="mt-3">
                        <p><strong>Special Instructions:</strong> <span class="text-muted">${order.special_instructions || 'None'}</span></p>
                        <p><strong>Order Date:</strong> ${new Date(order.created_at).toLocaleString()}</p>
                    </div>
                </div>
            `;
            document.getElementById('orderContent').innerHTML = html;
        }
    </script>
</body>
</html>