<?php
require_once 'check_admin.php';

$message = "";
$error = "";

// Handle CRUD Operations
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    
    /* =============== CREATE / ADD RESERVATION =============== */
    if (isset($_POST['action']) && $_POST['action'] == 'add') {
        $name = trim($_POST['name']);
        $email = trim($_POST['email']);
        $phone = trim($_POST['phone']);
        $guests = intval($_POST['guests']);
        $date = $_POST['reservation_date'];
        $time = $_POST['reservation_time'];

        // Validation
        if (empty($name) || empty($email) || empty($phone) || empty($date) || empty($time) || $guests <= 0) {
            $error = "Please fill in all required fields!";
        } else {
            $stmt = $conn->prepare("INSERT INTO reservations (name, email, phone, guests, res_date, res_time) VALUES (?, ?, ?, ?, ?, ?)");
            $stmt->bind_param("sssiis", $name, $email, $phone, $guests, $date, $time);
            
            if ($stmt->execute()) {
                $message = "✓ Reservation added successfully!";
            } else {
                $error = "✗ Failed to add reservation: " . $stmt->error;
            }
            $stmt->close();
        }
    }
    
    /* =============== UPDATE RESERVATION =============== */
    elseif (isset($_POST['action']) && $_POST['action'] == 'update') {
        $reservation_id = intval($_POST['reservation_id']);
        $name = trim($_POST['name']);
        $email = trim($_POST['email']);
        $phone = trim($_POST['phone']);
        $guests = intval($_POST['guests']);
        $date = $_POST['reservation_date'];
        $time = $_POST['reservation_time'];

        // Validation
        if (empty($name) || empty($email) || empty($phone) || empty($date) || empty($time) || $guests <= 0) {
            $error = "Please fill in all required fields!";
        } else {
            $stmt = $conn->prepare("UPDATE reservations SET name=?, email=?, phone=?, guests=?, res_date=?, res_time=? WHERE id=?");
            $stmt->bind_param("sssiiis", $name, $email, $phone, $guests, $date, $time, $reservation_id);
            
            if ($stmt->execute()) {
                $message = "✓ Reservation updated successfully!";
            } else {
                $error = "✗ Failed to update reservation: " . $stmt->error;
            }
            $stmt->close();
        }
    }
    
    /* =============== DELETE RESERVATION =============== */
    elseif (isset($_POST['action']) && $_POST['action'] == 'delete') {
        $reservation_id = intval($_POST['reservation_id']);
        $stmt = $conn->prepare("DELETE FROM reservations WHERE id=?");
        $stmt->bind_param("i", $reservation_id);
        
        if ($stmt->execute()) {
            $message = "✓ Reservation deleted successfully!";
        } else {
            $error = "✗ Failed to delete reservation: " . $stmt->error;
        }
        $stmt->close();
    }
}

// Get all reservations
$query = "SELECT * FROM reservations ORDER BY res_date DESC";
$result = $conn->query($query);
$reservations = [];
while ($row = $result->fetch_assoc()) {
    $reservations[] = $row;
}

// Get statistics
$stats = [
    'total' => $conn->query("SELECT COUNT(*) as count FROM reservations")->fetch_assoc()['count'],
];

// Get single reservation for edit
$edit_reservation = null;
if (isset($_GET['edit'])) {
    $edit_id = intval($_GET['edit']);
    $stmt = $conn->prepare("SELECT * FROM reservations WHERE id=?");
    $stmt->bind_param("i", $edit_id);
    $stmt->execute();
    $result = $stmt->get_result();
    $edit_reservation = $result->fetch_assoc();
    $stmt->close();
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Reservation Management - Admin</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/bootstrap/5.3.0/css/bootstrap.min.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <link rel="stylesheet" href="assets/css/admin-style.css">
    <style>
        .admin-form-section {
            background: #f8f9fa;
            padding: 25px;
            border-radius: 10px;
            margin-bottom: 30px;
            box-shadow: 0 2px 8px rgba(0,0,0,0.1);
        }
        
        .form-title {
            color: #931A2A;
            font-weight: 700;
            margin-bottom: 20px;
            font-size: 18px;
        }
        
        .form-group label {
            font-weight: 600;
            color: #333;
            margin-bottom: 8px;
        }
        
        .form-control, .form-select {
            border: 1px solid #ddd;
            padding: 10px 12px;
            border-radius: 6px;
            transition: all 0.3s ease;
        }
        
        .form-control:focus, .form-select:focus {
            border-color: #931A2A;
            box-shadow: 0 0 0 0.2rem rgba(147, 26, 42, 0.15);
        }
        
        .btn-primary-custom {
            background: #931A2A;
            border: none;
            padding: 10px 20px;
            border-radius: 6px;
            color: white;
            font-weight: 600;
            transition: all 0.3s ease;
        }
        
        .btn-primary-custom:hover {
            background: #6f1520;
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(147, 26, 42, 0.3);
        }
        
        .table-modern {
            border-collapse: separate;
            border-spacing: 0;
        }
        
        .table-modern thead th {
            background: #931A2A;
            color: white;
            padding: 15px;
            font-weight: 600;
            border: none;
        }
        
        .table-modern tbody td {
            padding: 12px 15px;
            border-bottom: 1px solid #eee;
        }
        
        .table-modern tbody tr:hover {
            background: #f5f5f5;
            transition: background 0.3s ease;
        }
        
        .action-buttons {
            display: flex;
            gap: 5px;
            flex-wrap: wrap;
        }
        
        .action-buttons .btn {
            padding: 6px 10px;
            font-size: 13px;
            border-radius: 5px;
            transition: all 0.3s ease;
        }
        
        .action-buttons .btn:hover {
            transform: scale(1.1);
        }
        
        .stat-card {
            padding: 20px;
            border-radius: 10px;
            text-align: center;
            transition: all 0.3s ease;
            box-shadow: 0 2px 8px rgba(0,0,0,0.1);
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
        }
        
        .stat-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 8px 20px rgba(0,0,0,0.15);
        }
        
        .stat-card h6 {
            font-size: 13px;
            color: rgba(255,255,255,0.8);
            margin-bottom: 10px;
            text-transform: uppercase;
            letter-spacing: 1px;
        }
        
        .stat-card .count {
            font-size: 32px;
            font-weight: 700;
            margin: 0;
        }
        
        .alert-custom {
            border-radius: 8px;
            border: none;
            padding: 15px 20px;
        }
        
        .alert-success { background: #d4edda; color: #155724; }
        .alert-danger { background: #f8d7da; color: #721c24; }
        
        .form-row {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 15px;
        }
        
        .edit-mode {
            border-left: 4px solid #931A2A;
        }
    </style>
</head>
<body>
    <div class="wrapper">
        <!-- Sidebar -->
        <nav class="sidebar">
            <div class="sidebar-header">
                <h3>Admin Panel</h3>
            </div>
            <ul class="nav">
                <li><a href="index.php"><i class="fas fa-dashboard"></i> Dashboard</a></li>
                <li><a href="menu.php"><i class="fas fa-utensils"></i> Menu Management</a></li>
                <li><a href="reservations_fixed.php" class="active"><i class="fas fa-calendar"></i> Reservations</a></li>
                <li><a href="contact.php"><i class="fas fa-envelope"></i> Contact Messages</a></li>
                <li><a href="orders.php"><i class="fas fa-shopping-cart"></i> Orders</a></li>
                <li><a href="logout.php"><i class="fas fa-sign-out"></i> Logout</a></li>
            </ul>
        </nav>

        <!-- Main Content -->
        <div class="main-content">
            <!-- Top Navigation -->
            <nav class="top-nav">
                <div class="nav-left">
                    <h2><i class="fas fa-calendar-alt"></i> Reservation Management</h2>
                </div>
                <div class="nav-right">
                    <span class="admin-name">Welcome, Admin</span>
                </div>
            </nav>

            <!-- Content -->
            <div class="content">
                <!-- Alert Messages -->
                <?php if (!empty($message)): ?>
                    <div class="alert alert-success alert-custom alert-dismissible fade show" role="alert">
                        <i class="fas fa-check-circle"></i> <?php echo htmlspecialchars($message); ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                <?php endif; ?>

                <?php if (!empty($error)): ?>
                    <div class="alert alert-danger alert-custom alert-dismissible fade show" role="alert">
                        <i class="fas fa-exclamation-circle"></i> <?php echo htmlspecialchars($error); ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                <?php endif; ?>

                <!-- ADD / EDIT FORM SECTION -->
                <div class="admin-form-section <?php echo $edit_reservation ? 'edit-mode' : ''; ?>">
                    <div class="d-flex justify-content-between align-items-center mb-3">
                        <h5 class="form-title">
                            <i class="fas fa-<?php echo $edit_reservation ? 'edit' : 'plus'; ?>"></i>
                            <?php echo $edit_reservation ? 'Edit Reservation' : 'Add New Reservation'; ?>
                        </h5>
                        <?php if ($edit_reservation): ?>
                            <a href="reservations_fixed.php" class="btn btn-secondary btn-sm">
                                <i class="fas fa-times"></i> Cancel Edit
                            </a>
                        <?php endif; ?>
                    </div>

                    <form method="POST" action="">
                        <input type="hidden" name="action" value="<?php echo $edit_reservation ? 'update' : 'add'; ?>">
                        <?php if ($edit_reservation): ?>
                            <input type="hidden" name="reservation_id" value="<?php echo $edit_reservation['id']; ?>">
                        <?php endif; ?>

                        <div class="form-row">
                            <!-- Name -->
                            <div class="form-group">
                                <label for="name">Full Name *</label>
                                <input type="text" class="form-control" id="name" name="name" 
                                    value="<?php echo $edit_reservation ? htmlspecialchars($edit_reservation['name']) : ''; ?>" required>
                            </div>

                            <!-- Email -->
                            <div class="form-group">
                                <label for="email">Email Address *</label>
                                <input type="email" class="form-control" id="email" name="email" 
                                    value="<?php echo $edit_reservation ? htmlspecialchars($edit_reservation['email']) : ''; ?>" required>
                            </div>

                            <!-- Phone -->
                            <div class="form-group">
                                <label for="phone">Phone Number *</label>
                                <input type="tel" class="form-control" id="phone" name="phone" 
                                    value="<?php echo $edit_reservation ? htmlspecialchars($edit_reservation['phone']) : ''; ?>" required>
                            </div>

                            <!-- Number of Guests -->
                            <div class="form-group">
                                <label for="guests">Number of Guests *</label>
                                <input type="number" class="form-control" id="guests" name="guests" min="1" max="100"
                                    value="<?php echo $edit_reservation ? $edit_reservation['guests'] : ''; ?>" required>
                            </div>

                            <!-- Reservation Date -->
                            <div class="form-group">
                                <label for="reservation_date">Reservation Date *</label>
                                <input type="date" class="form-control" id="reservation_date" name="reservation_date" 
                                    value="<?php echo $edit_reservation ? $edit_reservation['res_date'] : ''; ?>" required>
                            </div>

                            <!-- Reservation Time -->
                            <div class="form-group">
                                <label for="reservation_time">Reservation Time *</label>
                                <input type="time" class="form-control" id="reservation_time" name="reservation_time" 
                                    value="<?php echo $edit_reservation ? $edit_reservation['res_time'] : ''; ?>" required>
                            </div>
                        </div>

                        <div class="d-flex gap-2 mt-4">
                            <button type="submit" class="btn btn-primary-custom">
                                <i class="fas fa-save"></i> <?php echo $edit_reservation ? 'Update Reservation' : 'Add Reservation'; ?>
                            </button>
                            <?php if ($edit_reservation): ?>
                                <a href="reservations_fixed.php" class="btn btn-secondary">
                                    <i class="fas fa-redo"></i> Reset
                                </a>
                            <?php endif; ?>
                        </div>
                    </form>
                </div>

                <!-- Statistics -->
                <div class="row mb-4">
                    <div class="col-md-12">
                        <div class="stat-card">
                            <h6>Total Reservations</h6>
                            <p class="count"><?php echo $stats['total']; ?></p>
                        </div>
                    </div>
                </div>

                <!-- Reservations Table -->
                <div class="table-responsive">
                    <table class="table table-modern">
                        <thead>
                            <tr>
                                <th>ID</th>
                                <th>Name</th>
                                <th>Email</th>
                                <th>Phone</th>
                                <th>Date</th>
                                <th>Time</th>
                                <th>Guests</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if (count($reservations) > 0): ?>
                                <?php foreach ($reservations as $res): ?>
                                <tr>
                                    <td><strong>#<?php echo $res['id']; ?></strong></td>
                                    <td><?php echo htmlspecialchars($res['name']); ?></td>
                                    <td><?php echo htmlspecialchars($res['email']); ?></td>
                                    <td><?php echo htmlspecialchars($res['phone']); ?></td>
                                    <td><?php echo date('M d, Y', strtotime($res['res_date'])); ?></td>
                                    <td><?php echo date('H:i A', strtotime($res['res_time'])); ?></td>
                                    <td><span class="badge bg-primary"><?php echo $res['guests']; ?> persons</span></td>
                                    <td>
                                        <div class="action-buttons">
                                            <!-- Edit Button -->
                                            <a href="?edit=<?php echo $res['id']; ?>" class="btn btn-info btn-sm" title="Edit">
                                                <i class="fas fa-edit"></i> Edit
                                            </a>

                                            <!-- View Button -->
                                            <button class="btn btn-primary btn-sm" data-bs-toggle="modal" data-bs-target="#detailModal" 
                                                onclick="showDetails(<?php echo htmlspecialchars(json_encode($res)); ?>)" title="View Details">
                                                <i class="fas fa-eye"></i> View
                                            </button>

                                            <!-- Delete Button -->
                                            <form method="POST" style="display:inline;">
                                                <input type="hidden" name="action" value="delete">
                                                <input type="hidden" name="reservation_id" value="<?php echo $res['id']; ?>">
                                                <button type="submit" class="btn btn-danger btn-sm" 
                                                    onclick="return confirm('Are you sure you want to delete this reservation?')" title="Delete">
                                                    <i class="fas fa-trash"></i> Delete
                                                </button>
                                            </form>
                                        </div>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            <?php else: ?>
                                <tr>
                                    <td colspan="8" class="text-center py-5 text-muted">
                                        <i class="fas fa-inbox" style="font-size: 48px; opacity: 0.3;"></i>
                                        <p class="mt-3">No reservations found</p>
                                    </td>
                                </tr>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>

    <!-- Detail Modal -->
    <div class="modal fade" id="detailModal" tabindex="-1">
        <div class="modal-dialog modal-dialog-centered">
            <div class="modal-content">
                <div class="modal-header" style="background: #931A2A; color: white;">
                    <h5 class="modal-title"><i class="fas fa-info-circle"></i> Reservation Details</h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body" id="detailContent">
                    <!-- Details will be populated here -->
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdnjs.cloudflare.com/ajax/libs/bootstrap/5.3.0/js/bootstrap.bundle.min.js"></script>
    <script>
        function showDetails(reservation) {
            const html = `
                <div class="row">
                    <div class="col-md-6">
                        <p><strong><i class="fas fa-user"></i> Name:</strong></p>
                        <p style="margin-left: 20px; color: #555;">${escapeHtml(reservation.name)}</p>
                    </div>
                    <div class="col-md-6">
                        <p><strong><i class="fas fa-envelope"></i> Email:</strong></p>
                        <p style="margin-left: 20px; color: #555;">${escapeHtml(reservation.email || 'N/A')}</p>
                    </div>
                </div>
                
                <div class="row mt-3">
                    <div class="col-md-6">
                        <p><strong><i class="fas fa-phone"></i> Phone:</strong></p>
                        <p style="margin-left: 20px; color: #555;">${escapeHtml(reservation.phone)}</p>
                    </div>
                    <div class="col-md-6">
                        <p><strong><i class="fas fa-users"></i> Guests:</strong></p>
                        <p style="margin-left: 20px; color: #555;">${reservation.guests} persons</p>
                    </div>
                </div>
                
                <div class="row mt-3">
                    <div class="col-md-6">
                        <p><strong><i class="fas fa-calendar"></i> Date:</strong></p>
                        <p style="margin-left: 20px; color: #555;">${new Date(reservation.res_date).toLocaleDateString()}</p>
                    </div>
                    <div class="col-md-6">
                        <p><strong><i class="fas fa-clock"></i> Time:</strong></p>
                        <p style="margin-left: 20px; color: #555;">${reservation.res_time}</p>
                    </div>
                </div>
            `;
            document.getElementById('detailContent').innerHTML = html;
        }
        
        function escapeHtml(text) {
            const div = document.createElement('div');
            div.textContent = text;
            return div.innerHTML;
        }
    </script>
</body>
</html>
