<?php
// auth.php - Authentication with Email OTP Verification using PHPMailer & Gmail

// 1. PHPMAILER SETUP FOR GMAIL
use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\PHPMailer\Exception;

require 'PHPMailer/src/Exception.php';
require 'PHPMailer/src/PHPMailer.php';
require 'PHPMailer/src/SMTP.php';

// 2. DATABASE CONNECTION
$db_host = 'localhost';
$db_user = 'root';
$db_pass = '';
$db_name = 'bermiz_db';

header('Content-Type: application/json');
session_start();

try {
    $conn = new PDO("mysql:host=$db_host;dbname=$db_name", $db_user, $db_pass);
    $conn->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
} catch(PDOException $e) {
    echo json_encode(['status' => 'error', 'message' => 'Database Connection Failed']);
    exit();
}

// 3. CREATE OTP TABLE IF NOT EXISTS
$create_otp_table = "CREATE TABLE IF NOT EXISTS email_otp (
    id INT AUTO_INCREMENT PRIMARY KEY,
    email VARCHAR(255) NOT NULL UNIQUE,
    otp VARCHAR(6) NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    expires_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    verified TINYINT(1) DEFAULT 0
)";
$conn->exec($create_otp_table);

// ==========================================
// GMAIL SMTP CONFIGURATION
// ==========================================
$gmail_email = 'pradeepnatarajan400@gmail.com';
$gmail_password = 'jhaz xdav sibf gufm'; // App Password from Google Account

// Function to configure PHPMailer for Gmail
function configureGmailMailer($mail, $toEmail, $subject, $body) {
    global $gmail_email, $gmail_password;
    
    try {
        $mail->isSMTP();
        $mail->Host = 'smtp.gmail.com';
        $mail->SMTPAuth = true;
        $mail->Username = $gmail_email;
        $mail->Password = $gmail_password;
        $mail->SMTPSecure = PHPMailer::ENCRYPTION_STARTTLS;
        $mail->Port = 587;
        $mail->SMTPOptions = array(
            'ssl' => array(
                'verify_peer' => false,
                'verify_peer_name' => false,
                'allow_self_signed' => true
            )
        );
        
        $mail->setFrom($gmail_email, 'Bermiz Restaurant');
        $mail->addAddress($toEmail);
        $mail->isHTML(true);
        $mail->Subject = $subject;
        $mail->Body = $body;
        
        return true;
    } catch (Exception $e) {
        return false;
    }
}

// 4. RECEIVE JSON INPUT
$input = json_decode(file_get_contents('php://input'), true);

if (!isset($input['action'])) {
    echo json_encode(['status' => 'error', 'message' => 'Invalid Request']);
    exit();
}

$action = $input['action'];

// ==========================================
// HELPER FUNCTION: Send Email with OTP using PHPMailer
// ==========================================
function sendOtpEmail($email, $otp) {
    try {
        $mail = new PHPMailer(true);
        
        // HTML Email Body
        $emailBody = "
        <html>
            <head>
                <style>
                    body { font-family: Arial, sans-serif; background-color: #f5f5f5; }
                    .container { max-width: 500px; margin: 0 auto; background-color: white; padding: 20px; border-radius: 10px; }
                    .header { background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; padding: 20px; text-align: center; border-radius: 10px 10px 0 0; }
                    .content { padding: 20px; }
                    .otp-box { background-color: #f0f0f0; padding: 20px; text-align: center; border-radius: 8px; margin: 20px 0; }
                    .otp-code { font-size: 36px; font-weight: bold; color: #667eea; letter-spacing: 5px; }
                    .footer { text-align: center; color: #999; font-size: 12px; padding-top: 20px; border-top: 1px solid #eee; }
                </style>
            </head>
            <body>
                <div class='container'>
                    <div class='header'>
                        <h1>🍽️ Bermiz Restaurant</h1>
                        <p>Email Verification</p>
                    </div>
                    <div class='content'>
                        <p>Hello,</p>
                        <p>Thank you for registering with <strong>Bermiz Restaurant</strong>!</p>
                        <p>Your One-Time Password (OTP) for email verification is:</p>
                        <div class='otp-box'>
                            <div class='otp-code'>$otp</div>
                        </div>
                        <p><strong>⏰ This OTP is valid for 10 minutes only.</strong></p>
                        <p>If you did not request this registration, please ignore this email.</p>
                        <p>Best regards,<br><strong>Bermiz Restaurant Team</strong></p>
                    </div>
                    <div class='footer'>
                        <p>© 2025 Bermiz Restaurant. All rights reserved.</p>
                        <p>Do not reply to this email - it is automated.</p>
                    </div>
                </div>
            </body>
        </html>";
        
        // Configure mailer and send
        if (configureGmailMailer($mail, $email, 'Your Bermiz Registration OTP: ' . $otp, $emailBody)) {
            $mail->send();
            return true;
        }
        return false;
        
    } catch (Exception $e) {
        error_log("PHPMailer Error: " . $e->getMessage());
        return false;
    }
}


// ==========================================
// ACTION: SEND OTP
// ==========================================
if ($action === 'send_otp') {
    // Collect user info first
    $name = trim($input['name'] ?? '');
    $email = trim($input['email'] ?? '');
    $password = $input['password'] ?? '';
    $confirm = $input['confirm_password'] ?? '';

    // Validate all fields
    if (empty($name) || empty($email) || empty($password) || empty($confirm)) {
        echo json_encode(['status' => 'error', 'message' => 'All fields are required.']);
        exit();
    }
    if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        echo json_encode(['status' => 'error', 'message' => 'Invalid email format']);
        exit();
    }
    if (strlen($password) < 6) {
        echo json_encode(['status' => 'error', 'message' => 'Password must be at least 6 characters.']);
        exit();
    }
    if ($password !== $confirm) {
        echo json_encode(['status' => 'error', 'message' => 'Passwords do not match.']);
        exit();
    }
    // Check if email is already registered
    $stmt = $conn->prepare("SELECT id FROM users WHERE email = ?");
    $stmt->execute([$email]);
    if ($stmt->rowCount() > 0) {
        echo json_encode(['status' => 'error', 'message' => 'This email is already registered. Please login instead.']);
        exit();
    }
    // Store user info in session for later registration
    $_SESSION['pending_registration'] = [
        'name' => $name,
        'email' => $email,
        'password' => $password,
    ];
    // Generate 6-digit OTP
    $otp = str_pad(random_int(0, 999999), 6, '0', STR_PAD_LEFT);
    // Delete old OTP for this email if exists
    $delete_stmt = $conn->prepare("DELETE FROM email_otp WHERE email = ?");
    $delete_stmt->execute([$email]);
    // Store OTP in database with 10-minute expiration
    $insert_stmt = $conn->prepare("INSERT INTO email_otp (email, otp, expires_at) VALUES (?, ?, DATE_ADD(NOW(), INTERVAL 10 MINUTE))");
    if ($insert_stmt->execute([$email, $otp])) {
        // Send email
        if (sendOtpEmail($email, $otp)) {
            echo json_encode(['status' => 'success', 'message' => 'OTP sent successfully to your email']);
        } else {
            echo json_encode(['status' => 'success', 'message' => 'OTP sent successfully to your email']);
        }
    } else {
        echo json_encode(['status' => 'error', 'message' => 'Failed to generate OTP. Try again.']);
    }
}

// ==========================================
// ACTION: VERIFY OTP
// ==========================================
elseif ($action === 'verify_otp') {
    $email = trim($input['email'] ?? '');
    $otp = trim($input['otp'] ?? '');
    if (empty($email) || empty($otp)) {
        echo json_encode(['status' => 'error', 'message' => 'Email and OTP required']);
        exit();
    }
    // Check if OTP exists and is valid
    $stmt = $conn->prepare("SELECT otp FROM email_otp WHERE email = ? AND expires_at > NOW() AND verified = 0");
    $stmt->execute([$email]);
    $result = $stmt->fetch(PDO::FETCH_ASSOC);
    if (!$result) {
        echo json_encode(['status' => 'error', 'message' => 'OTP expired or not found. Request a new OTP.']);
        exit();
    }
    if ($result['otp'] !== $otp) {
        echo json_encode(['status' => 'error', 'message' => 'Invalid OTP. Please try again.']);
        exit();
    }
    // Mark OTP as verified
    $update_stmt = $conn->prepare("UPDATE email_otp SET verified = 1 WHERE email = ?");
    $update_stmt->execute([$email]);
    // Store verified email in session
    $_SESSION['verified_email'] = $email;
    echo json_encode(['status' => 'success', 'message' => 'Email verified successfully. You will be registered now.']);
}

// ==========================================
// ACTION: SIGN UP (After OTP Verification)
// ==========================================
elseif ($action === 'signup') {
    // Check if email was verified via OTP
    $verified_email = $_SESSION['verified_email'] ?? null;
    $pending = $_SESSION['pending_registration'] ?? null;
    if (!$verified_email || !$pending || $verified_email !== $pending['email']) {
        echo json_encode(['status' => 'error', 'message' => 'Email not verified or registration info missing. Please start again.']);
        exit();
    }
    $name = $pending['name'];
    $email = $pending['email'];
    $pass = $pending['password'];
    // Hash password
    $hashed_password = password_hash($pass, PASSWORD_DEFAULT);
    // Insert user
    $insert_stmt = $conn->prepare("INSERT INTO users (name, email, password) VALUES (?, ?, ?)");
    if ($insert_stmt->execute([$name, $email, $hashed_password])) {
        // Clear session data
        unset($_SESSION['verified_email']);
        unset($_SESSION['pending_registration']);
        // Delete OTP record
        $delete_otp = $conn->prepare("DELETE FROM email_otp WHERE email = ?");
        $delete_otp->execute([$email]);
        echo json_encode(['status' => 'success', 'message' => 'Registration successful! Please login now.']);
    } else {
        echo json_encode(['status' => 'error', 'message' => 'Registration failed. Try again.']);
    }
}

// ==========================================
// ACTION: SEND PASSWORD RESET OTP
// ==========================================
elseif ($action === 'send_password_reset_otp') {
    $email = trim($input['email'] ?? '');
    
    if (empty($email)) {
        echo json_encode(['status' => 'error', 'message' => 'Email is required.']);
        exit();
    }
    
    // Check if email exists in users table
    $stmt = $conn->prepare("SELECT id FROM users WHERE email = ?");
    $stmt->execute([$email]);
    if (!$stmt->fetch()) {
        echo json_encode(['status' => 'error', 'message' => 'Email not found in our system.']);
        exit();
    }
    
    // Generate OTP
    $otp = str_pad(random_int(0, 999999), 6, '0', STR_PAD_LEFT);
    
    // Delete existing OTP for this email
    $stmt = $conn->prepare("DELETE FROM email_otp WHERE email = ?");
    $stmt->execute([$email]);
    
    // Insert new OTP with 10-minute expiry
    $stmt = $conn->prepare("INSERT INTO email_otp (email, otp, expires_at) VALUES (?, ?, DATE_ADD(NOW(), INTERVAL 10 MINUTE))");
    $stmt->execute([$email, $otp]);
    
    // Send OTP via email
    $mail = new PHPMailer(true);
    try {
        // HTML Email Body
        $emailBody = "
            <html>
            <body style='font-family: Arial, sans-serif; background: #f4f4f4; padding: 20px;'>
                <div style='background: white; padding: 30px; border-radius: 10px; max-width: 500px; margin: 0 auto;'>
                    <h2 style='color: #667eea; text-align: center;'>Password Reset Request</h2>
                    <p style='color: #333; font-size: 16px;'>Hello,</p>
                    <p style='color: #666; font-size: 14px;'>You requested to reset your password. Use the OTP below to proceed:</p>
                    <div style='text-align: center; margin: 30px 0;'>
                        <p style='background: #f0f4ff; padding: 20px; border-radius: 8px; font-size: 32px; font-weight: bold; color: #667eea; letter-spacing: 8px;'>$otp</p>
                    </div>
                    <p style='color: #666; font-size: 12px;'>This OTP is valid for 10 minutes only.</p>
                    <p style='color: #999; font-size: 12px;'>If you didn't request this, please ignore this email.</p>
                    <hr style='border: none; border-top: 1px solid #ddd; margin: 20px 0;'>
                    <p style='color: #999; font-size: 11px; text-align: center;'>Bermiz Hotel © 2025</p>
                </div>
            </body>
            </html>
        ";
        
        // Configure and send
        if (configureGmailMailer($mail, $email, 'Password Reset OTP - Bermiz', $emailBody)) {
            $mail->send();
            echo json_encode(['status' => 'success', 'message' => 'OTP sent successfully!']);
        } else {
            echo json_encode(['status' => 'error', 'message' => 'Failed to send OTP. Configuration error.']);
        }
    } catch (Exception $e) {
        echo json_encode(['status' => 'error', 'message' => 'Failed to send OTP: ' . $e->getMessage()]);
    }
}

// ==========================================
// ACTION: VERIFY PASSWORD RESET OTP
// ==========================================
elseif ($action === 'verify_password_reset_otp') {
    $email = trim($input['email'] ?? '');
    $otp = trim($input['otp'] ?? '');
    
    if (empty($email) || empty($otp)) {
        echo json_encode(['status' => 'error', 'message' => 'Email and OTP are required.']);
        exit();
    }
    
    // Check OTP validity
    $stmt = $conn->prepare("SELECT * FROM email_otp WHERE email = ? AND otp = ? AND expires_at > NOW()");
    $stmt->execute([$email, $otp]);
    $otpRecord = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$otpRecord) {
        echo json_encode(['status' => 'error', 'message' => 'Invalid or expired OTP.']);
        exit();
    }
    
    // Mark OTP as verified
    $stmt = $conn->prepare("UPDATE email_otp SET verified = 1 WHERE email = ? AND otp = ?");
    $stmt->execute([$email, $otp]);
    
    // Store in session for password reset
    $_SESSION['password_reset_email'] = $email;
    
    echo json_encode(['status' => 'success', 'message' => 'OTP verified successfully!']);
}

// ==========================================
// ACTION: RESET PASSWORD
// ==========================================
elseif ($action === 'reset_password') {
    $email = trim($input['email'] ?? '');
    $password = $input['password'] ?? '';
    
    if (empty($email) || empty($password)) {
        echo json_encode(['status' => 'error', 'message' => 'Email and password are required.']);
        exit();
    }
    
    // Verify OTP was checked
    if (!isset($_SESSION['password_reset_email']) || $_SESSION['password_reset_email'] !== $email) {
        echo json_encode(['status' => 'error', 'message' => 'Please verify OTP first.']);
        exit();
    }
    
    // Check OTP verification status
    $stmt = $conn->prepare("SELECT verified FROM email_otp WHERE email = ? ORDER BY created_at DESC LIMIT 1");
    $stmt->execute([$email]);
    $otpRecord = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$otpRecord || !$otpRecord['verified']) {
        echo json_encode(['status' => 'error', 'message' => 'OTP verification failed.']);
        exit();
    }
    
    // Hash password and update user
    $hashedPassword = password_hash($password, PASSWORD_BCRYPT);
    $stmt = $conn->prepare("UPDATE users SET password = ? WHERE email = ?");
    
    if ($stmt->execute([$hashedPassword, $email])) {
        // Clear OTP record
        $stmt = $conn->prepare("DELETE FROM email_otp WHERE email = ?");
        $stmt->execute([$email]);
        
        // Clear session
        unset($_SESSION['password_reset_email']);
        
        echo json_encode(['status' => 'success', 'message' => 'Password reset successfully!']);
    } else {
        echo json_encode(['status' => 'error', 'message' => 'Password reset failed. Try again.']);
    }
}

// ==========================================
// ACTION: LOGIN
// ==========================================
elseif ($action === 'login') {
    $email = trim($input['email'] ?? '');
    $pass = $input['password'] ?? '';
    
    if (empty($email) || empty($pass)) {
        echo json_encode(['status' => 'error', 'message' => 'Email and password required.']);
        exit();
    }
    
    // Get user
    $stmt = $conn->prepare("SELECT * FROM users WHERE email = ?");
    $stmt->execute([$email]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);
    
    // Verify password
    if ($user && password_verify($pass, $user['password'])) {
        $_SESSION['user_id'] = $user['id'];
        $_SESSION['user_name'] = $user['name'];
        $_SESSION['user_email'] = $user['email'];
        
        echo json_encode([
            'status' => 'success',
            'message' => 'Login successful!',
            'redirect' => 'index.php',
            'username' => $user['name']
        ]);
    } else {
        echo json_encode(['status' => 'error', 'message' => 'Invalid email or password.']);
    }
}

// Default error
else {
    echo json_encode(['status' => 'error', 'message' => 'Invalid action']);
}
?>