<?php
session_start();
header('Content-Type: application/json');

require_once 'db_connect.php';

// Check if user is logged in
$is_logged_in = isset($_SESSION['user_id']);
$user_id = $is_logged_in ? $_SESSION['user_id'] : null;

$action = $_POST['action'] ?? '';

// --- CREATE ORDER ---
if ($action == 'create_order') {
    if (!$is_logged_in) {
        echo json_encode(["status" => "error", "message" => "User not logged in"]);
        exit;
    }

    $orderDataJson = $_POST['order_data'] ?? '{}';
    $orderData = json_decode($orderDataJson, true);

    if (empty($orderData)) {
        echo json_encode(["status" => "error", "message" => "Invalid order data"]);
        exit;
    }

    try {
        // Generate unique order ID
        $orderId = 'ORD' . date('YmdHis') . rand(1000, 9999);

        // Insert order into database
        $stmt = $conn->prepare(
            "INSERT INTO orders (order_id, user_id, user_name, user_phone, user_email, 
             delivery_address, delivery_street, delivery_city, delivery_state, delivery_postal, 
             special_instructions, payment_method, subtotal, delivery_charge, taxes, discount, 
             total_amount, order_status, created_at) 
             VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW())"
        );

        if (!$stmt) {
            throw new Exception($conn->error);
        }

        $orderStatus = 'pending';
        
        $stmt->bind_param(
            "sississsssssddddds",
            $orderId,
            $user_id,
            $orderData['user_name'],
            $orderData['user_phone'],
            $orderData['user_email'],
            $orderData['delivery_address'],
            $orderData['delivery_street'],
            $orderData['delivery_city'],
            $orderData['delivery_state'],
            $orderData['delivery_postal'],
            $orderData['special_instructions'],
            $orderData['payment_method'],
            $orderData['subtotal'],
            $orderData['delivery_charge'],
            $orderData['taxes'],
            $orderData['discount'],
            $orderData['total_amount'],
            $orderStatus
        );

        if (!$stmt->execute()) {
            throw new Exception($stmt->error);
        }

        $orderId_db = $conn->insert_id;
        $stmt->close();

        // Insert order items
        if (!empty($orderData['order_items']) && is_array($orderData['order_items'])) {
            $itemStmt = $conn->prepare(
                "INSERT INTO order_items (order_id, product_name, product_price, quantity, item_total) 
                 VALUES (?, ?, ?, ?, ?)"
            );

            if (!$itemStmt) {
                throw new Exception($conn->error);
            }

            foreach ($orderData['order_items'] as $item) {
                $itemName = $item['product_name'] ?? '';
                $itemPrice = floatval($item['product_price'] ?? 0);
                $itemQty = intval($item['quantity'] ?? 1);
                $itemTotal = $itemPrice * $itemQty;

                $itemStmt->bind_param(
                    "isdid",
                    $orderId_db,
                    $itemName,
                    $itemPrice,
                    $itemQty,
                    $itemTotal
                );

                if (!$itemStmt->execute()) {
                    throw new Exception($itemStmt->error);
                }
            }

            $itemStmt->close();
        }

        // Clear cart from database for this user
        $clearStmt = $conn->prepare("DELETE FROM cart WHERE user_id = ?");
        $clearStmt->bind_param("i", $user_id);
        $clearStmt->execute();
        $clearStmt->close();

        // Clear session cart
        $_SESSION['cart'] = [];

        echo json_encode([
            "status" => "success",
            "message" => "Order created successfully",
            "order_id" => $orderId
        ]);
        exit;

    } catch (Exception $e) {
        echo json_encode([
            "status" => "error",
            "message" => "Database error: " . $e->getMessage()
        ]);
        exit;
    }
}

// --- FETCH USER ORDERS ---
if ($action == 'fetch_user_orders') {
    if (!$is_logged_in) {
        echo json_encode(["status" => "error", "message" => "User not logged in"]);
        exit;
    }

    try {
        $stmt = $conn->prepare(
            "SELECT id, order_id, user_name, total_amount, order_status, created_at 
             FROM orders WHERE user_id = ? ORDER BY created_at DESC LIMIT 10"
        );
        $stmt->bind_param("i", $user_id);
        $stmt->execute();
        $result = $stmt->get_result();
        
        $orders = [];
        while ($row = $result->fetch_assoc()) {
            $orders[] = $row;
        }

        echo json_encode([
            "status" => "success",
            "orders" => $orders
        ]);
        exit;

    } catch (Exception $e) {
        echo json_encode(["status" => "error", "message" => $e->getMessage()]);
        exit;
    }
}

// --- FETCH ORDER DETAILS ---
if ($action == 'fetch_order_details') {
    if (!$is_logged_in) {
        echo json_encode(["status" => "error", "message" => "User not logged in"]);
        exit;
    }

    $orderId = $_POST['order_id'] ?? '';

    if (empty($orderId)) {
        echo json_encode(["status" => "error", "message" => "Order ID required"]);
        exit;
    }

    try {
        // Fetch order details (using auto-increment id)
        $orderStmt = $conn->prepare(
            "SELECT * FROM orders WHERE id = ? AND user_id = ?"
        );
        $orderStmt->bind_param("ii", $orderId, $user_id);
        $orderStmt->execute();
        $orderResult = $orderStmt->get_result();

        if ($orderResult->num_rows === 0) {
            echo json_encode(["status" => "error", "message" => "Order not found"]);
            exit;
        }

        $order = $orderResult->fetch_assoc();

        // Fetch order items
        $itemStmt = $conn->prepare(
            "SELECT product_name, product_price, quantity, item_total FROM order_items WHERE order_id = ?"
        );
        $itemStmt->bind_param("i", $orderId);
        $itemStmt->execute();
        $itemResult = $itemStmt->get_result();

        $items = [];
        while ($item = $itemResult->fetch_assoc()) {
            $items[] = $item;
        }

        echo json_encode([
            "status" => "success",
            "order" => $order,
            "items" => $items
        ]);
        exit;

    } catch (Exception $e) {
        echo json_encode(["status" => "error", "message" => $e->getMessage()]);
        exit;
    }
}

// --- UPDATE ORDER STATUS (ADMIN ONLY) ---
if ($action == 'update_order_status') {
    // Check if admin
    $is_admin = isset($_SESSION['admin_id']);
    if (!$is_admin) {
        echo json_encode(["status" => "error", "message" => "Admin access required"]);
        exit;
    }

    $orderId = $_POST['order_id'] ?? '';
    $status = $_POST['status'] ?? '';

    if (empty($orderId) || empty($status)) {
        echo json_encode(["status" => "error", "message" => "Order ID and status required"]);
        exit;
    }

    // Validate status
    $validStatuses = ['pending', 'confirmed', 'preparing', 'out_for_delivery', 'delivered', 'cancelled'];
    if (!in_array($status, $validStatuses)) {
        echo json_encode(["status" => "error", "message" => "Invalid status"]);
        exit;
    }

    try {
        $stmt = $conn->prepare("UPDATE orders SET order_status = ? WHERE id = ?");
        $stmt->bind_param("si", $status, $orderId);
        
        if ($stmt->execute()) {
            echo json_encode(["status" => "success", "message" => "Order status updated"]);
        } else {
            echo json_encode(["status" => "error", "message" => $stmt->error]);
        }
        $stmt->close();
        exit;

    } catch (Exception $e) {
        echo json_encode(["status" => "error", "message" => $e->getMessage()]);
        exit;
    }
}

// No valid action
echo json_encode(["status" => "error", "message" => "Invalid action"]);
$conn->close();
?>
