<?php
session_start();
include 'db_connect.php';

// Check if user is admin or has access to test page
// For development only - remove in production

$test_action = $_GET['test_action'] ?? '';
$test_email = $_POST['test_email'] ?? '';
$test_otp = $_POST['test_otp'] ?? '';

// Create table if not exists
$create_table = "CREATE TABLE IF NOT EXISTS email_otp (
    id INT AUTO_INCREMENT PRIMARY KEY,
    email VARCHAR(255) NOT NULL UNIQUE,
    otp VARCHAR(6) NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    expires_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    verified TINYINT(1) DEFAULT 0
)";
$conn->query($create_table);

$message = '';
$message_type = '';

// Handle test actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if ($test_action === 'view_otp' && !empty($test_email)) {
        $message = "Checking OTP for: " . htmlspecialchars($test_email);
        $message_type = 'info';
    } elseif ($test_action === 'verify_otp_test' && !empty($test_email) && !empty($test_otp)) {
        $message = "Testing OTP verification for: " . htmlspecialchars($test_email);
        $message_type = 'info';
    } elseif ($test_action === 'delete_email') {
        $stmt = $conn->query("DELETE FROM email_otp WHERE email = '" . $conn->real_escape_string($test_email) . "'");
        $message = "✅ Deleted OTP record for: " . htmlspecialchars($test_email);
        $message_type = 'success';
    } elseif ($test_action === 'clear_all') {
        $conn->query("DELETE FROM email_otp");
        $message = "✅ Cleared all OTP records";
        $message_type = 'success';
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>OTP System Test & Debug Page</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/bootstrap/5.3.0/css/bootstrap.min.css">
    <style>
        body { background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); min-height: 100vh; padding: 20px; }
        .container { max-width: 1000px; background: white; border-radius: 10px; padding: 30px; box-shadow: 0 10px 40px rgba(0,0,0,0.2); }
        h1 { color: #667eea; margin-bottom: 30px; }
        .section { margin-bottom: 30px; padding: 20px; background: #f8f9fa; border-radius: 8px; border-left: 4px solid #667eea; }
        .section h2 { color: #764ba2; font-size: 18px; margin-bottom: 15px; }
        .table-responsive { margin-top: 15px; }
        table { font-size: 13px; }
        .badge-active { background-color: #28a745; }
        .badge-inactive { background-color: #dc3545; }
        .badge-expires { background-color: #ffc107; }
        .form-group { margin-bottom: 15px; }
        button { border-radius: 5px; }
        .alert { border-radius: 8px; }
        .info-box { background: #e7f3ff; border: 1px solid #b3d9ff; padding: 15px; border-radius: 5px; margin-bottom: 15px; color: #004085; }
        .success-box { background: #d4edda; border: 1px solid #c3e6cb; padding: 15px; border-radius: 5px; margin-bottom: 15px; color: #155724; }
        .warning-box { background: #fff3cd; border: 1px solid #ffeaa7; padding: 15px; border-radius: 5px; margin-bottom: 15px; color: #856404; }
    </style>
</head>
<body>
    <div class="container">
        <h1>🔍 OTP System Test & Verification Page</h1>
        <p class="text-muted">Developer tool to test OTP functionality and verify database storage</p>

        <?php if (!empty($message)): ?>
            <div class="alert alert-<?php echo $message_type === 'success' ? 'success' : ($message_type === 'error' ? 'danger' : 'info'); ?> alert-dismissible fade show">
                <?php echo $message; ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
        <?php endif; ?>

        <!-- SECTION 1: Database Status -->
        <div class="section">
            <h2>📊 Database Status</h2>
            <?php
            // Check if email_otp table exists
            $tables = $conn->query("SHOW TABLES LIKE 'email_otp'");
            $table_exists = $tables->num_rows > 0;
            ?>
            <div class="info-box">
                <strong>email_otp Table:</strong> 
                <?php if ($table_exists): ?>
                    ✅ <span style="color: #28a745;">EXISTS</span>
                <?php else: ?>
                    ❌ <span style="color: #dc3545;">NOT FOUND - Will be created on first OTP send</span>
                <?php endif; ?>
            </div>

            <?php
            // Count records
            $count_result = $conn->query("SELECT COUNT(*) as total FROM email_otp");
            $count = $count_result->fetch_assoc()['total'] ?? 0;
            ?>
            <p><strong>Total OTP Records:</strong> <span class="badge bg-primary"><?php echo $count; ?></span></p>

            <?php
            // Check users table
            $users_result = $conn->query("SELECT COUNT(*) as total FROM users");
            $users_count = $users_result->fetch_assoc()['total'] ?? 0;
            ?>
            <p><strong>Total Registered Users:</strong> <span class="badge bg-success"><?php echo $users_count; ?></span></p>
        </div>

        <!-- SECTION 2: All OTP Records -->
        <div class="section">
            <h2>📋 All OTP Records in Database</h2>
            <?php
            try {
                $otps = $conn->query("SELECT * FROM email_otp ORDER BY created_at DESC");
                if ($otps && $otps->num_rows > 0):
            ?>
            <div class="table-responsive">
                <table class="table table-striped table-hover">
                    <thead class="table-dark">
                        <tr>
                            <th>Email</th>
                            <th>OTP Code</th>
                            <th>Created</th>
                            <th>Expires</th>
                            <th>Expired?</th>
                            <th>Verified?</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php while ($otp = $otps->fetch_assoc()): 
                            $is_expired = strtotime($otp['expires_at']) < time();
                        ?>
                        <tr>
                            <td><strong><?php echo htmlspecialchars($otp['email']); ?></strong></td>
                            <td>
                                <span style="font-size: 18px; font-weight: bold; color: #667eea; letter-spacing: 3px;">
                                    <?php echo $otp['otp']; ?>
                                </span>
                            </td>
                            <td><?php echo date('M d, Y H:i:s', strtotime($otp['created_at'])); ?></td>
                            <td><?php echo date('M d, Y H:i:s', strtotime($otp['expires_at'])); ?></td>
                            <td>
                                <?php if ($is_expired): ?>
                                    <span class="badge badge-expires">⏰ YES</span>
                                <?php else: ?>
                                    <span class="badge bg-success">✅ NO</span>
                                <?php endif; ?>
                            </td>
                            <td>
                                <?php if ($otp['verified'] == 1): ?>
                                    <span class="badge badge-active">✅ YES</span>
                                <?php else: ?>
                                    <span class="badge badge-inactive">❌ NO</span>
                                <?php endif; ?>
                            </td>
                            <td>
                                <form method="POST" style="display: inline;">
                                    <input type="hidden" name="test_action" value="delete_email">
                                    <input type="hidden" name="test_email" value="<?php echo htmlspecialchars($otp['email']); ?>">
                                    <button type="submit" class="btn btn-sm btn-danger">Delete</button>
                                </form>
                            </td>
                        </tr>
                        <?php endwhile; ?>
                    </tbody>
                </table>
            </div>
            <?php else: ?>
                <div class="alert alert-info">No OTP records found. Go to login page and send OTP to create records.</div>
            <?php 
                endif;
            } catch (Exception $e) {
                echo '<div class="alert alert-warning">Error reading OTP records: ' . htmlspecialchars($e->getMessage()) . '</div>';
            }
            ?>
        </div>

        <!-- SECTION 3: All Registered Users -->
        <div class="section">
            <h2>👥 Registered Users</h2>
            <?php
            try {
                $users = $conn->query("SELECT id, name, email, created_at FROM users ORDER BY created_at DESC");
                if ($users && $users->num_rows > 0):
            ?>
            <div class="table-responsive">
                <table class="table table-striped table-hover">
                    <thead class="table-dark">
                        <tr>
                            <th>ID</th>
                            <th>Name</th>
                            <th>Email</th>
                            <th>Registered</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php while ($user = $users->fetch_assoc()): ?>
                        <tr>
                            <td><?php echo $user['id']; ?></td>
                            <td><strong><?php echo htmlspecialchars($user['name']); ?></strong></td>
                            <td><?php echo htmlspecialchars($user['email']); ?></td>
                            <td><?php echo date('M d, Y H:i:s', strtotime($user['created_at'])); ?></td>
                        </tr>
                        <?php endwhile; ?>
                    </tbody>
                </table>
            </div>
            <?php else: ?>
                <div class="alert alert-info">No users registered yet. Complete the registration flow to create users.</div>
            <?php 
                endif;
            } catch (Exception $e) {
                echo '<div class="alert alert-warning">Error reading users: ' . htmlspecialchars($e->getMessage()) . '</div>';
            }
            ?>
        </div>

        <!-- SECTION 4: Test Functions -->
        <div class="section">
            <h2>🧪 Test Functions</h2>
            
            <div style="margin-bottom: 20px;">
                <h4>View OTP for Specific Email</h4>
                <form method="POST" class="row g-3">
                    <div class="col-md-6">
                        <input type="email" class="form-control" name="test_email" placeholder="Enter email address" required>
                    </div>
                    <div class="col-md-6">
                        <input type="hidden" name="test_action" value="view_otp">
                        <button type="submit" class="btn btn-primary w-100">🔍 Search</button>
                    </div>
                </form>
                <?php
                if ($test_action === 'view_otp' && !empty($test_email)) {
                    $result = $conn->query("SELECT * FROM email_otp WHERE email = '" . $conn->real_escape_string($test_email) . "'");
                    if ($result && $result->num_rows > 0) {
                        $otp = $result->fetch_assoc();
                        echo '<div class="success-box" style="margin-top: 15px;">';
                        echo '<strong>✅ Found OTP Record:</strong><br>';
                        echo '<strong>Email:</strong> ' . htmlspecialchars($otp['email']) . '<br>';
                        echo '<strong>OTP Code:</strong> <span style="font-size: 20px; letter-spacing: 2px;">' . $otp['otp'] . '</span><br>';
                        echo '<strong>Created:</strong> ' . date('M d, Y H:i:s', strtotime($otp['created_at'])) . '<br>';
                        echo '<strong>Expires:</strong> ' . date('M d, Y H:i:s', strtotime($otp['expires_at'])) . '<br>';
                        echo '<strong>Status:</strong> ' . (strtotime($otp['expires_at']) < time() ? '⏰ EXPIRED' : '✅ VALID') . '<br>';
                        echo '<strong>Verified:</strong> ' . ($otp['verified'] == 1 ? '✅ YES' : '❌ NO') . '<br>';
                        echo '</div>';
                    } else {
                        echo '<div class="warning-box" style="margin-top: 15px;">⚠️ No OTP found for this email</div>';
                    }
                }
                ?>
            </div>

            <hr>

            <div style="margin-bottom: 20px;">
                <h4>Delete OTP Record</h4>
                <form method="POST" class="row g-3">
                    <div class="col-md-6">
                        <input type="email" class="form-control" name="test_email" placeholder="Enter email to delete" required>
                    </div>
                    <div class="col-md-6">
                        <input type="hidden" name="test_action" value="delete_email">
                        <button type="submit" class="btn btn-danger w-100" onclick="return confirm('Delete OTP for this email?')">🗑️ Delete</button>
                    </div>
                </form>
            </div>

            <hr>

            <div>
                <h4>Clear All OTP Records</h4>
                <form method="POST">
                    <input type="hidden" name="test_action" value="clear_all">
                    <button type="submit" class="btn btn-dark" onclick="return confirm('Clear ALL OTP records? This cannot be undone!')">🗑️ Clear All</button>
                </form>
            </div>
        </div>

        <!-- SECTION 5: Registration Flow Checklist -->
        <div class="section">
            <h2>✅ Registration Flow Checklist</h2>
            <div class="info-box">
                <ol>
                    <li><strong>Email Verification:</strong> User enters email on login.php</li>
                    <li><strong>OTP Generation:</strong> auth.php generates 6-digit OTP</li>
                    <li><strong>OTP Storage:</strong> OTP stored in email_otp table ✅</li>
                    <li><strong>Email Sent:</strong> Email sent to user with OTP (check spam folder)</li>
                    <li><strong>OTP Entry:</strong> User enters OTP from email</li>
                    <li><strong>OTP Verification:</strong> auth.php validates OTP against database</li>
                    <li><strong>Mark Verified:</strong> email_otp.verified changed to 1</li>
                    <li><strong>Registration Form:</strong> User completes registration form</li>
                    <li><strong>User Creation:</strong> New user inserted into users table</li>
                    <li><strong>OTP Cleanup:</strong> OTP record deleted from database</li>
                    <li><strong>Login:</strong> User can now login with credentials</li>
                </ol>
            </div>
        </div>

        <!-- SECTION 6: How to Use -->
        <div class="section">
            <h2>🚀 How to Test Complete Registration</h2>
            <div class="warning-box">
                <strong>Step 1: Send OTP</strong>
                <ul>
                    <li>Go to: <a href="login.php?show=signup" target="_blank">http://localhost/hotel/login.php?show=signup</a></li>
                    <li>Click "Sign Up"</li>
                    <li>Enter email: test@example.com</li>
                    <li>Click "Send OTP"</li>
                    <li>Check this page - new record should appear above</li>
                </ul>

                <strong>Step 2: Get OTP from Database</strong>
                <ul>
                    <li>Look at "All OTP Records" table above</li>
                    <li>Find your email</li>
                    <li>Copy the OTP code (e.g., 456789)</li>
                </ul>

                <strong>Step 3: Verify OTP</strong>
                <ul>
                    <li>Go back to login.php</li>
                    <li>Enter the OTP code</li>
                    <li>Click "Verify OTP"</li>
                    <li>Check this page - "Verified?" should change to YES</li>
                </ul>

                <strong>Step 4: Complete Registration</strong>
                <ul>
                    <li>Fill in registration form</li>
                    <li>Click "Join Bermiz"</li>
                    <li>Check "Registered Users" table above - new user should appear</li>
                    <li>Check "All OTP Records" - record should be deleted</li>
                </ul>

                <strong>Step 5: Login</strong>
                <ul>
                    <li>Go back to login.php</li>
                    <li>Enter email and password</li>
                    <li>Click "Dine In"</li>
                    <li>Should redirect to homepage</li>
                </ul>
            </div>
        </div>

        <div style="text-align: center; margin-top: 30px; color: #999;">
            <p><small>🔐 This is a development/testing page. Remove it before going to production!</small></p>
            <p><small>Last Updated: December 24, 2025</small></p>
        </div>
    </div>

    <script src="https://cdnjs.cloudflare.com/ajax/libs/bootstrap/5.3.0/js/bootstrap.bundle.min.js"></script>
</body>
</html>
